<?php

namespace App\Services;

use App\Models\Event;
use Illuminate\Support\Facades\DB;
use RuntimeException;
use Symfony\Component\Process\Exception\ProcessTimedOutException;
use Symfony\Component\Process\Process;

class FaceRecognitionService
{
    private string $tempDir;

    public function __construct()
    {
        $this->tempDir = storage_path('app/face-temp');
        if (! is_dir($this->tempDir)) {
            mkdir($this->tempDir, 0755, true);
        }
    }

    // ---------------------------------------------------------------
    //  Public API
    // ---------------------------------------------------------------

    public function isAvailable(): bool
    {
        static $result = null;

        if ($result !== null) {
            return $result;
        }

        if (! config('face.enabled')) {
            return $result = false;
        }

        return $result = $this->checkPython();
    }

    public function match(Event $event, string $imagePath, ?int $limit = null): array
    {
        if (! config('face.enabled')) {
            return [];
        }

        $threshold = (float) config('face.match_threshold', 0.50);
        $limit = $limit ?? (int) config('face.max_results', 120);

        return $this->runMatch($event, $imagePath, $threshold, $limit);
    }

    public function index(Event $event, bool $force = false): void
    {
        if (! config('face.enabled')) {
            return;
        }

        $this->runIndex($event, $force);
    }

    // ---------------------------------------------------------------
    //  Availability Check
    // ---------------------------------------------------------------

    private function checkPython(): bool
    {
        try {
            $process = new Process([$this->pythonBinary(), '--version']);
            $process->setTimeout(5)->run();

            if (! $process->isSuccessful()) {
                return false;
            }
        } catch (\Throwable) {
            return false;
        }

        $scriptPath = config('face.script', base_path('scripts/face_recognition.py'));

        return is_file($scriptPath);
    }

    // ---------------------------------------------------------------
    //  Match
    // ---------------------------------------------------------------

    private function runMatch(Event $event, string $imagePath, float $threshold, int $limit): array
    {
        // Auto-index if event has ready images but no face embeddings yet
        $faceCount = DB::table('event_media_faces')->where('event_id', $event->id)->count();
        if ($faceCount === 0) {
            $imageCount = $event->media()->where('file_type', 'image')->where('status', 'ready')->count();
            if ($imageCount > 0) {
                $this->runIndex($event, false);
            }
        }

        $this->validatePythonEnvironment();

        $script = $this->scriptPath();
        $tempImagePath = $this->createTempImageCopy($imagePath, $event->id);
        $dbConfigPath = $this->createDatabaseConfigFile();

        try {
            $process = new Process([
                $this->pythonBinary(),
                $script,
                'match',
                '--event-id',
                (string) $event->id,
                '--image',
                $tempImagePath,
                '--threshold',
                (string) $threshold,
                '--limit',
                (string) $limit,
                '--model-dir',
                (string) config('face.model_dir', storage_path('app/face-models')),
                '--db-config',
                $dbConfigPath,
            ], base_path(), [
                'PYTHONIOENCODING' => 'utf-8',
                'PYTHONUNBUFFERED' => '1',
            ]);

            $process->setTimeout((int) config('face.timeout', 180));
            $process->run();

            if (! $process->isSuccessful()) {
                $error = trim($process->getErrorOutput()) ?: 'Face recognition process failed.';
                \Log::error('Face recognition failed', [
                    'event_id'  => $event->id,
                    'error'     => $error,
                    'exit_code' => $process->getExitCode(),
                ]);
                throw new RuntimeException($error);
            }

            $output = trim($process->getOutput());
            $lines = explode("\n", $output);
            $jsonCandidate = end($lines);
            $payload = json_decode($jsonCandidate, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                $payload = json_decode($output, true);
            }

            if (! is_array($payload) || ! array_key_exists('matched_ids', $payload)) {
                throw new RuntimeException('Face recognition returned an invalid response.');
            }

            $matched = array_map('intval', $payload['matched_ids'] ?? []);
            $matched = $this->validateMatchedIds($event, $matched);

            return array_values(array_unique(array_filter($matched)));
        } catch (ProcessTimedOutException $e) {
            \Log::error('Face recognition timeout', [
                'event_id' => $event->id,
                'timeout'  => config('face.timeout', 180),
            ]);
            throw new RuntimeException('Face recognition timed out. Please try again.');
        } finally {
            $this->cleanupTempFile($tempImagePath);
            $this->cleanupTempFile($dbConfigPath);
        }
    }

    // ---------------------------------------------------------------
    //  Index
    // ---------------------------------------------------------------

    private function runIndex(Event $event, bool $force): void
    {
        $this->validatePythonEnvironment();

        $script = $this->scriptPath();
        $dbConfigPath = $this->createDatabaseConfigFile();

        try {
            $process = new Process([
                $this->pythonBinary(),
                $script,
                'index',
                '--event-id',
                (string) $event->id,
                '--model-dir',
                (string) config('face.model_dir', storage_path('app/face-models')),
                '--db-config',
                $dbConfigPath,
                $force ? '--force' : '--no-force',
            ], base_path(), [
                'PYTHONIOENCODING' => 'utf-8',
                'PYTHONUNBUFFERED' => '1',
            ]);

            $process->setTimeout((int) config('face.timeout', 180));
            $process->run();

            if (! $process->isSuccessful()) {
                $error = trim($process->getErrorOutput()) ?: 'Face index process failed.';
                \Log::error('Face indexing failed', [
                    'event_id'  => $event->id,
                    'error'     => $error,
                    'exit_code' => $process->getExitCode(),
                ]);
                throw new RuntimeException($error);
            }
        } catch (ProcessTimedOutException $e) {
            \Log::error('Face indexing timeout', ['event_id' => $event->id]);
            throw new RuntimeException('Face indexing timed out. Please try again.');
        } finally {
            $this->cleanupTempFile($dbConfigPath);
        }
    }

    // ---------------------------------------------------------------
    //  Helpers
    // ---------------------------------------------------------------

    private function validatePythonEnvironment(): void
    {
        static $validated = false;

        if ($validated) {
            return;
        }

        $pythonBin = $this->pythonBinary();
        $process = new Process([$pythonBin, '--version']);
        $process->setTimeout(5)->run();

        if (! $process->isSuccessful()) {
            throw new RuntimeException("Python binary not found or invalid: {$pythonBin}");
        }

        $this->scriptPath();

        $modelDir = config('face.model_dir', storage_path('app/face-models'));
        if (! is_dir($modelDir)) {
            throw new RuntimeException("Face recognition model directory not found: {$modelDir}");
        }

        $validated = true;
    }

    private function createTempImageCopy(string $sourcePath, int $eventId): string
    {
        $extension = pathinfo($sourcePath, PATHINFO_EXTENSION);
        $safeFilename = 'face_' . $eventId . '_' . uniqid() . '.' . $extension;
        $tempPath = $this->tempDir . DIRECTORY_SEPARATOR . $safeFilename;

        if (! copy($sourcePath, $tempPath)) {
            throw new RuntimeException('Failed to create temporary image copy');
        }

        chmod($tempPath, 0600);

        return $tempPath;
    }

    private function createDatabaseConfigFile(): string
    {
        $connection = config('database.default');
        $config = [
            'connection' => $connection,
            'host'       => config("database.connections.{$connection}.host"),
            'port'       => config("database.connections.{$connection}.port"),
            'database'   => config("database.connections.{$connection}.database"),
            'username'   => config("database.connections.{$connection}.username"),
            'password'   => config("database.connections.{$connection}.password"),
        ];

        $tempPath = $this->tempDir . DIRECTORY_SEPARATOR . 'db_config_' . uniqid() . '.json';
        file_put_contents($tempPath, json_encode($config, JSON_PRETTY_PRINT));

        chmod($tempPath, 0600);

        return $tempPath;
    }

    private function validateMatchedIds(Event $event, array $matchedIds): array
    {
        if (empty($matchedIds)) {
            return [];
        }

        $validIds = $event->media()->whereIn('id', $matchedIds)->pluck('id')->toArray();

        return array_values(array_intersect($matchedIds, $validIds));
    }

    private function cleanupTempFile(?string $path): void
    {
        if ($path && file_exists($path)) {
            @unlink($path);
        }
    }

    private function pythonBinary(): string
    {
        return (string) config('face.python_bin', 'python');
    }

    private function scriptPath(): string
    {
        $path = (string) config('face.script', base_path('scripts/face_recognition.py'));
        if (! is_file($path)) {
            throw new RuntimeException('Face recognition script not found: ' . $path);
        }

        return $path;
    }
}
