<?php

namespace App\Services;

use App\Exceptions\PlanLimitReachedException;
use App\Exceptions\StorageLimitReachedException;
use App\Models\Event;
use App\Models\DriveImport;
use App\Models\DriveImportFile;
use App\Models\EventMedia;
use App\Jobs\ProcessEventMedia;
use GuzzleHttp\Cookie\CookieJar;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class GoogleDriveImportService
{
    private const ALLOWED_EXTENSIONS = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];
    private const USER_AGENT = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36';

    public function __construct(private EventMediaService $mediaService)
    {
    }

    /**
     * Import media from one or more Google Drive folder links.
     *
     * @return array{imported: int, skipped: int}
     */
    public function import(Event $event, string $links): array
    {
        $imported = 0;
        $skipped = 0;

        $lines = preg_split('/[\r\n,]+/', $links);

        foreach ($lines as $line) {
            $link = trim($line);
            if ($link === '') {
                continue;
            }

            if (! $this->looksLikeFolderLink($link)) {
                $skipped++;
                continue;
            }

            try {
                $this->startImportSession($event, $link);
                $result = $this->runForEvent($event, 5, 15);

                if ($result) {
                    $imported += $result->processed_files ?? 0;
                    $skipped += $result->failed_files ?? 0;
                }
            } catch (\Throwable $e) {
                $skipped++;
            }
        }

        return [
            'imported' => $imported,
            'skipped' => $skipped,
        ];
    }

    public function looksLikeFolderLink(string $link): bool
    {
        return (bool) $this->extractFolderId($link);
    }

    public function startImportSession(Event $event, string $folderLink): DriveImport
    {
        $folderId = $this->extractFolderId($folderLink);
        if (! $folderId) {
            throw new \InvalidArgumentException('Invalid Google Drive folder link.');
        }

        $resourceKey = $this->extractResourceKey($folderLink);

        return DriveImport::create([
            'event_id' => $event->id,
            'folder_link' => $folderLink,
            'folder_id' => $folderId,
            'resource_key' => $resourceKey,
            'status' => 'discovering',
        ]);
    }

    public function runBatch(?Event $event = null, int $batchSize = 10, int $maxSeconds = 30): ?DriveImport
    {
        $start = microtime(true);

        $query = DriveImport::query()
            ->whereIn('status', ['discovering', 'pending', 'running']);

        if ($event) {
            $query->where('event_id', $event->id);
        }

        /** @var DriveImport|null $import */
        $import = $query->orderBy('id')->first();
        if (! $import) {
            return null;
        }

        if (! $import->started_at) {
            $import->started_at = now();
        }

        if ($import->status === 'discovering') {
            $this->discoverFiles($import, $maxSeconds);
        }

        $remainingSeconds = $maxSeconds - (microtime(true) - $start);
        if ($remainingSeconds <= 0) {
            return $import->fresh();
        }

        $this->processFiles($import, $batchSize, $remainingSeconds);

        return $import->fresh();
    }

    public function runForEvent(Event $event, int $batchSize = 5, int $maxSeconds = 15): ?DriveImport
    {
        return $this->runBatch($event, $batchSize, $maxSeconds);
    }

    private function discoverFiles(DriveImport $import, int $maxSeconds): void
    {
        $start = microtime(true);

        $folderId = $import->folder_id;
        if (! $folderId) {
            $import->status = 'failed';
            $import->last_error = 'Folder ID missing.';
            $import->finished_at = now();
            $import->save();
            return;
        }

        $folderUrl = 'https://drive.google.com/drive/folders/' . $folderId;
        if ($import->resource_key) {
            $folderUrl .= '?resourcekey=' . $import->resource_key;
        }

        $result = $this->fetchFolderFileIds($folderUrl);
        if (! empty($result['error'])) {
            $import->status = 'failed';
            $import->last_error = $result['error'];
            $import->finished_at = now();
            $import->save();
            return;
        }

        $ids = $result['ids'] ?? [];
        if (! $ids) {
            $import->status = 'completed';
            $import->finished_at = now();
            $import->save();
            return;
        }

        $existing = DriveImportFile::query()
            ->where('drive_import_id', $import->id)
            ->pluck('file_id')
            ->all();

        $toInsert = [];
        foreach ($ids as $fileId) {
            if (in_array($fileId, $existing, true)) {
                continue;
            }
            $toInsert[] = [
                'drive_import_id' => $import->id,
                'file_id' => $fileId,
                'status' => 'pending',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        $inserted = 0;

        if ($toInsert) {
            foreach (array_chunk($toInsert, 500) as $chunk) {
                DriveImportFile::insert($chunk);
                $inserted += count($chunk);
                if ((microtime(true) - $start) >= $maxSeconds) {
                    break;
                }
            }
        }

        $import->discovered_files = DriveImportFile::query()
            ->where('drive_import_id', $import->id)
            ->count();
        $import->total_files = $import->discovered_files;
        if ($inserted < count($toInsert)) {
            $import->status = 'discovering';
        } else {
            $import->status = 'running';
        }
        $import->save();
    }

    private function processFiles(DriveImport $import, int $batchSize, int $maxSeconds): void
    {
        $start = microtime(true);
        $event = $import->event;
        $import->status = 'running';
        $import->save();

        $tempDir = storage_path('app/tmp/google-drive');
        File::ensureDirectoryExists($tempDir);

        $processedThisRun = 0;

        while ($processedThisRun < $batchSize && (microtime(true) - $start) < $maxSeconds) {
            /** @var DriveImportFile|null $file */
            $file = DriveImportFile::query()
                ->where('drive_import_id', $import->id)
                ->where('status', 'pending')
                ->orderBy('id')
                ->first();

            if (! $file) {
                break;
            }

            $file->status = 'processing';
            $file->save();

            $download = null;

            try {
                $download = $this->downloadFile($file->file_id, $tempDir);
                if (! $download || ! in_array($download['extension'], self::ALLOWED_EXTENSIONS, true)) {
                    throw new \RuntimeException('Unsupported or missing file.');
                }

                $media = $this->mediaService->storeLocalFile($event, $download['path'], $download['filename']);
                $this->processMediaNow($media);
                $this->cleanupTemp($download['path']);

                $file->status = 'completed';
                $file->file_name = $download['filename'];
                $file->mime_type = $download['content_type'] ?? null;
                $file->size = $download['size'] ?? null;
                $file->error = null;
                $file->save();

                $import->processed_files++;
                $processedThisRun++;
            } catch (PlanLimitReachedException|StorageLimitReachedException $exception) {
                $file->status = 'failed';
                $file->error = $exception->getMessage();
                $file->save();
                $import->failed_files++;
                $import->last_error = $exception->getMessage();
                break;
            } catch (\Throwable $exception) {
                $file->status = 'failed';
                $file->error = $exception->getMessage();
                $file->save();
                $import->failed_files++;
                $import->last_error = $exception->getMessage();
            }

            if ($download && isset($download['path'])) {
                $this->cleanupTemp($download['path']);
            }
        }

        if (! DriveImportFile::query()->where('drive_import_id', $import->id)->where('status', 'pending')->exists()) {
            $import->status = $import->failed_files > 0 ? 'completed_with_errors' : 'completed';
            $import->finished_at = now();
        }

        $import->save();
    }

    private function processMediaNow(EventMedia $media): void
    {
        $job = new ProcessEventMedia($media->id);
        app()->call([$job, 'handle']);
    }

    private function extractFileId(string $link): ?string
    {
        if ($link === '') {
            return null;
        }

        if (preg_match('/\/drive(?:\/u\/\d+)?\/folders\//', $link) || str_contains($link, 'folderview?id=')) {
            return null;
        }

        if (preg_match('/\/file\/d\/([A-Za-z0-9_-]+)/', $link, $matches)) {
            return $matches[1];
        }

        if (preg_match('/[?&]id=([A-Za-z0-9_-]+)/', $link, $matches)) {
            return $matches[1];
        }

        if (preg_match('/^[A-Za-z0-9_-]{10,}$/', $link)) {
            return $link;
        }

        return null;
    }

    private function extractFolderId(string $link): ?string
    {
        if ($link === '') {
            return null;
        }

        if (preg_match('/\/drive(?:\/u\/\d+)?\/folders\/([A-Za-z0-9_-]+)/', $link, $matches)) {
            return $matches[1];
        }

        if (preg_match('/folderview\?id=([A-Za-z0-9_-]+)/', $link, $matches)) {
            return $matches[1];
        }

        if (preg_match('/open\?id=([A-Za-z0-9_-]+)/', $link, $matches)) {
            return $matches[1];
        }

        return null;
    }

    private function extractResourceKey(string $link): ?string
    {
        if ($link === '') {
            return null;
        }

        $parts = parse_url($link);
        if (! $parts || empty($parts['query'])) {
            return null;
        }

        parse_str($parts['query'], $query);

        return isset($query['resourcekey']) ? trim((string) $query['resourcekey']) : null;
    }

    private function fetchFolderFileIds(string $folderUrl): array
    {
        $response = Http::timeout(60)
            ->withHeaders(['User-Agent' => self::USER_AGENT])
            ->get($folderUrl);

        if (! $response->ok()) {
            return ['error' => 'Unable to access the Google Drive folder. Make sure the link is public.'];
        }

        $body = $response->body();
        $ids = $this->extractFileIdsFromFolderHtml($body);

        if ($ids) {
            return ['ids' => $ids];
        }

        if ($this->looksLikeLoginPage($body)) {
            return ['error' => 'This Google Drive folder is not public.'];
        }

        return ['error' => 'No supported files found in the Google Drive folder.'];
    }

    private function extractFileIdsFromFolderHtml(string $html): array
    {
        $driveIds = $this->extractFileIdsFromDriveIvd($html);
        if ($driveIds) {
            return $driveIds;
        }

        $normalized = str_replace('\\u002F', '/', $html);
        $normalized = str_replace('\\/', '/', $normalized);
        $normalized = stripslashes($normalized);

        $ids = [];
        $patterns = [
            '/"mimeType":"(image\\/[^"]+|video\\/mp4)".*?"id":"([A-Za-z0-9_-]{10,})"/s',
            '/"id":"([A-Za-z0-9_-]{10,})".*?"mimeType":"(image\\/[^"]+|video\\/mp4)"/s',
        ];

        foreach ($patterns as $pattern) {
            if (! preg_match_all($pattern, $normalized, $matches, PREG_SET_ORDER)) {
                continue;
            }

            foreach ($matches as $match) {
                $id = $match[2] ?? $match[1] ?? null;
                if ($id) {
                    $ids[$id] = true;
                }
            }
        }

        return array_keys($ids);
    }

    private function extractFileIdsFromDriveIvd(string $html): array
    {
        if (! preg_match("/_DRIVE_ivd'\\]\\s*=\\s*'([^']+)'/", $html, $matches)) {
            return [];
        }

        $decoded = stripcslashes($matches[1]);
        $data = json_decode($decoded, true);

        if (! is_array($data)) {
            return [];
        }

        $ids = [];
        $this->collectDriveFileIds($data, $ids);

        return array_keys($ids);
    }

    private function collectDriveFileIds(array $node, array &$ids): void
    {
        if (count($node) > 3 && is_string($node[0]) && is_string($node[3])) {
            $mimeType = $node[3];
            if (str_starts_with($mimeType, 'image/') || $mimeType === 'video/mp4') {
                $ids[$node[0]] = true;
            }
        }

        foreach ($node as $item) {
            if (is_array($item)) {
                $this->collectDriveFileIds($item, $ids);
            }
        }
    }

    private function looksLikeLoginPage(string $html): bool
    {
        if (! str_contains($html, 'ServiceLogin')) {
            return false;
        }

        return str_contains($html, 'identifierId')
            || str_contains($html, 'name="Passwd"')
            || str_contains($html, 'accounts.google.com/signin');
    }

    private function downloadFile(string $fileId, string $tempDir): ?array
    {
        $cookieJar = new CookieJar();
        $tempPath = $tempDir . DIRECTORY_SEPARATOR . 'drive-' . Str::random(12);
        $downloadUrl = 'https://drive.google.com/uc?export=download&id=' . $fileId;

        $response = Http::withHeaders(['User-Agent' => self::USER_AGENT])
            ->withOptions([
                'cookies' => $cookieJar,
                'allow_redirects' => true,
                'sink' => $tempPath,
            ])->timeout(120)->get($downloadUrl);

        if (! $response->ok()) {
            $this->cleanupTemp($tempPath);
            return null;
        }

        $contentType = (string) $response->header('content-type');
        $disposition = $response->header('content-disposition');
        $contentLength = (int) $response->header('content-length');

        if ($this->looksLikeHtml($contentType, $tempPath)) {
            $token = $this->extractConfirmToken(@file_get_contents($tempPath) ?: '');
            if ($token) {
                $response = Http::withHeaders(['User-Agent' => self::USER_AGENT])
                    ->withOptions([
                        'cookies' => $cookieJar,
                        'allow_redirects' => true,
                        'sink' => $tempPath,
                    ])->timeout(120)->get('https://drive.google.com/uc?export=download&confirm=' . $token . '&id=' . $fileId);

                if (! $response->ok()) {
                    $this->cleanupTemp($tempPath);
                    return null;
                }

                $contentType = (string) $response->header('content-type');
                $disposition = $response->header('content-disposition');
                $contentLength = (int) $response->header('content-length');
            }
        }

        if ($this->looksLikeHtml($contentType, $tempPath)) {
            $this->cleanupTemp($tempPath);
            return null;
        }

        $filename = $this->resolveFilename($disposition, $fileId, $contentType);
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

        if (! $extension) {
            $extension = $this->extensionFromContentType($contentType) ?? 'bin';
            $filename .= '.' . $extension;
        }

        $finalPath = $tempDir . DIRECTORY_SEPARATOR . 'drive-' . Str::random(12) . '.' . $extension;
        if (! @rename($tempPath, $finalPath)) {
            $this->cleanupTemp($tempPath);
            return null;
        }

        return [
            'path' => $finalPath,
            'filename' => $filename,
            'extension' => $extension,
            'content_type' => $contentType,
            'size' => $contentLength > 0 ? $contentLength : filesize($finalPath),
        ];
    }

    private function resolveFilename(?string $disposition, string $fileId, ?string $contentType): string
    {
        if ($disposition && preg_match('/filename\*\=UTF-8\'\'([^;]+)/', $disposition, $matches)) {
            return trim(rawurldecode($matches[1]), '"\' ');
        }

        if ($disposition && preg_match('/filename=\"?([^\";]+)\"?/i', $disposition, $matches)) {
            return trim($matches[1], '"\' ');
        }

        $extension = $this->extensionFromContentType($contentType ?? '') ?? 'bin';

        return 'drive-' . $fileId . '.' . $extension;
    }

    private function extensionFromContentType(string $contentType): ?string
    {
        $type = strtolower(trim(strtok($contentType, ';')));

        return match ($type) {
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/webp' => 'webp',
            'video/mp4' => 'mp4',
            default => null,
        };
    }

    private function looksLikeHtml(string $contentType, string $path): bool
    {
        if (str_contains(strtolower($contentType), 'text/html')) {
            return true;
        }

        $sample = @file_get_contents($path, false, null, 0, 256);
        if (! $sample) {
            return false;
        }

        return str_contains(strtolower($sample), '<html') || str_contains(strtolower($sample), '<!doctype html');
    }

    private function extractConfirmToken(string $html): ?string
    {
        if ($html === '') {
            return null;
        }

        if (preg_match('/confirm=([0-9A-Za-z_]+)/', $html, $matches)) {
            return $matches[1];
        }

        return null;
    }

    private function cleanupTemp(string $path): void
    {
        if (is_file($path)) {
            @unlink($path);
        }
    }
}
