<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Authorize.Net payment gateway driver.
 *
 * Uses the Authorize.Net Accept Hosted API to create an embedded
 * payment form token, then verifies the transaction via the API.
 *
 * @see https://developer.authorize.net/api/reference/
 */
class AuthorizeNetDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://apitest.authorize.net/xml/v1/request.api';
    private const PROD_BASE    = 'https://api.authorize.net/xml/v1/request.api';

    private const SANDBOX_HOSTED = 'https://test.authorize.net/payment/payment';
    private const PROD_HOSTED    = 'https://accept.authorize.net/payment/payment';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_login_id',   'label' => 'API Login ID',   'type' => 'text',     'required' => true],
            ['key' => 'transaction_key', 'label' => 'Transaction Key', 'type' => 'password', 'required' => true],
            ['key' => 'signature_key',   'label' => 'Signature Key',   'type' => 'password', 'required' => false,
             'hint' => 'Required for webhook signature verification. Found in Account > Settings > API Credentials & Keys.'],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiLoginId    = $this->credential('api_login_id');
        $transactionKey = $this->credential('transaction_key');
        $apiUrl        = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        // Get a hosted payment page token
        $payload = [
            'getHostedPaymentPageRequest' => [
                'merchantAuthentication' => [
                    'name'           => $apiLoginId,
                    'transactionKey' => $transactionKey,
                ],
                'transactionRequest' => [
                    'transactionType' => 'authCaptureTransaction',
                    'amount'          => number_format((float) $order->amount, 2, '.', ''),
                    'order'           => [
                        'invoiceNumber' => $order->order_number,
                        'description'   => $this->paymentDescription($order),
                    ],
                ],
                'hostedPaymentSettings' => [
                    'setting' => [
                        [
                            'settingName'  => 'hostedPaymentReturnOptions',
                            'settingValue' => json_encode([
                                'showReceipt' => false,
                                'url'         => $callbackUrl,
                                'urlText'     => 'Return',
                                'cancelUrl'   => $callbackUrl . '?cancelled=1',
                            ]),
                        ],
                        [
                            'settingName'  => 'hostedPaymentButtonOptions',
                            'settingValue' => json_encode(['text' => 'Pay Now']),
                        ],
                    ],
                ],
            ],
        ];

        $response = $this->authorizeNetRequest($apiUrl, $payload);
        $json     = $response['json'] ?? [];
        $token    = $json['token'] ?? null;

        if ($token) {
            $hostedUrl = $this->isSandbox() ? self::SANDBOX_HOSTED : self::PROD_HOSTED;

            // Return an HTML form that auto-submits to Authorize.Net
            $eHostedUrl = htmlspecialchars($hostedUrl, ENT_QUOTES, 'UTF-8');
            $eToken     = htmlspecialchars($token, ENT_QUOTES, 'UTF-8');

            $html = <<<HTML
            <form id="authnet-form" method="POST" action="{$eHostedUrl}">
                <input type="hidden" name="token" value="{$eToken}" />
            </form>
            <script>document.getElementById('authnet-form').submit();</script>
            HTML;

            return ['html' => $html];
        }

        $errorMsg = $json['messages']['message'][0]['text'] ?? 'Failed to create Authorize.Net payment page.';
        throw new \RuntimeException("Authorize.Net: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        if (isset($payload['cancelled'])) {
            return PaymentResult::failure('Payment was cancelled.');
        }

        $transId  = $payload['transId'] ?? $payload['x_trans_id'] ?? null;
        $response = $payload['x_response_code'] ?? $payload['responseCode'] ?? null;

        if ($transId && (string) $response === '1') {
            // Verify the transaction server-side before trusting callback
            $verifyResult = $this->getTransactionDetails($transId);
            if ($verifyResult !== null) {
                return $verifyResult;
            }

            return PaymentResult::success(
                transactionId: $transId,
                message: 'Payment completed.',
                metadata: [
                    'authnet_trans_id' => $transId,
                    'authnet_response' => $response,
                ],
            );
        }

        if ($transId) {
            return PaymentResult::pending('Authorize.Net payment received, verifying...', $transId);
        }

        return PaymentResult::failure('Authorize.Net payment not completed.');
    }

    // -----------------------------------------------------------------
    //  Webhook signature verification (HMAC-SHA512)
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $signatureKey = $this->credential('signature_key');
        if (! $signatureKey) {
            \Illuminate\Support\Facades\Log::warning('Authorize.Net signature key not configured -- skipping webhook verification.');
            return true;
        }

        $signature = $headers['x-anet-signature'] ?? $headers['X-ANET-Signature'] ?? '';
        if (! $signature) {
            return false;
        }

        // X-ANET-Signature format: "sha512=HEXHASH"
        $signature = str_ireplace('sha512=', '', $signature);

        $expected = strtoupper(hash_hmac('sha512', $rawBody, hex2bin($signatureKey)));

        return hash_equals($expected, strtoupper($signature));
    }

    // -----------------------------------------------------------------
    //  Webhook handler
    // -----------------------------------------------------------------

    public function handleWebhook(array $payload): PaymentResult
    {
        $eventType = $payload['eventType'] ?? '';
        $webhookId = $payload['payload']['id'] ?? '';

        if (! $eventType || ! $webhookId) {
            return PaymentResult::failure('Invalid Authorize.Net webhook payload.');
        }

        // Handle payment-related events
        if (in_array($eventType, [
            'net.authorize.payment.authcapture.created',
            'net.authorize.payment.capture.created',
            'net.authorize.payment.priorAuthCapture.created',
        ])) {
            // Verify via getTransactionDetails API
            $result = $this->getTransactionDetails($webhookId);
            if ($result !== null) {
                return $result;
            }

            return PaymentResult::success(
                transactionId: $webhookId,
                message: "Authorize.Net webhook: {$eventType}",
                metadata: ['authnet_event_type' => $eventType],
            );
        }

        if ($eventType === 'net.authorize.payment.void.created') {
            return PaymentResult::failure("Authorize.Net: payment voided. Transaction: {$webhookId}");
        }

        if ($eventType === 'net.authorize.payment.refund.created') {
            return PaymentResult::failure("Authorize.Net: payment refunded. Transaction: {$webhookId}");
        }

        return PaymentResult::failure("Unhandled Authorize.Net webhook event: {$eventType}");
    }

    // -----------------------------------------------------------------
    //  Verify payment via getTransactionDetailsRequest
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $transactionId = $order->gateway_transaction_id;
        if (! $transactionId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        $result = $this->getTransactionDetails($transactionId);
        if ($result !== null) {
            return $result;
        }

        return PaymentResult::failure('Authorize.Net: Unable to verify transaction.');
    }

    // -----------------------------------------------------------------
    //  Helpers
    // -----------------------------------------------------------------

    /**
     * Query the Authorize.Net API for transaction details.
     */
    private function getTransactionDetails(string $transactionId): ?PaymentResult
    {
        $apiLoginId     = $this->credential('api_login_id');
        $transactionKey = $this->credential('transaction_key');
        $apiUrl         = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $payload = [
            'getTransactionDetailsRequest' => [
                'merchantAuthentication' => [
                    'name'           => $apiLoginId,
                    'transactionKey' => $transactionKey,
                ],
                'transId' => $transactionId,
            ],
        ];

        try {
            $response = $this->authorizeNetRequest($apiUrl, $payload);
            $json     = $response['json'] ?? [];

            $resultCode = $json['messages']['resultCode'] ?? '';
            if ($resultCode !== 'Ok') {
                return null; // API call failed, let caller handle fallback
            }

            $txn    = $json['transaction'] ?? [];
            $status = $txn['transactionStatus'] ?? '';

            if (in_array($status, ['settledSuccessfully', 'capturedPendingSettlement', 'authorizedPendingCapture'])) {
                return PaymentResult::success(
                    transactionId: $transactionId,
                    message: "Payment verified. Status: {$status}.",
                    metadata: [
                        'authnet_trans_id' => $transactionId,
                        'authnet_status'   => $status,
                    ],
                );
            }

            if (in_array($status, ['voided', 'declined', 'expired', 'failedReview'])) {
                return PaymentResult::failure("Authorize.Net transaction {$status}.");
            }

            // For other statuses (FDSPendingReview, etc.), return pending
            if ($status) {
                return PaymentResult::pending("Authorize.Net transaction status: {$status}", $transactionId);
            }
        } catch (\Exception $e) {
            // API call failed, return null to let caller use fallback logic
            return null;
        }

        return null;
    }

    private function authorizeNetRequest(string $url, array $data): array
    {
        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => json_encode($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/json', 'Accept: application/json'],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        $body  = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("Authorize.Net cURL error: {$error}");
        }

        // Authorize.Net may return BOM in JSON -- strip it
        $body = preg_replace('/^\xEF\xBB\xBF/', '', $body);

        return ['body' => $body, 'json' => json_decode($body, true)];
    }
}
