<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Braintree payment gateway driver.
 *
 * Uses the Braintree Gateway XML/HTTP API via cURL.
 * Generates a client token for Drop-in UI, then processes the
 * payment nonce returned from the client side.
 *
 * @see https://developer.paypal.com/braintree/docs/start/overview
 */
class BraintreeDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://api.sandbox.braintreegateway.com:443/merchants';
    private const PROD_BASE    = 'https://api.braintreegateway.com:443/merchants';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'merchant_id', 'label' => 'Merchant ID', 'type' => 'text',     'required' => true],
            ['key' => 'public_key',  'label' => 'Public Key',  'type' => 'text',     'required' => true],
            ['key' => 'private_key', 'label' => 'Private Key', 'type' => 'password', 'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- generate client token and return Drop-in UI form
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $merchantId = $this->credential('merchant_id');
        $publicKey  = $this->credential('public_key');
        $privateKey = $this->credential('private_key');

        // Generate a client token
        $tokenXml = '<client-token><version>2</version></client-token>';
        $response = $this->braintreeRequest(
            'POST',
            "/{$merchantId}/client_token",
            $tokenXml,
            $publicKey,
            $privateKey,
        );

        $clientToken = '';
        if (preg_match('/<value>(.*?)<\/value>/s', $response['body'], $matches)) {
            $clientToken = trim($matches[1]);
        }

        if (! $clientToken) {
            throw new \RuntimeException('Braintree: Unable to generate client token.');
        }

        $amount      = number_format((float) $order->amount, 2, '.', '');
        $orderNumber = $order->order_number;

        // Escape all values for safe JavaScript embedding
        $jsClientToken = json_encode($clientToken, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);
        $jsCallbackUrl = json_encode($callbackUrl, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);
        $jsOrderNumber = json_encode($orderNumber, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);
        $jsAmount      = json_encode($amount, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);
        $jsCsrfToken   = json_encode(csrf_token(), JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);

        // Return HTML with Braintree Drop-in UI
        $html = <<<HTML
        <div id="braintree-dropin-container"></div>
        <button id="braintree-submit-btn" class="btn btn-primary" disabled>Pay Now</button>
        <script src="https://js.braintreegateway.com/web/dropin/1.40.2/js/dropin.min.js"></script>
        <script>
            braintree.dropin.create({
                authorization: {$jsClientToken},
                container: '#braintree-dropin-container',
                card: { cardholderName: { required: true } }
            }, function(err, instance) {
                if (err) { console.error(err); return; }
                var btn = document.getElementById('braintree-submit-btn');
                btn.disabled = false;
                btn.addEventListener('click', function() {
                    btn.disabled = true;
                    instance.requestPaymentMethod(function(requestErr, payload) {
                        if (requestErr) { btn.disabled = false; console.error(requestErr); return; }
                        var form = document.createElement('form');
                        form.method = 'POST';
                        form.action = {$jsCallbackUrl};
                        var fields = {
                            payment_method_nonce: payload.nonce,
                            order_number: {$jsOrderNumber},
                            amount: {$jsAmount},
                            _token: {$jsCsrfToken}
                        };
                        for (var key in fields) {
                            var input = document.createElement('input');
                            input.type = 'hidden'; input.name = key; input.value = fields[key];
                            form.appendChild(input);
                        }
                        document.body.appendChild(form);
                        form.submit();
                    });
                });
            });
        </script>
        HTML;

        return ['html' => $html];
    }

    // -----------------------------------------------------------------
    //  Handle callback -- create transaction with nonce
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        $nonce = $payload['payment_method_nonce'] ?? null;

        if (! $nonce) {
            return PaymentResult::failure('Missing Braintree payment nonce.');
        }

        // Retrieve the order amount server-side (never trust client-submitted amount)
        $orderNumber = $payload['order_number'] ?? null;
        $order       = $orderNumber ? Order::where('order_number', $orderNumber)->first() : null;

        if (! $order) {
            return PaymentResult::failure('Braintree: could not find associated order.');
        }

        $amount = number_format((float) $order->amount, 2, '.', '');

        $merchantId = $this->credential('merchant_id');
        $publicKey  = $this->credential('public_key');
        $privateKey = $this->credential('private_key');

        // Escape values to prevent XML injection
        $escapedAmount = htmlspecialchars($amount, ENT_XML1, 'UTF-8');
        $escapedNonce  = htmlspecialchars($nonce, ENT_XML1, 'UTF-8');

        $transactionXml = <<<XML
        <transaction>
            <type>sale</type>
            <amount>{$escapedAmount}</amount>
            <payment-method-nonce>{$escapedNonce}</payment-method-nonce>
            <options>
                <submit-for-settlement>true</submit-for-settlement>
            </options>
        </transaction>
        XML;

        $response = $this->braintreeRequest(
            'POST',
            "/{$merchantId}/transactions",
            $transactionXml,
            $publicKey,
            $privateKey,
        );

        $body = $response['body'];

        // Parse the transaction ID and status from XML response
        $transactionId = '';
        $status        = '';

        if (preg_match('/<id>(.*?)<\/id>/', $body, $idMatch)) {
            $transactionId = $idMatch[1];
        }
        if (preg_match('/<status>(.*?)<\/status>/', $body, $statusMatch)) {
            $status = $statusMatch[1];
        }

        if (in_array($status, ['authorized', 'submitted_for_settlement', 'settled', 'settling'])) {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Payment processed successfully.',
                metadata: [
                    'braintree_status' => $status,
                ],
            );
        }

        // Check for API error message
        $errorMsg = '';
        if (preg_match('/<message>(.*?)<\/message>/', $body, $msgMatch)) {
            $errorMsg = $msgMatch[1];
        }

        return PaymentResult::failure("Braintree transaction failed. Status: {$status}. {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle webhook -- Braintree sends bt_signature + bt_payload
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $privateKey = $this->credential('private_key');
        if (! $privateKey) {
            \Illuminate\Support\Facades\Log::warning('Braintree private key not configured -- skipping webhook signature verification.');
            return true;
        }

        // Parse form-encoded body to extract bt_signature and bt_payload
        parse_str($rawBody, $params);
        $btSignature = $params['bt_signature'] ?? '';
        $btPayload   = $params['bt_payload'] ?? '';

        if (! $btSignature || ! $btPayload) {
            return false;
        }

        // bt_signature format: "public_key|hmac_signature"
        $parts = explode('|', $btSignature, 2);
        if (count($parts) !== 2) {
            return false;
        }

        $receivedSignature = $parts[1];

        // Braintree derives the HMAC key: sha256(sha256(private_key) + public_key)
        $publicKey = $this->credential('public_key');
        $hmacKey   = hash('sha256', hash('sha256', $privateKey, true) . $publicKey);
        $expectedSignature = hash_hmac('sha1', $btPayload, $hmacKey);

        return hash_equals($expectedSignature, $receivedSignature);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $btPayload = $payload['bt_payload'] ?? '';

        if (! $btPayload) {
            return PaymentResult::failure('Missing Braintree webhook payload (bt_payload).');
        }

        // Decode the base64 payload to get notification XML
        $xml = base64_decode($btPayload, true);
        if ($xml === false) {
            return PaymentResult::failure('Invalid Braintree webhook payload encoding.');
        }

        // Parse notification XML fields
        $kind          = '';
        $transactionId = '';
        $status        = '';

        if (preg_match('/<kind>(.*?)<\/kind>/s', $xml, $m)) {
            $kind = trim($m[1]);
        }
        if (preg_match('/<id>(.*?)<\/id>/', $xml, $m)) {
            $transactionId = trim($m[1]);
        }
        if (preg_match('/<status>(.*?)<\/status>/', $xml, $m)) {
            $status = trim($m[1]);
        }

        // Handle transaction settlement webhooks
        if ($kind === 'transaction_settled') {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Braintree webhook: transaction settled.',
                metadata: [
                    'braintree_kind'   => $kind,
                    'braintree_status' => $status,
                ],
            );
        }

        if ($kind === 'transaction_settlement_declined') {
            return PaymentResult::failure("Braintree webhook: settlement declined. Transaction: {$transactionId}");
        }

        // Handle disbursement notifications
        if ($kind === 'disbursement') {
            return PaymentResult::success(
                transactionId: $transactionId ?: 'disbursement',
                message: 'Braintree webhook: disbursement received.',
                metadata: ['braintree_kind' => $kind],
            );
        }

        return PaymentResult::failure("Braintree webhook received. Kind: {$kind}");
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $transactionId = $order->gateway_transaction_id;
        if (! $transactionId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        $merchantId = $this->credential('merchant_id');
        $publicKey  = $this->credential('public_key');
        $privateKey = $this->credential('private_key');

        $response = $this->braintreeRequest(
            'GET',
            "/{$merchantId}/transactions/{$transactionId}",
            '',
            $publicKey,
            $privateKey,
        );

        $body   = $response['body'];
        $status = '';

        if (preg_match('/<status>(.*?)<\/status>/', $body, $match)) {
            $status = $match[1];
        }

        if (in_array($status, ['authorized', 'submitted_for_settlement', 'settled', 'settling'])) {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: "Payment verified. Status: {$status}.",
            );
        }

        return PaymentResult::failure("Payment not verified. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Braintree HTTP helper (XML-based API)
    // -----------------------------------------------------------------

    private function braintreeRequest(string $method, string $path, string $xmlBody, string $publicKey, string $privateKey): array
    {
        $url = $this->baseUrl() . $path;
        $ch  = curl_init();

        $headers = [
            'Content-Type: application/xml',
            'Accept: application/xml',
            'X-ApiVersion: 6',
            'User-Agent: Braintree PHP',
        ];

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_USERPWD        => $publicKey . ':' . $privateKey,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        if ($method === 'POST' && $xmlBody) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlBody);
        } elseif ($method === 'PUT' && $xmlBody) {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlBody);
        }

        $body   = curl_exec($ch);
        $status = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("Braintree cURL error: {$error}");
        }

        return ['status' => $status, 'body' => $body];
    }

    private function baseUrl(): string
    {
        return $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;
    }
}
