<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Cashfree payment gateway driver (India).
 *
 * Uses the Cashfree Payment Gateway API to create an order session
 * and redirect the customer.
 *
 * @see https://docs.cashfree.com/docs/payment-gateway
 */
class CashfreeDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://sandbox.cashfree.com/pg';
    private const PROD_BASE    = 'https://api.cashfree.com/pg';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'app_id',     'label' => 'App ID',     'type' => 'text',     'required' => true],
            ['key' => 'secret_key', 'label' => 'Secret Key', 'type' => 'password', 'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $appId     = $this->credential('app_id');
        $secretKey = $this->credential('secret_key');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $orderId = 'CF_' . $order->order_number . '_' . time();

        $payload = [
            'order_id'       => $orderId,
            'order_amount'   => (float) $order->amount,
            'order_currency' => strtoupper($order->currency ?? 'INR'),
            'customer_details' => [
                'customer_id'    => (string) ($order->user_id ?? 'guest_' . time()),
                'customer_name'  => $order->user->name ?? 'Customer',
                'customer_email' => $order->user->email ?? throw new \RuntimeException('Cashfree requires a customer email address.'),
                'customer_phone' => '9999999999',
            ],
            'order_meta' => [
                'return_url'   => $callbackUrl . '?order_id={order_id}',
                'notify_url'   => route('payment.webhook', ['gateway' => 'cashfree']),
            ],
            'order_note' => $this->paymentDescription($order),
        ];

        $response = $this->httpRequest('POST', $baseUrl . '/orders', $payload, [
            'x-client-id: ' . $appId,
            'x-client-secret: ' . $secretKey,
            'x-api-version: 2023-08-01',
        ]);

        $json = $response['json'] ?? [];

        if (isset($json['payment_session_id'])) {
            // Cashfree returns a session ID; build the checkout URL
            $sessionId = $json['payment_session_id'];
            $env       = $this->isSandbox() ? 'sandbox' : 'production';

            $jsEnv       = json_encode($env, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);
            $jsSessionId = json_encode($sessionId, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);

            $html = <<<HTML
            <script src="https://sdk.cashfree.com/js/v3/cashfree.js"></script>
            <script>
                const cashfree = Cashfree({ mode: {$jsEnv} });
                cashfree.checkout({
                    paymentSessionId: {$jsSessionId},
                    redirectTarget: '_self'
                });
            </script>
            HTML;

            return ['html' => $html];
        }

        $errorMsg = $json['message'] ?? 'Failed to create Cashfree order.';
        throw new \RuntimeException("Cashfree: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $orderId = $payload['order_id'] ?? null;

        if (! $orderId) {
            return PaymentResult::failure('Missing Cashfree order ID.');
        }

        // Verify order status
        $appId     = $this->credential('app_id');
        $secretKey = $this->credential('secret_key');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $response = $this->httpRequest('GET', $baseUrl . "/orders/{$orderId}", [], [
            'x-client-id: ' . $appId,
            'x-client-secret: ' . $secretKey,
            'x-api-version: 2023-08-01',
        ]);

        $json   = $response['json'] ?? [];
        $status = $json['order_status'] ?? '';

        if ($status === 'PAID') {
            return PaymentResult::success(
                transactionId: $json['cf_order_id'] ?? $orderId,
                message: 'Payment successful.',
                metadata: [
                    'cashfree_order_id' => $orderId,
                    'cf_order_id'       => $json['cf_order_id'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("Cashfree order status: {$status}");
    }

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $secretKey = $this->credential('secret_key');
        if (! $secretKey) {
            \Illuminate\Support\Facades\Log::warning('Cashfree webhook secret not configured -- cannot verify signature.');
            return false;
        }

        $timestamp = $headers['x-cashfree-timestamp'] ?? $headers['X-Cashfree-Timestamp'] ?? '';
        $signature = $headers['x-cashfree-signature'] ?? $headers['X-Cashfree-Signature'] ?? '';

        if (! $timestamp || ! $signature) {
            return false;
        }

        $signedPayload = $timestamp . $rawBody;
        $expected = base64_encode(hash_hmac('sha256', $signedPayload, $secretKey, true));

        return hash_equals($expected, $signature);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $type = $payload['type'] ?? '';
        $data = $payload['data'] ?? [];

        if ($type === 'PAYMENT_SUCCESS_WEBHOOK') {
            $order = $data['order'] ?? [];

            return PaymentResult::success(
                transactionId: $order['order_id'] ?? '',
                message: 'Cashfree webhook confirmed payment.',
                metadata: ['cashfree_event' => $type],
            );
        }

        return PaymentResult::failure("Unhandled Cashfree webhook: {$type}");
    }
}
