<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * CinetPay payment gateway driver (West Africa).
 *
 * Uses the CinetPay API to create a payment and redirect the customer.
 *
 * @see https://docs.cinetpay.com/
 */
class CinetPayDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://api-checkout.cinetpay.com/v2';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_key',    'label' => 'API Key',    'type' => 'text',     'required' => true],
            ['key' => 'site_id',    'label' => 'Site ID',    'type' => 'text',     'required' => true],
            ['key' => 'secret_key', 'label' => 'Secret Key', 'type' => 'password', 'required' => false,
             'hint' => 'Optional. Only needed for webhook signature verification.'],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiKey = $this->credential('api_key');
        $siteId = $this->credential('site_id');

        $transactionId = 'CINET_' . $order->order_number . '_' . time();

        $payload = [
            'apikey'         => $apiKey,
            'site_id'        => $siteId,
            'transaction_id' => $transactionId,
            'amount'         => (int) round($order->amount),
            'currency'       => strtoupper($order->currency ?? 'XOF'),
            'description'    => $this->paymentDescription($order),
            'return_url'     => $callbackUrl,
            'notify_url'     => route('payment.webhook', ['gateway' => 'cinetpay']),
            'channels'       => 'ALL',
            'metadata'       => json_encode([
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ]),
            'customer_name'    => $order->user->name ?? 'Customer',
            'customer_email'   => $order->user->email ?? throw new \RuntimeException('CinetPay requires a customer email address.'),
            'customer_surname' => 'User',
        ];

        $response = $this->httpRequest('POST', self::API_BASE . '/payment', $payload);
        $json     = $response['json'] ?? [];

        if (($json['code'] ?? '') === '201' && isset($json['data']['payment_url'])) {
            return ['redirect_url' => $json['data']['payment_url']];
        }

        $errorMsg = $json['message'] ?? 'Failed to create CinetPay payment.';
        throw new \RuntimeException("CinetPay: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $transactionId = $payload['transaction_id'] ?? $payload['cpm_trans_id'] ?? null;

        if (! $transactionId) {
            return PaymentResult::failure('Missing CinetPay transaction ID.');
        }

        // Verify by checking transaction status
        $apiKey = $this->credential('api_key');
        $siteId = $this->credential('site_id');

        $response = $this->httpRequest('POST', self::API_BASE . '/payment/check', [
            'apikey'         => $apiKey,
            'site_id'        => $siteId,
            'transaction_id' => $transactionId,
        ]);

        $json     = $response['json'] ?? [];
        $respCode = $json['code'] ?? '';
        $status   = $json['data']['status'] ?? '';

        if ($respCode !== '00') {
            return PaymentResult::failure("CinetPay check API error (code: {$respCode}): " . ($json['message'] ?? 'Unknown error'));
        }

        if ($status === 'ACCEPTED') {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Payment accepted.',
                metadata: [
                    'cinetpay_transaction_id' => $transactionId,
                    'cinetpay_payment_method' => $json['data']['payment_method'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("CinetPay transaction status: {$status}");
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $transactionId = $payload['cpm_trans_id'] ?? null;

        if (! $transactionId) {
            return PaymentResult::failure('Missing CinetPay transaction ID in webhook.');
        }

        // Always verify with the check API -- do not trust webhook payload alone
        return $this->handleCallback(['transaction_id' => $transactionId]);
    }
}
