<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Flutterwave (Rave) payment gateway driver.
 *
 * Uses the Flutterwave v3 Standard Payment API via cURL.
 * Creates a payment link, redirects the customer, then verifies
 * the transaction on callback.
 *
 * @see https://developer.flutterwave.com/docs/collecting-payments/standard/
 */
class FlutterwaveDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://api.flutterwave.com/v3';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'public_key',     'label' => 'Public Key',     'type' => 'text',     'required' => true],
            ['key' => 'secret_key',     'label' => 'Secret Key',     'type' => 'password', 'required' => true],
            ['key' => 'encryption_key', 'label' => 'Encryption Key', 'type' => 'password', 'required' => false,
             'hint' => 'Optional. Only needed for inline/encrypted payment flows.'],
            ['key' => 'secret_hash',   'label' => 'Webhook Secret Hash', 'type' => 'password', 'required' => false,
             'hint' => 'Optional. Only needed if you set up webhooks in Flutterwave Dashboard.'],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- create Flutterwave standard payment
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $secretKey = $this->credential('secret_key');

        $txRef = 'FLW_' . $order->order_number . '_' . time();

        $payload = [
            'tx_ref'       => $txRef,
            'amount'       => (float) $order->amount,
            'currency'     => strtoupper($order->currency ?? 'NGN'),
            'redirect_url' => $callbackUrl,
            'payment_options' => 'card,mobilemoney,ussd,banktransfer',
            'customer' => [
                'email' => $order->user->email ?? throw new \RuntimeException('Flutterwave requires a customer email address.'),
                'name'  => $order->user->name ?? 'Customer',
            ],
            'customizations' => [
                'title'       => config('app.name'),
                'description' => $this->paymentDescription($order),
            ],
            'meta' => [
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ],
        ];

        $response = $this->flwRequest('POST', '/payments', $payload, $secretKey);
        $json     = $response['json'] ?? [];

        if (($json['status'] ?? '') === 'success' && isset($json['data']['link'])) {
            return ['redirect_url' => $json['data']['link']];
        }

        $errorMsg = $json['message'] ?? 'Failed to create Flutterwave payment.';
        throw new \RuntimeException("Flutterwave: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle callback -- verify transaction
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        $status        = $payload['status'] ?? '';
        $transactionId = $payload['transaction_id'] ?? null;
        $txRef         = $payload['tx_ref'] ?? null;

        if ($status === 'cancelled') {
            return PaymentResult::failure('Payment was cancelled by the customer.');
        }

        if (! $transactionId && ! $txRef) {
            return PaymentResult::failure('Missing Flutterwave transaction ID.');
        }

        // Verify the transaction via the correct endpoint
        $secretKey = $this->credential('secret_key');

        if ($transactionId) {
            // Numeric transaction ID: use /transactions/{id}/verify
            $response = $this->flwRequest('GET', "/transactions/{$transactionId}/verify", [], $secretKey);
        } else {
            // tx_ref only: use /transactions/verify_by_reference
            $response = $this->flwRequest('GET', "/transactions/verify_by_reference?tx_ref=" . urlencode($txRef), [], $secretKey);
        }

        $json = $response['json'] ?? [];

        if (($json['status'] ?? '') === 'success' && ($json['data']['status'] ?? '') === 'successful') {
            // Verify amount and currency match the order to prevent fraud
            $paidAmount   = (float) ($json['data']['amount'] ?? 0);
            $paidCurrency = strtoupper($json['data']['currency'] ?? '');
            $orderNumber  = $json['data']['meta']['order_number'] ?? $json['data']['tx_ref'] ?? null;

            if ($orderNumber) {
                $order = \App\Models\Order::where('order_number', $orderNumber)
                    ->orWhere('order_number', explode('_', $orderNumber)[1] ?? '')
                    ->first();

                if ($order) {
                    $expectedAmount   = (float) $order->amount;
                    $expectedCurrency = strtoupper($order->currency ?? 'NGN');

                    if (abs($paidAmount - $expectedAmount) > 0.01) {
                        return PaymentResult::failure("Flutterwave amount mismatch: expected {$expectedAmount}, got {$paidAmount}.");
                    }
                    if ($paidCurrency && $expectedCurrency && $paidCurrency !== $expectedCurrency) {
                        return PaymentResult::failure("Flutterwave currency mismatch: expected {$expectedCurrency}, got {$paidCurrency}.");
                    }
                }
            }

            return PaymentResult::success(
                transactionId: (string) ($json['data']['id'] ?? $transactionId ?? $txRef),
                message: 'Payment verified successfully.',
                metadata: [
                    'flw_ref'         => $json['data']['flw_ref'] ?? null,
                    'tx_ref'          => $json['data']['tx_ref'] ?? null,
                    'payment_type'    => $json['data']['payment_type'] ?? null,
                    'amount_settled'  => $json['data']['amount_settled'] ?? null,
                    'currency'        => $json['data']['currency'] ?? null,
                ],
            );
        }

        $txStatus = $json['data']['status'] ?? 'unknown';
        return PaymentResult::failure("Flutterwave transaction not successful. Status: {$txStatus}");
    }

    // -----------------------------------------------------------------
    //  Handle webhook (charge.completed)
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $secretHash = $this->credential('secret_hash');
        if (! $secretHash) {
            \Illuminate\Support\Facades\Log::warning('Flutterwave secret hash not configured -- skipping webhook signature verification.');
            return true;
        }

        $signature = $headers['verif-hash'] ?? $headers['Verif-Hash'] ?? '';

        return hash_equals($secretHash, $signature);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $event = $payload['event'] ?? '';
        $data  = $payload['data'] ?? [];

        if ($event === 'charge.completed' && ($data['status'] ?? '') === 'successful') {
            return PaymentResult::success(
                transactionId: (string) ($data['id'] ?? ''),
                message: 'Webhook confirmed successful charge.',
                metadata: [
                    'flw_ref'      => $data['flw_ref'] ?? null,
                    'tx_ref'       => $data['tx_ref'] ?? null,
                    'payment_type' => $data['payment_type'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("Unhandled Flutterwave webhook event: {$event}");
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $transactionId = $order->gateway_transaction_id;
        if (! $transactionId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        return $this->handleCallback(['transaction_id' => $transactionId]);
    }

    // -----------------------------------------------------------------
    //  Flutterwave HTTP helper
    // -----------------------------------------------------------------

    private function flwRequest(string $method, string $path, array $data, string $secretKey): array
    {
        $url = self::API_BASE . $path;
        $ch  = curl_init();

        $headers = [
            'Authorization: Bearer ' . $secretKey,
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $body   = curl_exec($ch);
        $status = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("Flutterwave cURL error: {$error}");
        }

        return ['status' => $status, 'body' => $body, 'json' => json_decode($body, true)];
    }
}
