<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Instamojo payment gateway driver (India).
 *
 * Creates a payment request via the Instamojo API and redirects
 * the customer to the Instamojo payment page.
 *
 * @see https://docs.instamojo.com/reference/create-a-payment-request
 */
class InstamojoDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://test.instamojo.com/api/1.1';
    private const PROD_BASE    = 'https://www.instamojo.com/api/1.1';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_key',    'label' => 'API Key',    'type' => 'text',     'required' => true],
            ['key' => 'auth_token', 'label' => 'Auth Token', 'type' => 'password', 'required' => true],
            ['key' => 'salt',       'label' => 'Salt',       'type' => 'password', 'required' => false,
             'hint' => 'Optional. Only needed for webhook signature verification.'],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiKey    = $this->credential('api_key');
        $authToken = $this->credential('auth_token');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $payload = [
            'purpose'      => $this->paymentDescription($order),
            'amount'       => number_format((float) $order->amount, 2, '.', ''),
            'buyer_name'   => $order->user->name ?? 'Customer',
            'email'        => $order->user->email ?? throw new \RuntimeException('Instamojo requires a customer email address.'),
            'redirect_url' => $callbackUrl,
            'webhook'      => route('payment.webhook', ['gateway' => 'instamojo']),
            'allow_repeated_payments' => false,
            'send_email'   => false,
        ];

        $response = $this->httpFormPost($baseUrl . '/payment-requests/', $payload, [
            'X-Api-Key: ' . $apiKey,
            'X-Auth-Token: ' . $authToken,
        ]);

        $json = $response['json'] ?? [];

        if (($json['success'] ?? false) === true && isset($json['payment_request']['longurl'])) {
            return ['redirect_url' => $json['payment_request']['longurl']];
        }

        $errorMsg = isset($json['message']) ? json_encode($json['message']) : 'Failed to create Instamojo payment request.';
        throw new \RuntimeException("Instamojo: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $paymentId        = $payload['payment_id'] ?? null;
        $paymentRequestId = $payload['payment_request_id'] ?? null;
        $paymentStatus    = $payload['payment_status'] ?? '';

        if (! $paymentId || ! $paymentRequestId) {
            return PaymentResult::failure('Missing Instamojo payment parameters.');
        }

        $apiKey    = $this->credential('api_key');
        $authToken = $this->credential('auth_token');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $response = $this->httpRequest('GET', $baseUrl . "/payment-requests/{$paymentRequestId}/{$paymentId}/", [], [
            'X-Api-Key: ' . $apiKey,
            'X-Auth-Token: ' . $authToken,
        ]);

        $json   = $response['json'] ?? [];
        $status = $json['payment_request']['payment']['status'] ?? $paymentStatus;

        if ($status === 'Credit') {
            return PaymentResult::success(
                transactionId: $paymentId,
                message: 'Payment successful.',
                metadata: [
                    'instamojo_payment_id'  => $paymentId,
                    'instamojo_request_id'  => $paymentRequestId,
                ],
            );
        }

        return PaymentResult::failure("Instamojo payment status: {$status}");
    }

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $salt = $this->credential('salt');
        if (! $salt) {
            \Illuminate\Support\Facades\Log::warning('Instamojo webhook salt not configured -- skipping signature verification.');
            return true;
        }

        // Parse form-encoded webhook body
        parse_str($rawBody, $params);

        $mac = $params['mac'] ?? '';
        if (! $mac) {
            return false;
        }

        unset($params['mac']);
        ksort($params);
        $message     = implode('|', $params);
        $expectedMac = hash_hmac('sha1', $message, $salt);

        return hash_equals($expectedMac, $mac);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $paymentId = $payload['payment_id'] ?? null;
        $status    = $payload['status'] ?? '';

        if ($status === 'Credit') {
            return PaymentResult::success(
                transactionId: $paymentId ?? '',
                message: 'Instamojo webhook confirmed payment.',
                metadata: ['instamojo_status' => $status],
            );
        }

        return PaymentResult::failure("Instamojo webhook status: {$status}");
    }
}
