<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * iyzico payment gateway driver (Turkey).
 *
 * Uses the iyzico Checkout Form API to create a payment session
 * and redirect the customer.
 *
 * @see https://dev.iyzipay.com/en/checkout-form
 */
class IyzicoDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://sandbox-api.iyzipay.com';
    private const PROD_BASE    = 'https://api.iyzipay.com';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_key',    'label' => 'API Key',    'type' => 'text',     'required' => true],
            ['key' => 'secret_key', 'label' => 'Secret Key', 'type' => 'password', 'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiKey    = $this->credential('api_key');
        $secretKey = $this->credential('secret_key');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $conversationId = 'iyz_' . $order->order_number;
        $price          = number_format((float) $order->amount, 2, '.', '');

        $payload = [
            'locale'         => 'en',
            'conversationId' => $conversationId,
            'price'          => $price,
            'paidPrice'      => $price,
            'currency'       => strtoupper($order->currency ?? 'TRY'),
            'basketId'       => $order->order_number,
            'paymentGroup'   => 'PRODUCT',
            'callbackUrl'    => $callbackUrl,
            'enabledInstallments' => [1],
            'buyer' => [
                'id'                  => (string) ($order->user_id ?? 'GUEST'),
                'name'                => $order->user->name ?? 'Customer',
                'surname'             => 'User',
                'email'               => $order->user->email ?? throw new \RuntimeException('iyzico requires a customer email address.'),
                'identityNumber'      => '11111111111',
                'registrationAddress' => 'N/A',
                'city'                => 'N/A',
                'country'             => 'N/A',
                'ip'                  => request()->ip() ?? '127.0.0.1',
            ],
            'shippingAddress' => [
                'contactName' => $order->user->name ?? 'Customer',
                'city'        => 'N/A',
                'country'     => 'N/A',
                'address'     => 'N/A',
            ],
            'billingAddress' => [
                'contactName' => $order->user->name ?? 'Customer',
                'city'        => 'N/A',
                'country'     => 'N/A',
                'address'     => 'N/A',
            ],
            'basketItems' => [
                [
                    'id'            => 'ITEM_' . $order->order_number,
                    'name'          => $this->paymentDescription($order),
                    'category1'     => 'Photography',
                    'itemType'      => 'VIRTUAL',
                    'price'         => $price,
                ],
            ],
        ];

        // Generate authorization header using PKI string (NOT JSON)
        $authHeaders = $this->iyzicoAuthHeaders($apiKey, $secretKey, $payload);

        $response = $this->httpRequest('POST', $baseUrl . '/payment/iyzipos/checkoutform/initialize/auth/ecom', $payload, $authHeaders);

        $json = $response['json'] ?? [];

        if (($json['status'] ?? '') === 'success' && isset($json['paymentPageUrl'])) {
            return ['redirect_url' => $json['paymentPageUrl']];
        }

        // Some versions return checkoutFormContent (HTML) instead
        if (isset($json['checkoutFormContent'])) {
            return ['html' => $json['checkoutFormContent']];
        }

        $errorMsg = $json['errorMessage'] ?? 'Failed to initialize iyzico checkout.';
        throw new \RuntimeException("iyzico: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $token = $payload['token'] ?? null;

        if (! $token) {
            return PaymentResult::failure('Missing iyzico token in callback.');
        }

        $apiKey    = $this->credential('api_key');
        $secretKey = $this->credential('secret_key');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        $checkPayload = ['locale' => 'en', 'token' => $token];
        $authHeaders  = $this->iyzicoAuthHeaders($apiKey, $secretKey, $checkPayload);

        $response = $this->httpRequest('POST', $baseUrl . '/payment/iyzipos/checkoutform/auth/ecom/detail', $checkPayload, $authHeaders);

        $json          = $response['json'] ?? [];
        $paymentStatus = $json['paymentStatus'] ?? $json['status'] ?? '';

        if ($paymentStatus === 'SUCCESS' || ($json['status'] ?? '') === 'success') {
            return PaymentResult::success(
                transactionId: $json['paymentId'] ?? $token,
                message: 'Payment completed.',
                metadata: [
                    'iyzico_payment_id'         => $json['paymentId'] ?? null,
                    'iyzico_payment_transaction' => $json['paymentTransactionId'] ?? null,
                ],
            );
        }

        $errorMsg = $json['errorMessage'] ?? "Payment status: {$paymentStatus}";
        return PaymentResult::failure("iyzico: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  iyzico PKI string and auth helpers
    // -----------------------------------------------------------------

    /**
     * Generate iyzico authorization headers using PKI string format.
     *
     * iyzico requires a bracket-enclosed key=value string (PKI format),
     * NOT the JSON payload, for HMAC signature generation.
     *
     * @see https://dev.iyzipay.com/en/api-authentication
     */
    private function iyzicoAuthHeaders(string $apiKey, string $secretKey, array $payload): array
    {
        $pkiString = $this->buildPkiString($payload);
        $randomKey = (string) (microtime(true) . mt_rand(1, 999999));
        $hashStr   = $apiKey . $randomKey . $secretKey . $pkiString;
        $hash      = base64_encode(hash('sha1', $hashStr, true));
        $authValue = 'IYZWS ' . $apiKey . ':' . $hash;

        return [
            'Authorization: ' . $authValue,
            'x-iyzi-rnd: ' . $randomKey,
        ];
    }

    /**
     * Build iyzico PKI (bracket-enclosed) string from an array.
     *
     * Format: [key=value, key2=[nested=val], key3=[[item1=v1], [item2=v2]]]
     */
    private function buildPkiString(array $data): string
    {
        $parts = [];
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                if ($this->isIndexedArray($value)) {
                    // Indexed array: check if items are arrays (objects) or scalars
                    $subParts = [];
                    foreach ($value as $item) {
                        if (is_array($item)) {
                            $subParts[] = '[' . $this->buildPkiPairs($item) . ']';
                        } else {
                            $subParts[] = (string) $item;
                        }
                    }
                    $parts[] = $key . '=[' . implode(', ', $subParts) . ']';
                } else {
                    // Associative array (nested object)
                    $parts[] = $key . '=[' . $this->buildPkiPairs($value) . ']';
                }
            } else {
                $parts[] = $key . '=' . (string) $value;
            }
        }
        return '[' . implode(', ', $parts) . ']';
    }

    private function buildPkiPairs(array $data): string
    {
        $pairs = [];
        foreach ($data as $k => $v) {
            if (is_array($v)) {
                if ($this->isIndexedArray($v)) {
                    $subParts = [];
                    foreach ($v as $item) {
                        if (is_array($item)) {
                            $subParts[] = '[' . $this->buildPkiPairs($item) . ']';
                        } else {
                            $subParts[] = (string) $item;
                        }
                    }
                    $pairs[] = $k . '=[' . implode(', ', $subParts) . ']';
                } else {
                    $pairs[] = $k . '=[' . $this->buildPkiPairs($v) . ']';
                }
            } else {
                $pairs[] = $k . '=' . (string) $v;
            }
        }
        return implode(', ', $pairs);
    }

    private function isIndexedArray(array $arr): bool
    {
        return array_keys($arr) === range(0, count($arr) - 1);
    }
}
