<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Mollie payment gateway driver.
 *
 * Uses the Mollie REST API v2 via cURL.
 * Creates a payment, redirects the customer to Mollie-hosted checkout,
 * then verifies the payment status on callback / webhook.
 *
 * @see https://docs.mollie.com/reference/v2/payments-api/create-payment
 */
class MollieDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://api.mollie.com/v2';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_key', 'label' => 'API Key', 'type' => 'password', 'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- create Mollie payment
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiKey = $this->credential('api_key');

        $payload = [
            'amount' => [
                'currency' => strtoupper($order->currency ?? 'EUR'),
                'value'    => number_format((float) $order->amount, 2, '.', ''),
            ],
            'description' => $this->paymentDescription($order),
            'redirectUrl' => $callbackUrl . (str_contains($callbackUrl, '?') ? '&' : '?') . 'order_number=' . urlencode($order->order_number),
            'webhookUrl'  => route('payment.webhook', ['gateway' => 'mollie']),
            'metadata'    => [
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ],
        ];

        $response = $this->mollieRequest('POST', '/payments', $payload, $apiKey);
        $json     = $response['json'] ?? [];

        if (isset($json['_links']['checkout']['href'])) {
            return ['redirect_url' => $json['_links']['checkout']['href']];
        }

        $errorMsg = $json['detail'] ?? $json['title'] ?? 'Failed to create Mollie payment.';
        throw new \RuntimeException("Mollie: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle callback -- check payment status
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        // Mollie redirects with no payload -- we need the payment ID from
        // the webhook or from our stored metadata.  If webhook already
        // processed it, the controller can skip this.

        // For webhook calls, Mollie sends { "id": "tr_xxx" }
        $paymentId = $payload['id'] ?? $payload['payment_id'] ?? null;

        if (! $paymentId) {
            // On redirect Mollie does not include payment data.
            // The controller should look up the order and call verify() instead.
            return PaymentResult::pending('Redirected from Mollie. Awaiting webhook confirmation.');
        }

        return $this->checkPaymentStatus($paymentId);
    }

    // -----------------------------------------------------------------
    //  Handle webhook
    // -----------------------------------------------------------------

    public function handleWebhook(array $payload): PaymentResult
    {
        $paymentId = $payload['id'] ?? null;
        if (! $paymentId) {
            return PaymentResult::failure('Missing Mollie payment ID in webhook.');
        }

        return $this->checkPaymentStatus($paymentId);
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $paymentId = $order->gateway_transaction_id;
        if (! $paymentId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        return $this->checkPaymentStatus($paymentId);
    }

    // -----------------------------------------------------------------
    //  Internal: fetch and evaluate Mollie payment status
    // -----------------------------------------------------------------

    private function checkPaymentStatus(string $paymentId): PaymentResult
    {
        $apiKey   = $this->credential('api_key');
        $response = $this->mollieRequest('GET', "/payments/{$paymentId}", [], $apiKey);
        $json     = $response['json'] ?? [];
        $status   = $json['status'] ?? '';

        if ($status === 'paid') {
            // Verify amount and currency match the order to prevent fraud
            $paidAmount   = (float) ($json['amount']['value'] ?? 0);
            $paidCurrency = strtoupper($json['amount']['currency'] ?? '');
            $orderNumber  = $json['metadata']['order_number'] ?? null;

            if ($orderNumber) {
                $order = \App\Models\Order::where('order_number', $orderNumber)->first();
                if ($order) {
                    $expectedAmount   = (float) $order->amount;
                    $expectedCurrency = strtoupper($order->currency ?? 'EUR');

                    if (abs($paidAmount - $expectedAmount) > 0.01) {
                        return PaymentResult::failure("Mollie amount mismatch: expected {$expectedAmount}, got {$paidAmount}.");
                    }
                    if ($paidCurrency && $expectedCurrency && $paidCurrency !== $expectedCurrency) {
                        return PaymentResult::failure("Mollie currency mismatch: expected {$expectedCurrency}, got {$paidCurrency}.");
                    }
                }
            }

            return PaymentResult::success(
                transactionId: $paymentId,
                message: 'Payment completed.',
                metadata: [
                    'mollie_payment_id' => $paymentId,
                    'method'            => $json['method'] ?? null,
                    'paid_at'           => $json['paidAt'] ?? null,
                ],
            );
        }

        if (in_array($status, ['pending', 'open', 'authorized'])) {
            return PaymentResult::pending("Mollie payment status: {$status}", $paymentId);
        }

        return PaymentResult::failure("Mollie payment not completed. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Mollie HTTP helper
    // -----------------------------------------------------------------

    private function mollieRequest(string $method, string $path, array $data, string $apiKey): array
    {
        $url = self::API_BASE . $path;
        $ch  = curl_init();

        $headers = [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $body   = curl_exec($ch);
        $status = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("Mollie cURL error: {$error}");
        }

        return ['status' => $status, 'body' => $body, 'json' => json_decode($body, true)];
    }
}
