<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Offline / manual payment driver.
 *
 * Displays custom instructions configured by the admin (e.g. cash,
 * cheque, money order, in-person payment). The order stays pending
 * until the admin manually marks it as paid.
 */
class OfflineDriver extends AbstractGatewayDriver
{
    public static function credentialFields(): array
    {
        return [
            ['key' => 'payment_instructions', 'label' => 'Payment Instructions', 'type' => 'textarea', 'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $instructions = $this->credential('payment_instructions', 'Please contact us for payment instructions.');

        $amount   = number_format((float) $order->amount, 2);
        $currency = strtoupper($order->currency ?? 'USD');

        $escapedInstructions = nl2br(htmlspecialchars($instructions, ENT_QUOTES, 'UTF-8'));
        $escapedOrderNumber  = htmlspecialchars($order->order_number, ENT_QUOTES, 'UTF-8');

        $html = <<<HTML
        <div class="offline-payment-details" style="max-width:500px;margin:0 auto;padding:24px;border:1px solid #e0e0e0;border-radius:8px;">
            <h3 style="margin-top:0;">Offline Payment</h3>
            <p>Amount due: <strong>{$currency} {$amount}</strong></p>
            <p>Order reference: <strong>{$escapedOrderNumber}</strong></p>
            <div style="margin-top:16px;padding:16px;background:#f9f9f9;border-radius:4px;">
                {$escapedInstructions}
            </div>
            <p style="margin-top:16px;color:#666;font-size:0.9em;">
                Your order has been placed and will be activated once the payment is confirmed by our team.
            </p>
            <form method="POST" action="{$callbackUrl}" style="margin-top:16px;">
                <input type="hidden" name="_token" value="{$this->csrfToken()}" />
                <input type="hidden" name="order_number" value="{$escapedOrderNumber}" />
                <input type="hidden" name="method" value="offline" />
                <button type="submit" class="btn btn-primary" style="padding:10px 24px;cursor:pointer;">
                    Confirm Order
                </button>
            </form>
        </div>
        HTML;

        return ['html' => $html];
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $orderNumber = $payload['order_number'] ?? '';

        return PaymentResult::pending(
            message: 'Order placed. Awaiting offline payment confirmation.',
            transactionId: 'OFFLINE_' . $orderNumber . '_' . time(),
            metadata: [
                'payment_method' => 'offline',
                'status'         => 'pending',
            ],
        );
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        return PaymentResult::pending('Offline payments do not have webhooks.');
    }

    public function verify(Order $order): PaymentResult
    {
        return PaymentResult::pending(
            'Offline payment must be verified manually by an administrator.',
            $order->gateway_transaction_id,
        );
    }

    private function csrfToken(): string
    {
        try {
            return csrf_token();
        } catch (\Exception $e) {
            return '';
        }
    }
}
