<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Paddle payment gateway driver.
 *
 * Uses the Paddle API to create a transaction/checkout link.
 *
 * @see https://developer.paddle.com/api-reference/
 */
class PaddleDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://sandbox-api.paddle.com';
    private const PROD_BASE    = 'https://api.paddle.com';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'api_key',        'label' => 'API Key',            'type' => 'password', 'required' => true],
            ['key' => 'product_id',     'label' => 'Product ID',         'type' => 'text',     'required' => true],
            ['key' => 'webhook_secret', 'label' => 'Webhook Secret Key', 'type' => 'password', 'required' => false,
             'hint' => 'Optional. Only needed if you set up webhooks in Paddle Dashboard.'],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $apiKey    = $this->credential('api_key');
        $productId = $this->credential('product_id');
        $baseUrl   = $this->isSandbox() ? self::SANDBOX_BASE : self::PROD_BASE;

        // Paddle Billing API -- create a transaction with non-catalog price
        $payload = [
            'items' => [
                [
                    'price' => [
                        'description'   => $this->paymentDescription($order),
                        'product_id'    => $productId,
                        'billing_cycle' => null, // one-time charge (not subscription)
                        'unit_price'    => [
                            'amount'        => (string) $this->amountInSmallestUnit($order),
                            'currency_code' => strtoupper($order->currency ?? 'USD'),
                        ],
                    ],
                    'quantity' => 1,
                ],
            ],
            'custom_data' => [
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ],
        ];

        $response = $this->httpRequest('POST', $baseUrl . '/transactions', $payload, [
            'Authorization: Bearer ' . $apiKey,
        ]);

        $json = $response['json'] ?? [];

        if (isset($json['data']['checkout']['url'])) {
            return ['redirect_url' => $json['data']['checkout']['url']];
        }

        $errorMsg = $json['error']['detail'] ?? 'Failed to create Paddle transaction.';
        throw new \RuntimeException("Paddle: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $transactionId = $payload['transaction_id'] ?? $payload['_ptxn'] ?? null;
        $status        = $payload['status'] ?? '';

        if (! $transactionId) {
            return PaymentResult::pending('Awaiting Paddle webhook confirmation.');
        }

        if ($status === 'completed' || $status === 'paid') {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Paddle payment completed.',
                metadata: ['paddle_transaction_id' => $transactionId],
            );
        }

        return PaymentResult::pending('Paddle payment pending confirmation.', $transactionId);
    }

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $webhookSecret = $this->credential('webhook_secret');
        if (! $webhookSecret) {
            \Illuminate\Support\Facades\Log::warning('Paddle webhook secret not configured -- skipping signature verification.');
            return true;
        }

        $signature = $headers['paddle-signature'] ?? $headers['Paddle-Signature'] ?? '';
        if (! $signature) {
            return false;
        }

        // Parse Paddle-Signature: ts=timestamp;h1=hash
        $parts = [];
        foreach (explode(';', $signature) as $item) {
            [$key, $value] = explode('=', $item, 2) + [1 => ''];
            $parts[$key] = $value;
        }

        $timestamp = $parts['ts'] ?? '';
        $hash      = $parts['h1'] ?? '';

        if (! $timestamp || ! $hash) {
            return false;
        }

        $signedPayload = $timestamp . ':' . $rawBody;
        $expected = hash_hmac('sha256', $signedPayload, $webhookSecret);

        return hash_equals($expected, $hash);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $eventType = $payload['event_type'] ?? '';
        $data      = $payload['data'] ?? [];

        if ($eventType === 'transaction.completed') {
            return PaymentResult::success(
                transactionId: $data['id'] ?? '',
                message: 'Paddle webhook confirmed payment.',
                metadata: [
                    'paddle_event'          => $eventType,
                    'paddle_transaction_id' => $data['id'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("Unhandled Paddle webhook event: {$eventType}");
    }
}
