<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * PayPal payment gateway driver.
 *
 * Uses the PayPal REST API v2 (Orders API) via cURL.
 * Creates an order, redirects to PayPal for approval, then captures on callback.
 *
 * @see https://developer.paypal.com/docs/api/orders/v2/
 */
class PayPalDriver extends AbstractGatewayDriver
{
    private const SANDBOX_BASE = 'https://api-m.sandbox.paypal.com';
    private const LIVE_BASE    = 'https://api-m.paypal.com';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'client_id',     'label' => 'Client ID',     'type' => 'text',     'required' => true],
            ['key' => 'client_secret', 'label' => 'Client Secret', 'type' => 'password', 'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- create PayPal order and redirect
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $accessToken = $this->getAccessToken();

        $payload = [
            'intent' => 'CAPTURE',
            'purchase_units' => [
                [
                    'reference_id' => $order->order_number,
                    'description'  => $this->paymentDescription($order),
                    'amount' => [
                        'currency_code' => strtoupper($order->currency ?? 'USD'),
                        'value'         => number_format((float) $order->amount, 2, '.', ''),
                    ],
                ],
            ],
            'payment_source' => [
                'paypal' => [
                    'experience_context' => [
                        'return_url'                 => $callbackUrl . '?success=1',
                        'cancel_url'                 => $callbackUrl . '?cancelled=1',
                        'brand_name'                 => config('app.name'),
                        'user_action'                => 'PAY_NOW',
                        'payment_method_preference'  => 'IMMEDIATE_PAYMENT_REQUIRED',
                    ],
                ],
            ],
        ];

        $response = $this->httpRequest('POST', $this->baseUrl() . '/v2/checkout/orders', $payload, [
            'Authorization: Bearer ' . $accessToken,
        ]);

        $json = $response['json'] ?? [];

        if (isset($json['id'])) {
            $approveUrl = collect($json['links'] ?? [])
                ->firstWhere('rel', 'approve')['href'] ?? null;

            if ($approveUrl) {
                return ['redirect_url' => $approveUrl];
            }
        }

        $errorMsg = $json['message'] ?? 'Failed to create PayPal order.';
        throw new \RuntimeException("PayPal: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle callback -- capture the approved order
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        if (isset($payload['cancelled'])) {
            return PaymentResult::failure('Payment was cancelled by the customer.');
        }

        $paypalOrderId = $payload['token'] ?? $payload['order_id'] ?? null;
        if (! $paypalOrderId) {
            return PaymentResult::failure('Missing PayPal order ID in callback.');
        }

        // Capture the order
        $accessToken = $this->getAccessToken();
        $response    = $this->httpRequest(
            'POST',
            $this->baseUrl() . "/v2/checkout/orders/{$paypalOrderId}/capture",
            [],
            ['Authorization: Bearer ' . $accessToken],
        );

        $json   = $response['json'] ?? [];
        $status = $json['status'] ?? '';

        if ($status === 'COMPLETED') {
            $capture       = $json['purchase_units'][0]['payments']['captures'][0] ?? [];
            $transactionId = $capture['id'] ?? $paypalOrderId;

            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Payment captured successfully.',
                metadata: [
                    'paypal_order_id'   => $paypalOrderId,
                    'paypal_capture_id' => $transactionId,
                    'payer_email'       => $json['payer']['email_address'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("PayPal order not completed. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Handle webhook (PAYMENT.CAPTURE.COMPLETED)
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        // PayPal webhook verification requires a webhook_id configured in PayPal dashboard.
        // Since our primary flow uses redirect callbacks (not webhooks), we accept
        // webhooks without signature verification but always verify via the Orders/Captures API.
        return true;
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $eventType = $payload['event_type'] ?? '';
        $resource  = $payload['resource'] ?? [];

        if ($eventType === 'PAYMENT.CAPTURE.COMPLETED') {
            $captureId = $resource['id'] ?? '';

            // Verify capture via API (never trust webhook payload alone)
            if ($captureId) {
                $accessToken = $this->getAccessToken();
                $response = $this->httpRequest(
                    'GET',
                    $this->baseUrl() . "/v2/payments/captures/{$captureId}",
                    [],
                    ['Authorization: Bearer ' . $accessToken],
                );

                $json   = $response['json'] ?? [];
                $status = $json['status'] ?? '';

                if ($status === 'COMPLETED') {
                    return PaymentResult::success(
                        transactionId: $captureId,
                        message: 'Webhook confirmed and verified payment capture.',
                        metadata: [
                            'paypal_event_type' => $eventType,
                            'amount'            => $json['amount']['value'] ?? null,
                            'currency'          => $json['amount']['currency_code'] ?? null,
                        ],
                    );
                }

                return PaymentResult::failure("PayPal capture status: {$status}");
            }
        }

        return PaymentResult::failure("Unhandled PayPal webhook event: {$eventType}");
    }

    // -----------------------------------------------------------------
    //  Verify payment by retrieving the order
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $transactionId = $order->gateway_transaction_id;
        if (! $transactionId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        $accessToken = $this->getAccessToken();

        // The stored transaction ID is a capture ID (from handleCallback),
        // so use the Captures API endpoint, not the Orders API.
        $response = $this->httpRequest(
            'GET',
            $this->baseUrl() . "/v2/payments/captures/{$transactionId}",
            [],
            ['Authorization: Bearer ' . $accessToken],
        );

        $json   = $response['json'] ?? [];
        $status = $json['status'] ?? '';

        if ($status === 'COMPLETED') {
            return PaymentResult::success(
                transactionId: $transactionId,
                message: 'Payment verified successfully.',
            );
        }

        return PaymentResult::failure("Payment not verified. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Helpers
    // -----------------------------------------------------------------

    private function baseUrl(): string
    {
        return $this->isSandbox() ? self::SANDBOX_BASE : self::LIVE_BASE;
    }

    private function getAccessToken(): string
    {
        $clientId     = $this->credential('client_id');
        $clientSecret = $this->credential('client_secret');

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => $this->baseUrl() . '/v1/oauth2/token',
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => 'grant_type=client_credentials',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 15,
            CURLOPT_USERPWD        => $clientId . ':' . $clientSecret,
            CURLOPT_HTTPHEADER     => ['Accept: application/json', 'Accept-Language: en_US'],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        $body  = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("PayPal token request failed: {$error}");
        }

        $json = json_decode($body, true);

        if (isset($json['access_token'])) {
            return $json['access_token'];
        }

        throw new \RuntimeException('PayPal: Unable to obtain access token.');
    }
}
