<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * PayTR payment gateway driver (Turkey).
 *
 * Uses the PayTR iFrame API to create a payment token and
 * render the PayTR iframe for card input.
 *
 * @see https://dev.paytr.com/en/iframe-api
 */
class PayTRDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://www.paytr.com/odeme/api/get-token';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'merchant_id',   'label' => 'Merchant ID',   'type' => 'text',     'required' => true],
            ['key' => 'merchant_key',  'label' => 'Merchant Key',  'type' => 'password', 'required' => true],
            ['key' => 'merchant_salt', 'label' => 'Merchant Salt', 'type' => 'password', 'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $merchantId   = $this->credential('merchant_id');
        $merchantKey  = $this->credential('merchant_key');
        $merchantSalt = $this->credential('merchant_salt');

        $merchantOid  = 'PAYTR_' . $order->order_number . '_' . time();
        $email        = $order->user->email ?? throw new \RuntimeException('PayTR requires a customer email address.');
        $amount       = (int) round($order->amount * 100); // kuruş
        $currency     = strtoupper($order->currency ?? 'TL');
        $testMode     = $this->isSandbox() ? '1' : '0';
        $noInstallment = '1';
        $maxInstallment = '0';
        $userName     = $order->user->name ?? 'Customer';
        $userAddress  = 'N/A';
        $userPhone    = '0000000000';
        $userIp       = request()->ip() ?? '127.0.0.1';

        // Basket (JSON array of arrays: [name, price_kuruş, quantity])
        $basket = base64_encode(json_encode([
            [$this->paymentDescription($order), $amount, 1],
        ]));

        // Generate token hash
        $hashStr = $merchantId . $userIp . $merchantOid . $email . $amount . $basket
            . $noInstallment . $maxInstallment . $currency . $testMode;
        $paytrToken = base64_encode(
            hash_hmac('sha256', $hashStr . $merchantSalt, $merchantKey, true)
        );

        $payload = [
            'merchant_id'      => $merchantId,
            'user_ip'          => $userIp,
            'merchant_oid'     => $merchantOid,
            'email'            => $email,
            'payment_amount'   => $amount,
            'paytr_token'      => $paytrToken,
            'user_basket'      => $basket,
            'debug_on'         => $this->isSandbox() ? '1' : '0',
            'no_installment'   => $noInstallment,
            'max_installment'  => $maxInstallment,
            'user_name'        => $userName,
            'user_address'     => $userAddress,
            'user_phone'       => $userPhone,
            'merchant_ok_url'  => $callbackUrl . '?status=success',
            'merchant_fail_url' => $callbackUrl . '?status=failed',
            'timeout_limit'    => '30',
            'currency'         => $currency,
            'test_mode'        => $testMode,
        ];

        $response = $this->httpFormPost(self::API_BASE, $payload);
        $json     = $response['json'] ?? [];

        if (($json['status'] ?? '') === 'success' && isset($json['token'])) {
            $eIframeSrc = htmlspecialchars('https://www.paytr.com/odeme/guvenli/' . $json['token'], ENT_QUOTES, 'UTF-8');

            $html = <<<HTML
            <iframe src="{$eIframeSrc}" id="paytriframe" frameborder="0" scrolling="no"
                    style="width:100%;min-height:600px;"></iframe>
            HTML;

            return ['html' => $html];
        }

        $errorMsg = $json['reason'] ?? 'Failed to generate PayTR token.';
        throw new \RuntimeException("PayTR: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $status      = $payload['status'] ?? '';
        $merchantOid = $payload['merchant_oid'] ?? null;
        $totalAmount = $payload['total_amount'] ?? '';
        $hash        = $payload['hash'] ?? '';

        if ($status === 'failed') {
            return PaymentResult::failure('PayTR payment failed.');
        }

        // Verify hash
        $merchantKey  = $this->credential('merchant_key');
        $merchantSalt = $this->credential('merchant_salt');

        $hashStr      = $merchantOid . $merchantSalt . $status . $totalAmount;
        $expectedHash = base64_encode(hash_hmac('sha256', $hashStr, $merchantKey, true));

        if (! $hash || ! hash_equals($expectedHash, $hash)) {
            return PaymentResult::failure('Invalid or missing PayTR callback hash.');
        }

        if ($status === 'success') {
            return PaymentResult::success(
                transactionId: $merchantOid ?? '',
                message: 'Payment successful.',
                metadata: [
                    'paytr_merchant_oid' => $merchantOid,
                    'paytr_total_amount' => $totalAmount,
                ],
            );
        }

        return PaymentResult::failure("PayTR status: {$status}");
    }
}
