<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * PayU payment gateway driver (India / LatAm).
 *
 * Uses the PayU Money / PayU Biz API. Creates a hash-based form
 * and redirects the customer to PayU for payment.
 *
 * @see https://devguide.payu.in/docs/payu-checkout-integration/
 */
class PayUDriver extends AbstractGatewayDriver
{
    private const SANDBOX_URL = 'https://sandboxsecure.payu.in/_payment';
    private const PROD_URL    = 'https://secure.payu.in/_payment';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'merchant_key',  'label' => 'Merchant Key',  'type' => 'text',     'required' => true],
            ['key' => 'merchant_salt', 'label' => 'Merchant Salt', 'type' => 'password', 'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $merchantKey  = $this->credential('merchant_key');
        $merchantSalt = $this->credential('merchant_salt');

        $txnId       = 'PAYU_' . $order->order_number . '_' . time();
        $amount      = number_format((float) $order->amount, 2, '.', '');
        $productInfo = $this->paymentDescription($order);
        $firstName   = $order->user->name ?? 'Customer';
        $email       = $order->user->email ?? throw new \RuntimeException('PayU requires a customer email address.');
        $phone       = '9999999999';
        $udf1        = $order->order_number;
        $udf2        = (string) $order->id;

        // Generate hash: sha512(key|txnid|amount|productinfo|firstname|email|udf1|udf2|udf3|udf4|udf5||||||SALT)
        $hashString = "{$merchantKey}|{$txnId}|{$amount}|{$productInfo}|{$firstName}|{$email}|{$udf1}|{$udf2}||||||||{$merchantSalt}";
        $hash       = strtolower(hash('sha512', $hashString));

        $paymentUrl = $this->isSandbox() ? self::SANDBOX_URL : self::PROD_URL;

        $e = fn($v) => htmlspecialchars((string) $v, ENT_QUOTES, 'UTF-8');

        $html = <<<HTML
        <form id="payu-form" method="POST" action="{$e($paymentUrl)}">
            <input type="hidden" name="key" value="{$e($merchantKey)}" />
            <input type="hidden" name="txnid" value="{$e($txnId)}" />
            <input type="hidden" name="amount" value="{$e($amount)}" />
            <input type="hidden" name="productinfo" value="{$e($productInfo)}" />
            <input type="hidden" name="firstname" value="{$e($firstName)}" />
            <input type="hidden" name="email" value="{$e($email)}" />
            <input type="hidden" name="phone" value="{$e($phone)}" />
            <input type="hidden" name="surl" value="{$e($callbackUrl)}" />
            <input type="hidden" name="furl" value="{$e($callbackUrl . '?status=failed')}" />
            <input type="hidden" name="hash" value="{$e($hash)}" />
            <input type="hidden" name="udf1" value="{$e($order->order_number)}" />
            <input type="hidden" name="udf2" value="{$e($order->id)}" />
        </form>
        <script>document.getElementById('payu-form').submit();</script>
        HTML;

        return ['html' => $html];
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $status       = $payload['status'] ?? '';
        $txnId        = $payload['txnid'] ?? null;
        $mihpayid     = $payload['mihpayid'] ?? null;
        $responseHash = $payload['hash'] ?? '';

        if (! $txnId) {
            return PaymentResult::failure('Missing PayU transaction ID.');
        }

        // Verify reverse hash (mandatory)
        $merchantKey  = $this->credential('merchant_key');
        $merchantSalt = $this->credential('merchant_salt');
        $amount       = $payload['amount'] ?? '';
        $productInfo  = $payload['productinfo'] ?? '';
        $firstName    = $payload['firstname'] ?? '';
        $email        = $payload['email'] ?? '';
        $udf1         = $payload['udf1'] ?? '';
        $udf2         = $payload['udf2'] ?? '';
        $udf3         = $payload['udf3'] ?? '';
        $udf4         = $payload['udf4'] ?? '';
        $udf5         = $payload['udf5'] ?? '';

        // Reverse hash: sha512(SALT|status||||||udf5|udf4|udf3|udf2|udf1|email|firstname|productinfo|amount|txnid|key)
        $reverseHashString = "{$merchantSalt}|{$status}||||||{$udf5}|{$udf4}|{$udf3}|{$udf2}|{$udf1}|{$email}|{$firstName}|{$productInfo}|{$amount}|{$txnId}|{$merchantKey}";
        $expectedHash      = strtolower(hash('sha512', $reverseHashString));

        if (! $responseHash) {
            return PaymentResult::failure('Missing PayU response hash.');
        }

        if (! hash_equals($expectedHash, $responseHash)) {
            return PaymentResult::failure('Invalid PayU response hash.');
        }

        if ($status === 'success') {
            return PaymentResult::success(
                transactionId: $mihpayid ?? $txnId,
                message: 'Payment successful.',
                metadata: [
                    'payu_txnid'    => $txnId,
                    'payu_mihpayid' => $mihpayid,
                    'payu_mode'     => $payload['mode'] ?? null,
                ],
            );
        }

        $errorMsg = $payload['error_Message'] ?? "Payment status: {$status}";
        return PaymentResult::failure("PayU: {$errorMsg}");
    }
}
