<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Paystack payment gateway driver.
 *
 * Uses the Paystack API via cURL.
 * Initializes a transaction, redirects to Paystack-hosted checkout,
 * then verifies the transaction reference on callback.
 *
 * @see https://paystack.com/docs/api/transaction/
 */
class PaystackDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://api.paystack.co';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'public_key', 'label' => 'Public Key', 'type' => 'text',     'required' => true],
            ['key' => 'secret_key', 'label' => 'Secret Key', 'type' => 'password', 'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- initialize Paystack transaction
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $secretKey = $this->credential('secret_key');

        $payload = [
            'email'        => $order->user->email ?? throw new \RuntimeException('Paystack requires a customer email address.'),
            'amount'       => $this->amountInSmallestUnit($order), // kobo
            'currency'     => strtoupper($order->currency ?? 'NGN'),
            'reference'    => $order->order_number . '_' . time(),
            'callback_url' => $callbackUrl,
            'metadata'     => [
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ],
        ];

        $response = $this->paystackRequest('POST', '/transaction/initialize', $payload, $secretKey);
        $json     = $response['json'] ?? [];

        if (($json['status'] ?? false) === true && isset($json['data']['authorization_url'])) {
            return ['redirect_url' => $json['data']['authorization_url']];
        }

        $errorMsg = $json['message'] ?? 'Failed to initialize Paystack transaction.';
        throw new \RuntimeException("Paystack: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle callback -- verify transaction
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        $reference = $payload['reference'] ?? $payload['trxref'] ?? null;

        if (! $reference) {
            return PaymentResult::failure('Missing Paystack transaction reference.');
        }

        $secretKey = $this->credential('secret_key');
        $response  = $this->paystackRequest('GET', "/transaction/verify/" . urlencode($reference), [], $secretKey);
        $json      = $response['json'] ?? [];

        if (($json['status'] ?? false) === true && ($json['data']['status'] ?? '') === 'success') {
            // Verify amount and currency match the order to prevent fraud
            // Paystack returns amount in kobo (smallest unit)
            $paidAmountKobo = (int) ($json['data']['amount'] ?? 0);
            $paidCurrency   = strtoupper($json['data']['currency'] ?? '');
            $orderNumber    = $json['data']['metadata']['order_number'] ?? null;

            if ($orderNumber) {
                $order = \App\Models\Order::where('order_number', $orderNumber)->first();
                if ($order) {
                    $expectedAmountKobo = (int) round((float) $order->amount * 100);
                    $expectedCurrency   = strtoupper($order->currency ?? 'NGN');

                    if ($paidAmountKobo !== $expectedAmountKobo) {
                        return PaymentResult::failure("Paystack amount mismatch: expected {$expectedAmountKobo}, got {$paidAmountKobo} (kobo).");
                    }
                    if ($paidCurrency && $expectedCurrency && $paidCurrency !== $expectedCurrency) {
                        return PaymentResult::failure("Paystack currency mismatch: expected {$expectedCurrency}, got {$paidCurrency}.");
                    }
                }
            }

            return PaymentResult::success(
                transactionId: (string) ($json['data']['id'] ?? $reference),
                message: 'Payment successful.',
                metadata: [
                    'paystack_reference'  => $reference,
                    'paystack_channel'    => $json['data']['channel'] ?? null,
                    'paystack_ip'         => $json['data']['ip_address'] ?? null,
                    'gateway_response'    => $json['data']['gateway_response'] ?? null,
                ],
            );
        }

        $status = $json['data']['status'] ?? 'unknown';
        return PaymentResult::failure("Paystack transaction not successful. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Handle webhook (charge.success)
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $secretKey = $this->credential('secret_key');
        if (! $secretKey) {
            \Illuminate\Support\Facades\Log::warning('Paystack secret key not configured -- skipping webhook signature verification.');
            return true;
        }

        $signature = $headers['x-paystack-signature'] ?? $headers['X-Paystack-Signature'] ?? '';
        if (! $signature) {
            return false;
        }

        $expected = hash_hmac('sha512', $rawBody, $secretKey);

        return hash_equals($expected, $signature);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $event = $payload['event'] ?? '';
        $data  = $payload['data'] ?? [];

        if ($event === 'charge.success' && ($data['status'] ?? '') === 'success') {
            return PaymentResult::success(
                transactionId: (string) ($data['id'] ?? $data['reference'] ?? ''),
                message: 'Webhook confirmed successful charge.',
                metadata: [
                    'paystack_event'     => $event,
                    'paystack_reference' => $data['reference'] ?? null,
                    'channel'            => $data['channel'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("Unhandled Paystack webhook event: {$event}");
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $reference = $order->gateway_transaction_id;
        if (! $reference) {
            return PaymentResult::failure('No transaction reference available for verification.');
        }

        return $this->handleCallback(['reference' => $reference]);
    }

    // -----------------------------------------------------------------
    //  Paystack HTTP helper
    // -----------------------------------------------------------------

    private function paystackRequest(string $method, string $path, array $data, string $secretKey): array
    {
        $url = self::API_BASE . $path;
        $ch  = curl_init();

        $headers = [
            'Authorization: Bearer ' . $secretKey,
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $body   = curl_exec($ch);
        $status = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("Paystack cURL error: {$error}");
        }

        return ['status' => $status, 'body' => $body, 'json' => json_decode($body, true)];
    }
}
