<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Paytm payment gateway driver.
 *
 * Uses the Paytm Business API via cURL.
 * Creates a transaction token, renders a redirect form to Paytm checkout,
 * then verifies the transaction status on callback.
 *
 * @see https://business.paytm.com/docs/api/initiate-transaction-api/
 */
class PaytmDriver extends AbstractGatewayDriver
{
    private const STAGING_BASE = 'https://securegw-stage.paytm.in';
    private const PROD_BASE    = 'https://securegw.paytm.in';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'merchant_id',  'label' => 'Merchant ID',  'type' => 'text',     'required' => true],
            ['key' => 'merchant_key', 'label' => 'Merchant Key', 'type' => 'password', 'required' => true],
            ['key' => 'website',      'label' => 'Website',      'type' => 'text',     'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- create Paytm transaction token and redirect
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $merchantId  = $this->credential('merchant_id');
        $merchantKey = $this->credential('merchant_key');
        $website     = $this->credential('website');

        $orderId = 'PAYTM_' . $order->order_number . '_' . time();

        $body = [
            'body' => [
                'requestType'   => 'Payment',
                'mid'           => $merchantId,
                'websiteName'   => $website,
                'orderId'       => $orderId,
                'callbackUrl'   => $callbackUrl,
                'txnAmount'     => [
                    'value'    => number_format((float) $order->amount, 2, '.', ''),
                    'currency' => strtoupper($order->currency ?? 'INR'),
                ],
                'userInfo'      => [
                    'custId' => (string) ($order->user_id ?? 'GUEST'),
                ],
            ],
        ];

        // Generate checksum
        $bodyJson = json_encode($body['body']);
        $checksum = $this->generateChecksum($bodyJson, $merchantKey);

        $head = [
            'signature' => $checksum,
        ];

        $payload = ['body' => $body['body'], 'head' => $head];

        $url      = $this->baseUrl() . "/theia/api/v1/initiateTransaction?mid={$merchantId}&orderId={$orderId}";
        $response = $this->httpRequest('POST', $url, $payload);
        $json     = $response['json'] ?? [];

        $resultCode = $json['body']['resultInfo']['resultCode'] ?? '';
        $txnToken   = $json['body']['txnToken'] ?? null;

        if ($resultCode === 'S' && $txnToken) {
            // Build redirect URL to Paytm checkout page
            $redirectUrl = $this->baseUrl() . "/theia/api/v1/showPaymentPage?mid={$merchantId}&orderId={$orderId}&txnToken={$txnToken}";

            return ['redirect_url' => $redirectUrl];
        }

        $errorMsg = $json['body']['resultInfo']['resultMsg'] ?? 'Failed to initiate Paytm transaction.';
        throw new \RuntimeException("Paytm: {$errorMsg}");
    }

    // -----------------------------------------------------------------
    //  Handle callback -- verify transaction status
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        $orderId     = $payload['ORDERID'] ?? null;
        $bankTxnId   = $payload['BANKTXNID'] ?? null;
        $status      = $payload['STATUS'] ?? '';
        $checksumStr = $payload['CHECKSUMHASH'] ?? '';

        if (! $orderId) {
            return PaymentResult::failure('Missing Paytm order ID in callback.');
        }

        // Verify checksum
        $merchantKey = $this->credential('merchant_key');
        $paramsToVerify = $payload;
        unset($paramsToVerify['CHECKSUMHASH']);

        if (! $checksumStr || ! $this->verifyChecksum($paramsToVerify, $merchantKey, $checksumStr)) {
            return PaymentResult::failure('Invalid or missing Paytm checksum.');
        }

        // Additionally, verify with Paytm's Transaction Status API
        $result = $this->queryTransactionStatus($orderId);

        if ($result !== null) {
            return $result;
        }

        // Fall back to callback status
        if ($status === 'TXN_SUCCESS') {
            return PaymentResult::success(
                transactionId: $bankTxnId ?? $orderId,
                message: 'Payment successful.',
                metadata: [
                    'paytm_order_id' => $orderId,
                    'bank_txn_id'    => $bankTxnId,
                    'txn_id'         => $payload['TXNID'] ?? null,
                ],
            );
        }

        $respMsg = $payload['RESPMSG'] ?? "Transaction status: {$status}";
        return PaymentResult::failure("Paytm: {$respMsg}");
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $orderId = $order->gateway_transaction_id;
        if (! $orderId) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        $result = $this->queryTransactionStatus($orderId);
        return $result ?? PaymentResult::failure('Unable to verify Paytm transaction.');
    }

    // -----------------------------------------------------------------
    //  Query Paytm Transaction Status API
    // -----------------------------------------------------------------

    private function queryTransactionStatus(string $orderId): ?PaymentResult
    {
        $merchantId  = $this->credential('merchant_id');
        $merchantKey = $this->credential('merchant_key');

        $body     = ['mid' => $merchantId, 'orderId' => $orderId];
        $bodyJson = json_encode($body);
        $checksum = $this->generateChecksum($bodyJson, $merchantKey);

        $payload = [
            'body' => $body,
            'head' => ['signature' => $checksum],
        ];

        $url      = $this->baseUrl() . "/v3/order/status";
        $response = $this->httpRequest('POST', $url, $payload);
        $json     = $response['json'] ?? [];

        $resultCode = $json['body']['resultInfo']['resultCode'] ?? '';
        $resultStatus = $json['body']['resultInfo']['resultStatus'] ?? '';

        if ($resultCode === '01' || $resultStatus === 'TXN_SUCCESS') {
            return PaymentResult::success(
                transactionId: $json['body']['txnId'] ?? $orderId,
                message: 'Payment verified successfully.',
                metadata: [
                    'paytm_order_id' => $orderId,
                    'bank_txn_id'    => $json['body']['bankTxnId'] ?? null,
                    'txn_date'       => $json['body']['txnDate'] ?? null,
                ],
            );
        }

        if ($resultStatus === 'PENDING') {
            return PaymentResult::pending('Paytm transaction is pending.', $orderId);
        }

        return null;
    }

    // -----------------------------------------------------------------
    //  Helpers
    // -----------------------------------------------------------------

    private function baseUrl(): string
    {
        return $this->isSandbox() ? self::STAGING_BASE : self::PROD_BASE;
    }

    /**
     * Generate a Paytm-compatible checksum.
     *
     * Algorithm (matches Paytm PHP SDK):
     * 1. Generate 4-byte random salt
     * 2. SHA-256 hash of (body|salt) -- plain hash, NOT HMAC
     * 3. Concatenate hash + salt
     * 4. AES-128-CBC encrypt with merchant key and IV @@@@&&&&####$$$$
     * 5. openssl_encrypt returns base64 by default
     */
    private function generateChecksum(string $body, string $key): string
    {
        $salt       = $this->generateRandomString(4);
        $hash       = hash('sha256', $body . '|' . $salt);
        $hashString = $hash . $salt;
        $iv         = '@@@@&&&&####$$$$';

        $encrypted = openssl_encrypt($hashString, 'AES-128-CBC', $key, 0, $iv);

        if ($encrypted === false) {
            throw new \RuntimeException('Paytm: Failed to generate checksum (AES encryption failed).');
        }

        return $encrypted;
    }

    /**
     * Verify a Paytm checksum.
     *
     * Algorithm (matches Paytm PHP SDK):
     * 1. AES-128-CBC decrypt checksum with merchant key and IV @@@@&&&&####$$$$
     * 2. Extract salt (last 4 chars) and hash (first 64 hex chars)
     * 3. Recompute SHA-256 of pipe-separated sorted params + salt
     * 4. Compare hashes
     */
    private function verifyChecksum(array $params, string $key, string $checksum): bool
    {
        $iv        = '@@@@&&&&####$$$$';
        $decrypted = openssl_decrypt($checksum, 'AES-128-CBC', $key, 0, $iv);

        if ($decrypted === false || strlen($decrypted) < 68) {
            return false;
        }

        // Extract hash (first 64 hex chars) and salt (last 4 chars)
        $providedHash = substr($decrypted, 0, 64);
        $salt         = substr($decrypted, 64);

        // Sort parameters and build pipe-separated string (same as Paytm SDK)
        ksort($params);
        $str          = implode('|', $params);
        $expectedHash = hash('sha256', $str . '|' . $salt);

        return hash_equals($expectedHash, $providedHash);
    }

    private function generateRandomString(int $length): string
    {
        return substr(bin2hex(random_bytes($length)), 0, $length);
    }
}
