<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * Tap Payments gateway driver (Middle East).
 *
 * Uses the Tap Charges API to create a payment charge
 * and redirect the customer.
 *
 * @see https://developers.tap.company/reference/create-a-charge
 */
class TapDriver extends AbstractGatewayDriver
{
    private const API_BASE = 'https://api.tap.company/v2';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'secret_key',      'label' => 'Secret Key',      'type' => 'password', 'required' => true],
            ['key' => 'publishable_key', 'label' => 'Publishable Key', 'type' => 'text',     'required' => true],
        ];
    }

    public function initiate(Order $order, string $callbackUrl): array
    {
        $secretKey = $this->credential('secret_key');

        $payload = [
            'amount'   => (float) $order->amount,
            'currency' => strtoupper($order->currency ?? 'KWD'),
            'customer' => [
                'first_name' => $order->user->name ?? 'Customer',
                'email'      => $order->user->email ?? throw new \RuntimeException('Tap requires a customer email address.'),
            ],
            'source'   => ['id' => 'src_all'],
            'redirect' => ['url' => $callbackUrl],
            'post'     => ['url' => route('payment.webhook', ['gateway' => 'tap'])],
            'reference' => [
                'transaction' => $order->order_number,
                'order'       => $order->order_number,
            ],
            'description' => $this->paymentDescription($order),
            'metadata'    => [
                'order_id'     => $order->id,
                'order_number' => $order->order_number,
            ],
        ];

        $response = $this->httpRequest('POST', self::API_BASE . '/charges', $payload, [
            'Authorization: Bearer ' . $secretKey,
        ]);

        $json = $response['json'] ?? [];

        if (isset($json['transaction']['url'])) {
            return ['redirect_url' => $json['transaction']['url']];
        }

        $errorMsg = $json['errors'][0]['description'] ?? 'Failed to create Tap charge.';
        throw new \RuntimeException("Tap: {$errorMsg}");
    }

    public function handleCallback(array $payload): PaymentResult
    {
        $tapId = $payload['tap_id'] ?? $payload['id'] ?? null;

        if (! $tapId) {
            return PaymentResult::failure('Missing Tap charge ID.');
        }

        // Retrieve the charge to verify status
        $secretKey = $this->credential('secret_key');
        $response  = $this->httpRequest('GET', self::API_BASE . "/charges/{$tapId}", [], [
            'Authorization: Bearer ' . $secretKey,
        ]);

        $json   = $response['json'] ?? [];
        $status = $json['status'] ?? '';

        if ($status === 'CAPTURED') {
            return PaymentResult::success(
                transactionId: $json['id'] ?? $tapId,
                message: 'Payment captured.',
                metadata: [
                    'tap_charge_id' => $json['id'] ?? null,
                    'tap_receipt'   => $json['receipt']['id'] ?? null,
                ],
            );
        }

        return PaymentResult::failure("Tap charge status: {$status}");
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $chargeId = $payload['id'] ?? '';

        if (! $chargeId) {
            return PaymentResult::failure('Missing Tap charge ID in webhook.');
        }

        // Verify charge status by retrieving it from Tap API (never trust webhook payload alone)
        $secretKey = $this->credential('secret_key');
        $response  = $this->httpRequest('GET', self::API_BASE . "/charges/{$chargeId}", [], [
            'Authorization: Bearer ' . $secretKey,
        ]);

        $json   = $response['json'] ?? [];
        $status = $json['status'] ?? '';

        if ($status === 'CAPTURED') {
            return PaymentResult::success(
                transactionId: $json['id'] ?? $chargeId,
                message: 'Tap webhook confirmed payment.',
                metadata: [
                    'tap_charge_id' => $json['id'] ?? null,
                    'tap_status'    => $status,
                ],
            );
        }

        return PaymentResult::failure("Tap webhook status: {$status}");
    }
}
