<?php

namespace App\Services\Payment\Drivers;

use App\Models\Order;
use App\Services\Payment\AbstractGatewayDriver;
use App\Services\Payment\PaymentResult;

/**
 * 2Checkout (Verifone) payment gateway driver.
 *
 * Uses the 2Checkout API via cURL.
 * Creates a hosted checkout session / buy-link and redirects the
 * customer to the 2Checkout-hosted payment page.
 *
 * @see https://verifone.cloud/docs/2checkout/API-Integration/
 */
class TwoCheckoutDriver extends AbstractGatewayDriver
{
    private const API_BASE     = 'https://api.2checkout.com/rest/6.0';
    private const CHECKOUT_URL = 'https://secure.2checkout.com/checkout/buy';

    public static function credentialFields(): array
    {
        return [
            ['key' => 'merchant_code',   'label' => 'Merchant Code',    'type' => 'text',     'required' => true],
            ['key' => 'secret_key',      'label' => 'Secret Key',       'type' => 'password', 'required' => true],
            ['key' => 'buy_link_secret', 'label' => 'Buy Link Secret',  'type' => 'password', 'required' => true],
        ];
    }

    // -----------------------------------------------------------------
    //  Initiate -- create 2Checkout buy link
    // -----------------------------------------------------------------

    public function initiate(Order $order, string $callbackUrl): array
    {
        $merchantCode  = $this->credential('merchant_code');
        $buyLinkSecret = $this->credential('buy_link_secret');

        $currency    = strtoupper($order->currency ?? 'USD');
        $amount      = number_format((float) $order->amount, 2, '.', '');
        $orderNumber = $order->order_number;
        $now         = gmdate('Y-m-d H:i:s');

        // Build the HMAC signature for the buy link
        // Parameters must be in specific order for signature generation
        $signParams = [
            strlen($merchantCode) . $merchantCode,
            strlen($now) . $now,
            strlen($amount) . $amount,
            strlen($currency) . $currency,
            strlen($orderNumber) . $orderNumber,
        ];

        $signString = implode('', $signParams);
        $signature  = hash_hmac('sha256', $signString, $buyLinkSecret);

        // Build the buy link URL with parameters
        $params = [
            'merchant'        => $merchantCode,
            'dynamic'         => 1,
            'order-ext-ref'   => $orderNumber,
            'item-ext-ref'    => 'ORDER_' . $orderNumber,
            'prod'            => $this->paymentDescription($order),
            'price'           => $amount,
            'qty'             => 1,
            'type'            => 'PRODUCT',
            'currency'        => $currency,
            'return-url'      => $callbackUrl,
            'return-type'     => 'redirect',
            'expiration'      => gmdate('Y-m-d H:i:s', time() + 3600),
            'order-date'      => $now,
            'signature'       => $signature,
        ];

        $redirectUrl = self::CHECKOUT_URL . '?' . http_build_query($params);

        return ['redirect_url' => $redirectUrl];
    }

    // -----------------------------------------------------------------
    //  Handle callback -- verify IPN / return data
    // -----------------------------------------------------------------

    public function handleCallback(array $payload): PaymentResult
    {
        $refNo   = $payload['refno'] ?? $payload['REFNO'] ?? null;
        $orderno = $payload['order-ext-ref'] ?? $payload['REFNOEXT'] ?? null;

        if (! $refNo) {
            return PaymentResult::failure('Missing 2Checkout reference number.');
        }

        // Verify the order via API
        $secretKey    = $this->credential('secret_key');
        $merchantCode = $this->credential('merchant_code');

        $response = $this->twoCheckoutRequest('GET', "/orders/{$refNo}/", [], $merchantCode, $secretKey);
        $json     = $response['json'] ?? [];
        $status   = $json['Status'] ?? $json['status'] ?? '';

        if (in_array($status, ['COMPLETE', 'AUTHRECEIVED'])) {
            return PaymentResult::success(
                transactionId: $refNo,
                message: 'Payment completed.',
                metadata: [
                    'twocheckout_ref'     => $refNo,
                    'twocheckout_status'  => $status,
                    'external_ref'        => $orderno,
                ],
            );
        }

        if ($status === 'PENDING') {
            return PaymentResult::pending('2Checkout payment is pending.', $refNo);
        }

        return PaymentResult::failure("2Checkout order not completed. Status: {$status}");
    }

    // -----------------------------------------------------------------
    //  Handle IPN webhook
    // -----------------------------------------------------------------

    public function verifyWebhookSignature(string $rawBody, array $headers): bool
    {
        $secretKey = $this->credential('secret_key');
        if (! $secretKey) {
            \Illuminate\Support\Facades\Log::warning('2Checkout secret key not configured -- skipping webhook signature verification.');
            return true;
        }

        // Parse IPN form data
        parse_str($rawBody, $params);

        $receivedHash = $params['HASH'] ?? '';
        if (! $receivedHash) {
            return true; // Not all 2CO webhooks include HASH
        }

        // Build IPN hash: HMAC-MD5 of length-prefixed sorted params
        $ipnParams = $params;
        unset($ipnParams['HASH']);

        $hashString = '';
        foreach ($ipnParams as $value) {
            $val = (string) $value;
            $hashString .= strlen($val) . $val;
        }

        $expected = hash_hmac('md5', $hashString, $secretKey);

        return hash_equals($expected, $receivedHash);
    }

    public function handleWebhook(array $payload): PaymentResult
    {
        $ipnType = $payload['ORDERSTATUS'] ?? $payload['IPN_PID'] ?? '';
        $refNo   = $payload['REFNO'] ?? '';

        if (in_array($ipnType, ['COMPLETE', 'AUTHRECEIVED'])) {
            return PaymentResult::success(
                transactionId: $refNo,
                message: '2Checkout IPN confirmed payment.',
                metadata: [
                    'ipn_status'      => $ipnType,
                    'twocheckout_ref' => $refNo,
                ],
            );
        }

        return PaymentResult::failure("Unhandled 2Checkout IPN status: {$ipnType}");
    }

    // -----------------------------------------------------------------
    //  Verify payment
    // -----------------------------------------------------------------

    public function verify(Order $order): PaymentResult
    {
        $refNo = $order->gateway_transaction_id;
        if (! $refNo) {
            return PaymentResult::failure('No transaction ID available for verification.');
        }

        return $this->handleCallback(['refno' => $refNo]);
    }

    // -----------------------------------------------------------------
    //  2Checkout HTTP helper
    // -----------------------------------------------------------------

    private function twoCheckoutRequest(string $method, string $path, array $data, string $merchantCode, string $secretKey): array
    {
        $url  = self::API_BASE . $path;
        $now  = gmdate('Y-m-d H:i:s');
        $hash = hash_hmac('md5', strlen($merchantCode) . $merchantCode . strlen($now) . $now, $secretKey);

        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
            'X-Avangate-Authentication: code="' . $merchantCode . '" date="' . $now . '" hash="' . $hash . '"',
        ];

        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $body   = curl_exec($ch);
        $status = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new \RuntimeException("2Checkout cURL error: {$error}");
        }

        return ['status' => $status, 'body' => $body, 'json' => json_decode($body, true)];
    }
}
