<?php

namespace App\Services;

use App\Exceptions\PlanLimitReachedException;
use App\Models\Event;
use App\Models\EventMedia;
use App\Models\Plan;
use App\Models\User;
use Illuminate\Support\Facades\Schema;

class PlanLimitService
{
    private static ?bool $plansTableExists = null;
    private array $resolvedPlans = [];

    private function plansTableExists(): bool
    {
        if (self::$plansTableExists === null) {
            try {
                self::$plansTableExists = Schema::hasTable('plans');
            } catch (\Exception $e) {
                self::$plansTableExists = false;
            }
        }
        return self::$plansTableExists;
    }

    public function assertCanCreateEvent(User $user, bool $isActive): void
    {
        if (! $this->shouldEnforce($user)) {
            return;
        }

        $plan = $this->resolvePlan($user);
        if (! $plan || $plan->events_limit === null) {
            return;
        }

        $activeEvents = Event::query()
            ->where('created_by', $user->id)
            ->where('is_active', true)
            ->count();

        $projected = $activeEvents + ($isActive ? 1 : 0);
        if ($projected > $plan->events_limit) {
            throw new PlanLimitReachedException('events', $activeEvents, (int) $plan->events_limit);
        }
    }

    public function assertCanUpload(Event $event, string $fileType, int $bytesToAdd, int $additionalCount = 1): void
    {
        $owner = $this->resolveOwner($event);
        if (! $owner || ! $this->shouldEnforce($owner)) {
            return;
        }

        $plan = $this->resolvePlan($owner);
        if (! $plan) {
            return;
        }

        if ($fileType === 'image' && $plan->images_limit !== null) {
            $existing = EventMedia::query()
                ->where('event_media.file_type', 'image')
                ->join('events', 'events.id', '=', 'event_media.event_id')
                ->where('events.created_by', $owner->id)
                ->count();

            if ($existing + max(1, $additionalCount) > $plan->images_limit) {
                throw new PlanLimitReachedException('images', $existing, (int) $plan->images_limit);
            }
        }

        // Check videos limit
        if ($fileType === 'video' && $plan->videos_limit !== null) {
            $existing = EventMedia::query()
                ->where('event_media.file_type', 'video')
                ->join('events', 'events.id', '=', 'event_media.event_id')
                ->where('events.created_by', $owner->id)
                ->count();

            if ($existing + max(1, $additionalCount) > $plan->videos_limit) {
                throw new PlanLimitReachedException('videos', $existing, (int) $plan->videos_limit);
            }
        }

        // Check max file size
        if ($plan->max_file_size_mb !== null && $plan->max_file_size_mb > 0) {
            $maxBytes = $plan->max_file_size_mb * 1024 * 1024;
            if ($bytesToAdd > $maxBytes) {
                throw new PlanLimitReachedException('file_size', $bytesToAdd, $maxBytes);
            }
        }

        $storageLimitBytes = $this->planStorageLimitBytes($plan);
        if ($storageLimitBytes > 0) {
            $usedBytes = (int) EventMedia::query()
                ->join('events', 'events.id', '=', 'event_media.event_id')
                ->where('events.created_by', $owner->id)
                ->sum('event_media.size');

            if ($usedBytes + max(0, $bytesToAdd) > $storageLimitBytes) {
                throw new PlanLimitReachedException('storage', $usedBytes, $storageLimitBytes);
            }
        }
    }

    /**
     * Assert that a specific plan feature is enabled for the event owner.
     *
     * @throws PlanLimitReachedException
     */
    public function assertFeatureEnabled(User $user, string $feature): void
    {
        if (! $this->shouldEnforce($user)) {
            return;
        }

        $plan = $this->resolvePlan($user);
        if (! $plan) {
            return;
        }

        $value = $plan->getAttribute($feature);
        if ($value === null) {
            return; // Column doesn't exist yet or no restriction
        }

        if (! (bool) $value) {
            throw new PlanLimitReachedException($feature, 0, 0);
        }
    }

    /**
     * Check if a plan feature is enabled without throwing.
     */
    public function hasFeature(User $user, string $feature): bool
    {
        if (! $this->shouldEnforce($user)) {
            return true;
        }

        $plan = $this->resolvePlan($user);
        if (! $plan) {
            return true;
        }

        $value = $plan->getAttribute($feature);
        if ($value === null) {
            return true;
        }

        return (bool) $value;
    }

    public function getUsage(User $user): array
    {
        $plan = $this->resolvePlan($user);
        
        $activeEvents = Event::query()
            ->where('created_by', $user->id)
            ->where('is_active', true)
            ->count();
            
        $totalImages = EventMedia::query()
            ->where('event_media.file_type', 'image')
            ->join('events', 'events.id', '=', 'event_media.event_id')
            ->where('events.created_by', $user->id)
            ->count();

        $usedBytes = (int) EventMedia::query()
            ->join('events', 'events.id', '=', 'event_media.event_id')
            ->where('events.created_by', $user->id)
            ->sum('event_media.size');
            
        return [
            'plan' => $plan,
            'events' => [
                'used' => $activeEvents,
                'limit' => $plan ? (int) $plan->events_limit : null,
            ],
            'images' => [
                'used' => $totalImages,
                'limit' => $plan ? (int) $plan->images_limit : null,
            ],
            'videos' => [
                'used' => EventMedia::query()
                    ->where('event_media.file_type', 'video')
                    ->join('events', 'events.id', '=', 'event_media.event_id')
                    ->where('events.created_by', $user->id)
                    ->count(),
                'limit' => $plan ? $plan->videos_limit : null,
            ],
            'storage' => [
                'used' => $usedBytes,
                'limit' => $plan ? $this->planStorageLimitBytes($plan) : 0,
            ],
            'max_file_size_mb' => $plan ? $plan->max_file_size_mb : null,
            'features' => [
                'has_watermark' => $plan ? (bool) $plan->has_watermark : true,
                'has_design_editor' => $plan ? (bool) $plan->has_design_editor : false,
                'has_guest_upload' => $plan ? (bool) $plan->has_guest_upload : true,
                'has_google_drive_import' => $plan ? (bool) $plan->has_google_drive_import : false,
                'has_ftp_import' => $plan ? (bool) $plan->has_ftp_import : false,
                'has_custom_branding' => $plan ? (bool) $plan->has_custom_branding : false,
                'has_cloud_storage' => $plan ? (bool) $plan->has_cloud_storage : false,
            ]
        ];
    }

    private function resolveOwner(Event $event): ?User
    {
        if ($event->relationLoaded('createdBy') && $event->createdBy) {
            return $event->createdBy;
        }

        if ($event->created_by) {
            return User::query()->find($event->created_by);
        }

        return auth()->user();
    }

    private function resolvePlan(User $user): ?Plan
    {
        $cacheKey = $user->id;
        if (isset($this->resolvedPlans[$cacheKey])) {
            return $this->resolvedPlans[$cacheKey];
        }

        if (method_exists($user, 'plan') && $user->relationLoaded('plan') && $user->plan) {
            return $this->resolvedPlans[$cacheKey] = $user->plan;
        }

        if (method_exists($user, 'plan') && $user->plan) {
            return $this->resolvedPlans[$cacheKey] = $user->plan;
        }

        $planId = $user->getAttribute('plan_id');
        if ($planId && $this->plansTableExists()) {
            $plan = Plan::query()->find($planId);
            if ($plan) {
                return $this->resolvedPlans[$cacheKey] = $plan;
            }
        }

        $planSlug = $user->getAttribute('plan_slug') ?: $user->getAttribute('plan');
        if (is_string($planSlug) && $planSlug !== '' && $this->plansTableExists()) {
            $plan = Plan::query()->where('slug', $planSlug)->first();
            if ($plan) {
                return $this->resolvedPlans[$cacheKey] = $plan;
            }
        }

        $planName = $user->getAttribute('plan_name');
        if (is_string($planName) && $planName !== '' && $this->plansTableExists()) {
            $plan = Plan::query()->where('name', $planName)->first();
            if ($plan) {
                return $this->resolvedPlans[$cacheKey] = $plan;
            }
        }

        return $this->resolvedPlans[$cacheKey] = $this->resolveDefaultPlan();
    }

    private function resolveDefaultPlan(): ?Plan
    {
        if (! $this->plansTableExists()) {
            return null;
        }

        return Plan::query()
            ->where('slug', 'free')
            ->orWhere('name', 'Free')
            ->orWhere('price', 0)
            ->orderByRaw("CASE WHEN slug = 'free' THEN 0 WHEN name = 'Free' THEN 1 ELSE 2 END")
            ->orderBy('id')
            ->first();
    }

    private function planStorageLimitBytes(Plan $plan): int
    {
        $limit = $plan->storage_limit_gb;
        if ($limit === null || $limit <= 0) {
            return 0;
        }

        return (int) round((float) $limit * 1024 * 1024 * 1024);
    }

    private function shouldEnforce(User $user): bool
    {
        return ! (method_exists($user, 'hasAnyRole') && $user->hasAnyRole(['Super Admin', 'Admin']));
    }
}
