<?php

namespace App\Services;

use App\Exceptions\StorageLimitReachedException;
use App\Models\User;
use App\Models\EventMedia;
use App\Support\CloudStorageSettings;
use App\Support\EventMediaSettings;

class StorageUsageService
{
    private ?int $cachedUsed = null;
    private ?int $cachedLimit = null;

    public function usedBytes(?User $user = null): int
    {
        $query = EventMedia::query();
        if ($user) {
            $query->whereHas('event', function ($q) use ($user) {
                $q->where('created_by', $user->id);
            });
        }

        return (int) $query->sum('size');
    }

    public function limitBytes(?User $user = null): int
    {
        if ($user && $user->plan) {
            $limitGb = (float) $user->plan->storage_limit_gb;

            return $limitGb > 0 ? $this->convertToBytes($limitGb, 'gb') : 0;
        }

        $limitConfigured = (bool) CloudStorageSettings::getValue('limits.configured', false);
        $cloudEnabled = (bool) CloudStorageSettings::getValue('limits.enabled', false);
        $limitValue = CloudStorageSettings::getValue('limits.value');
        $limitUnit = (string) CloudStorageSettings::getValue('limits.unit', 'gb');

        if ($limitConfigured) {
            if (! $cloudEnabled) {
                return 0;
            }

            if ($limitValue === null || $limitValue === '') {
                return 0;
            }

            $limitValue = (float) $limitValue;
            if ($limitValue <= 0) {
                return 0;
            }

            return $this->convertToBytes($limitValue, $limitUnit);
        }

        $legacyLimit = (float) EventMediaSettings::getValue('storage_limit_gb', config('events.media.storage_limit_gb', 6.1));
        if ($legacyLimit <= 0) {
            return 0;
        }

        return $this->convertToBytes($legacyLimit, 'gb');
    }

    public function percentUsed(?User $user = null): int
    {
        $limit = $this->limitBytes($user);
        if ($limit <= 0) {
            return 0;
        }

        return min(100, (int) round(($this->usedBytes($user) / $limit) * 100));
    }

    public function warningLevel(?User $user = null): ?int
    {
        $percent = $this->percentUsed($user);

        if ($percent >= 90) {
            return 90;
        }

        if ($percent >= 80) {
            return 80;
        }

        return null;
    }

    public function assertCanStore(int $bytesToAdd, ?User $user = null): void
    {
        $limit = $this->limitBytes($user);
        if ($limit <= 0) {
            return;
        }

        $projected = $this->usedBytes($user) + max(0, $bytesToAdd);
        if ($projected > $limit) {
            throw new StorageLimitReachedException($this->usedBytes($user), $limit);
        }
    }

    public function summary(?User $user = null): array
    {
        $used = $this->usedBytes($user);
        $limit = $this->limitBytes($user);
        $warning = $this->warningLevel($user);

        return [
            'used_bytes' => $used,
            'limit_bytes' => $limit,
            'percent' => $this->percentUsed($user),
            'used_label' => self::formatBytes($used),
            'limit_label' => $limit > 0 ? self::formatBytes($limit) : 'Unlimited',
            'warning' => $warning,
        ];
    }

    public static function formatBytes(int $bytes): string
    {
        if ($bytes <= 0) {
            return '0 B';
        }

        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $power = (int) floor(log($bytes, 1024));
        $power = min($power, count($units) - 1);
        $value = $bytes / (1024 ** $power);
        $precision = $power === 0 ? 0 : 1;

        return number_format($value, $precision) . ' ' . $units[$power];
    }

    private function convertToBytes(float $value, string $unit): int
    {
        $unit = strtolower(trim($unit));

        return match ($unit) {
            'tb' => (int) round($value * 1024 * 1024 * 1024 * 1024),
            default => (int) round($value * 1024 * 1024 * 1024),
        };
    }
}
