<?php

namespace App\Support;

use App\Models\Currency;
use App\Models\SystemSetting;
use DateTimeInterface;
use Illuminate\Support\Facades\Schema;

class FormatSettings
{
    private static ?array $generalCache = null;
    private static ?array $currencyCache = null;

    public static function date(?DateTimeInterface $value): string
    {
        if (! $value) {
            return '-';
        }

        return $value->format(self::dateFormat());
    }

    public static function time(?DateTimeInterface $value): string
    {
        if (! $value) {
            return '-';
        }

        return $value->format(self::timeFormat());
    }

    public static function dateTime(?DateTimeInterface $value): string
    {
        if (! $value) {
            return '-';
        }

        return $value->format(self::dateFormat() . ' ' . self::timeFormat());
    }

    public static function currency($amount, ?int $precision = null): string
    {
        if ($amount === null || $amount === '') {
            return '-';
        }

        $config = self::currencyConfig();
        $decimals = $precision ?? ($config['precision'] ?? 2);
        $formatted = number_format((float) $amount, $decimals);

        $symbol = $config['symbol'] ?? '';
        $code = $config['code'] ?? '';

        if ($symbol !== '') {
            return $symbol . $formatted;
        }

        if ($code !== '') {
            return $code . ' ' . $formatted;
        }

        return $formatted;
    }

    public static function reset(): void
    {
        self::$generalCache = null;
        self::$currencyCache = null;
    }

    private static function dateFormat(): string
    {
        $general = self::generalSettings();
        $format = trim((string) ($general['date_format'] ?? ''));

        return $format !== '' ? $format : 'M d, Y';
    }

    private static function timeFormat(): string
    {
        $general = self::generalSettings();
        $format = trim((string) ($general['time_format'] ?? ''));

        return $format !== '' ? $format : 'H:i';
    }

    private static function generalSettings(): array
    {
        if (self::$generalCache !== null) {
            return self::$generalCache;
        }

        $defaults = [
            'default_currency' => '',
            'date_format' => '',
            'time_format' => '',
        ];

        try {
            if (! Schema::hasTable('system_settings')) {
                self::$generalCache = $defaults;
                return $defaults;
            }
        } catch (\Exception $e) {
            self::$generalCache = $defaults;
            return $defaults;
        }

        $payload = SystemSetting::query()->value('payload');
        if (is_string($payload)) {
            $decoded = json_decode($payload, true);
            $payload = is_array($decoded) ? $decoded : [];
        }

        $general = is_array($payload) ? ($payload['general'] ?? []) : [];
        $settings = array_merge($defaults, is_array($general) ? $general : []);

        self::$generalCache = $settings;

        return $settings;
    }

    public static function currencyConfig(): array
    {
        if (self::$currencyCache !== null) {
            return self::$currencyCache;
        }

        $defaults = [
            'code' => '',
            'symbol' => '',
            'precision' => 2,
        ];

        $general = self::generalSettings();
        $code = trim((string) ($general['default_currency'] ?? ''));

        $hasCurrenciesTable = false;
        try { $hasCurrenciesTable = Schema::hasTable('currencies'); } catch (\Exception $e) {}
        if ($code === '' || ! $hasCurrenciesTable) {
            self::$currencyCache = $defaults;
            return $defaults;
        }

        $currency = Currency::query()->where('code', $code)->first();

        if (! $currency) {
            self::$currencyCache = array_merge($defaults, ['code' => $code]);
            return self::$currencyCache;
        }

        self::$currencyCache = [
            'code' => $currency->code,
            'symbol' => $currency->symbol ?? '',
            'precision' => $currency->precision ?? 2,
        ];

        return self::$currencyCache;
    }
}
