<?php

namespace App\Support;

use Illuminate\Support\Str;

class GuestTokenService
{
    public function issue(int $eventId, int $minutes = 30): string
    {
        $payload = [
            'event_id' => $eventId,
            'expires_at' => now()->addMinutes($minutes)->timestamp,
            'nonce' => Str::random(12),
        ];

        $encoded = $this->encode($payload);
        $signature = $this->sign($encoded);

        return $encoded.'.'.$signature;
    }

    public function validate(string $token): ?array
    {
        if ($token === '' || ! str_contains($token, '.')) {
            return null;
        }

        [$encoded, $signature] = explode('.', $token, 2);

        if (! hash_equals($this->sign($encoded), $signature)) {
            return null;
        }

        $payload = $this->decode($encoded);

        if (! is_array($payload)) {
            return null;
        }

        $expires = (int) ($payload['expires_at'] ?? 0);

        if ($expires <= 0 || $expires < now()->timestamp) {
            return null;
        }

        return $payload;
    }

    private function encode(array $payload): string
    {
        return rtrim(strtr(base64_encode(json_encode($payload)), '+/', '-_'), '=');
    }

    private function decode(string $encoded): ?array
    {
        $decoded = base64_decode(strtr($encoded, '-_', '+/'));
        if ($decoded === false) {
            return null;
        }

        $payload = json_decode($decoded, true);

        return is_array($payload) ? $payload : null;
    }

    private function sign(string $encoded): string
    {
        $key = config('app.key');

        if (! $key) {
            throw new \RuntimeException('APP_KEY must be set for guest token signing');
        }

        return hash_hmac('sha256', $encoded, $key);
    }
}
