<?php

namespace App\Support;

use Illuminate\Support\Facades\Storage;

/**
 * Protects media files on disk by prepending a 32-byte header.
 *
 * Header layout: 8-byte magic ("SNAPFILE") + 24 random bytes.
 * Files become unreadable by image viewers/explorers. Combined with
 * UUID filenames (no extension), stored files are fully opaque.
 *
 * Deobfuscation happens only when serving through MediaPreviewController
 * (signed URLs) or during background image processing.
 */
class MediaObfuscator
{
    public const MAGIC = 'SNAPFILE';   // 8 bytes
    public const HEADER_SIZE = 32;     // 8 magic + 24 random

    /**
     * Generate a 32-byte obfuscation header.
     */
    public static function generateHeader(): string
    {
        return self::MAGIC . random_bytes(24);
    }

    /**
     * Obfuscate a local file in-place (streaming, memory-efficient).
     */
    public static function obfuscate(string $filePath): void
    {
        if (! is_file($filePath)) {
            return;
        }

        $tempPath = $filePath . '.obf';
        $src = fopen($filePath, 'rb');
        $dst = fopen($tempPath, 'wb');

        fwrite($dst, self::generateHeader());
        stream_copy_to_stream($src, $dst);

        fclose($dst);
        fclose($src);

        rename($tempPath, $filePath);
    }

    /**
     * Check if a local file starts with the obfuscation magic.
     */
    public static function isObfuscated(string $filePath): bool
    {
        if (! is_file($filePath)) {
            return false;
        }

        $f = fopen($filePath, 'rb');
        if (! $f) {
            return false;
        }

        $magic = fread($f, 8);
        fclose($f);

        return $magic === self::MAGIC;
    }

    /**
     * Create a clean (deobfuscated) temp copy of an obfuscated local file.
     * Caller must @unlink() the returned path when done.
     */
    public static function deobfuscateToTemp(string $filePath): string
    {
        $dir = storage_path('app/tmp/media-processing');
        if (! is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }

        $temp = tempnam($dir, 'deobf_');
        $src = fopen($filePath, 'rb');
        $dst = fopen($temp, 'wb');

        // Skip the 32-byte header
        fread($src, self::HEADER_SIZE);
        stream_copy_to_stream($src, $dst);

        fclose($dst);
        fclose($src);

        return $temp;
    }

    /**
     * Obfuscate a file on any disk (local or cloud).
     */
    public static function obfuscateOnDisk(string $diskName, string $path): void
    {
        $disk = Storage::disk($diskName);

        if (config("filesystems.disks.{$diskName}.driver") === 'local') {
            self::obfuscate($disk->path($path));
            return;
        }

        // Cloud: download → prepend header → re-upload
        $stream = $disk->readStream($path);
        if (! $stream) {
            return;
        }

        $temp = tempnam(sys_get_temp_dir(), 'obf_');
        $dst = fopen($temp, 'wb');
        fwrite($dst, self::generateHeader());
        stream_copy_to_stream($stream, $dst);
        fclose($dst);

        if (is_resource($stream)) {
            fclose($stream);
        }

        $disk->put($path, fopen($temp, 'rb'));
        @unlink($temp);
    }

    /**
     * Create a StreamedResponse that auto-detects and strips obfuscation.
     * Works for both obfuscated and legacy (non-obfuscated) files.
     */
    public static function streamResponse(string $diskName, string $path, string $fileName, string $mimeType): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        return response()->stream(function () use ($diskName, $path) {
            $stream = Storage::disk($diskName)->readStream($path);

            // Read first 8 bytes to check for magic
            $magic = fread($stream, 8);

            if ($magic === self::MAGIC) {
                // Obfuscated — skip remaining 24 bytes of header
                fread($stream, 24);
            } else {
                // Not obfuscated — output the 8 bytes we already consumed
                echo $magic;
            }

            fpassthru($stream);

            if (is_resource($stream)) {
                fclose($stream);
            }
        }, 200, [
            'Content-Type' => $mimeType ?: 'application/octet-stream',
            'Content-Disposition' => 'inline; filename="' . addslashes($fileName) . '"',
            'Cache-Control' => 'private, max-age=600',
        ]);
    }
}
