<?php

namespace App\Support;

use App\Models\BlogPost;
use App\Models\Page;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;

class SeoFileManager
{
    public static function sync(array $settings): void
    {
        $publicPath = public_path();

        self::writeTextFile($publicPath . DIRECTORY_SEPARATOR . 'robots.txt', $settings['robots_txt'] ?? '');
        self::writeTextFile($publicPath . DIRECTORY_SEPARATOR . 'llm.txt', $settings['llm_txt'] ?? '');

        $sitemapEnabled = (bool) ($settings['sitemap_enabled'] ?? false);

        if ($sitemapEnabled) {
            $sitemap = self::buildSitemapXml();
            if ($sitemap !== '') {
                File::put($publicPath . DIRECTORY_SEPARATOR . 'sitemap.xml', $sitemap);
            }
        } else {
            self::deleteFile($publicPath . DIRECTORY_SEPARATOR . 'sitemap.xml');
        }
    }

    private static function writeTextFile(string $path, string $content): void
    {
        $content = trim($content);

        if ($content === '') {
            self::deleteFile($path);
            return;
        }

        File::put($path, $content . PHP_EOL);
    }

    private static function deleteFile(string $path): void
    {
        if (File::exists($path)) {
            File::delete($path);
        }
    }

    private static function buildSitemapXml(): string
    {
        $baseUrl = rtrim((string) config('app.url'), '/');

        if ($baseUrl === '') {
            return '';
        }

        $urls = [[
            'loc' => $baseUrl,
            'lastmod' => now()->toAtomString(),
        ]];

        $hasPagesTable = false;
        try { $hasPagesTable = Schema::hasTable('pages'); } catch (\Exception $e) {}
        if ($hasPagesTable) {
            Page::query()
                ->where('is_active', true)
                ->orderBy('updated_at', 'desc')
                ->get(['slug', 'updated_at'])
                ->each(function (Page $page) use (&$urls, $baseUrl) {
                    if (! $page->slug) {
                        return;
                    }

                    $urls[] = [
                        'loc' => $baseUrl . '/pages/' . ltrim($page->slug, '/'),
                        'lastmod' => optional($page->updated_at)->toAtomString(),
                    ];
                });
        }

        $hasBlogPostsTable = false;
        try { $hasBlogPostsTable = Schema::hasTable('blog_posts'); } catch (\Exception $e) {}
        if ($hasBlogPostsTable) {
            BlogPost::query()
                ->where('is_active', true)
                ->orderBy('updated_at', 'desc')
                ->get(['slug', 'updated_at', 'published_at'])
                ->each(function (BlogPost $post) use (&$urls, $baseUrl) {
                    if (! $post->slug) {
                        return;
                    }

                    $lastmod = $post->published_at ?: $post->updated_at;

                    $urls[] = [
                        'loc' => $baseUrl . '/blog/' . ltrim($post->slug, '/'),
                        'lastmod' => optional($lastmod)->toAtomString(),
                    ];
                });
        }

        $lines = [
            '<?xml version="1.0" encoding="UTF-8"?>',
            '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">',
        ];

        foreach ($urls as $url) {
            $lines[] = '  <url>';
            $lines[] = '    <loc>' . self::escapeXml($url['loc']) . '</loc>';
            if (! empty($url['lastmod'])) {
                $lines[] = '    <lastmod>' . self::escapeXml($url['lastmod']) . '</lastmod>';
            }
            $lines[] = '  </url>';
        }

        $lines[] = '</urlset>';

        return implode(PHP_EOL, $lines) . PHP_EOL;
    }

    private static function escapeXml(string $value): string
    {
        return htmlspecialchars($value, ENT_XML1 | ENT_QUOTES, 'UTF-8');
    }
}
