<?php

namespace App\Support;

use App\Models\Plan;
use App\Models\User;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;

class UserDefaults
{
    public const DEFAULT_ROLE_NAME = 'User';
    public const DEFAULT_PLAN_NAME = 'Free';

    public static function ensureDefaultPlan(): ?Plan
    {
        $slug = Str::slug(self::DEFAULT_PLAN_NAME);
        $plan = Plan::query()->where('slug', $slug)->first();

        if (! $plan) {
            $plan = Plan::query()->where('name', self::DEFAULT_PLAN_NAME)->first();
        }

        if (! $plan) {
            $plan = Plan::create([
                'name' => self::DEFAULT_PLAN_NAME,
                'slug' => $slug,
                'price' => 0,
                'offer_price' => null,
                'features' => ['1 active event', '500 photos', 'Basic sharing'],
                'images_limit' => 500,
                'events_limit' => 1,
                'storage_limit_gb' => null,
                'has_watermark' => true,
                'has_design_editor' => false,
                'duration_type' => 'lifetime',
                'duration_months' => null,
                'is_active' => true,
            ]);
        }

        return $plan;
    }

    public static function defaultPlanId(): ?int
    {
        return self::ensureDefaultPlan()?->id;
    }

    public static function ensureDefaultRole(): ?Role
    {
        return Role::firstOrCreate([
            'name' => self::DEFAULT_ROLE_NAME,
            'guard_name' => 'web',
        ]);
    }

    public static function assignDefaultRole(User $user): void
    {
        if ($user->roles()->exists()) {
            return;
        }

        $role = self::ensureDefaultRole();

        if ($role) {
            $user->assignRole($role);
        }
    }

    public static function assignDefaultPlan(User $user): void
    {
        if ($user->plan_id) {
            return;
        }

        $plan = self::ensureDefaultPlan();

        if ($plan) {
            $user->plan()->associate($plan);
            $user->save();
        }
    }

    public static function assignDefaults(User $user): void
    {
        self::assignDefaultPlan($user);
        self::assignDefaultRole($user);
    }
}
