<?php

use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use App\Http\Middleware\EnsureAdminPermission;
use App\Http\Middleware\EnsureInstalled;
use App\Http\Middleware\SetLocale;
use App\Http\Middleware\SetStorageUrl;
use App\Http\Middleware\EnsureMaintenanceMode;
use App\Http\Middleware\EnsureBrowserAllowed;
use App\Http\Middleware\EnsureIpNotBlocked;
use App\Http\Middleware\EnsureLocationNotBlocked;
use App\Http\Middleware\LogAdminActivity;
use App\Support\SystemLogger;
use Spatie\Permission\Middleware\PermissionMiddleware;
use Spatie\Permission\Middleware\RoleMiddleware;
use Spatie\Permission\Middleware\RoleOrPermissionMiddleware;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: __DIR__.'/../routes/web.php',
        api: __DIR__.'/../routes/api.php',
        commands: __DIR__.'/../routes/console.php',
        channels: __DIR__.'/../routes/channels.php',
        health: '/up',
    )
    ->withMiddleware(function (Middleware $middleware): void {
        $middleware->web(prepend: [
            EnsureInstalled::class,
        ]);

        $middleware->alias([
            'role' => RoleMiddleware::class,
            'permission' => PermissionMiddleware::class,
            'role_or_permission' => RoleOrPermissionMiddleware::class,
            'guest.token' => \App\Http\Middleware\EnsureGuestToken::class,
            'admin.permission' => EnsureAdminPermission::class,
            'api.permission' => \App\Http\Middleware\EnsureApiPermission::class,
            'install.session' => \App\Http\Middleware\UseFileSession::class,
        ]);

        $middleware->validateCsrfTokens(except: [
            'install/execute',
            'payment/callback/*',
            'payment/webhook/*',
        ]);

        $middleware->web(append: [
            \App\Http\Middleware\SecurityHeaders::class,
            SetStorageUrl::class,
            SetLocale::class,
            EnsureIpNotBlocked::class,
            EnsureLocationNotBlocked::class,
            EnsureMaintenanceMode::class,
            EnsureBrowserAllowed::class,
            LogAdminActivity::class,
        ]);

        $middleware->statefulApi()->throttleApi();
        $middleware->api(append: [
            SetStorageUrl::class,
            \App\Http\Middleware\SetApiLocale::class,
            EnsureIpNotBlocked::class,
            EnsureLocationNotBlocked::class,
            \App\Http\Middleware\EnsureMaintenanceModeApi::class,
            \App\Http\Middleware\AddRateLimitHeaders::class,
        ]);
    })
    ->withExceptions(function (Exceptions $exceptions): void {
        $exceptions->reportable(function (\Throwable $exception) {
            if (app()->runningInConsole()) {
                return;
            }

            if (! app()->bound('request')) {
                return;
            }

            $request = request();
            $routeName = $request->route()?->getName();

            if (! $routeName || ! str_starts_with($routeName, 'admin.')) {
                return;
            }

            try {
                SystemLogger::fromException($exception, $request);
            } catch (\Throwable) {
                // Cannot log when core services unavailable (e.g., no APP_KEY)
            }
        });
    })->create();
