<?php

namespace Database\Seeders;

use App\Models\BlogCategory;
use App\Models\BlogPost;
use App\Models\FactStat;
use App\Models\Faq;
use App\Models\Page;
use App\Models\Testimonial;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class ContentSeeder extends Seeder
{
    public function run(): void
    {
        $creatorId = User::query()->where('email', 'test@example.com')->value('id');

        $appName = config('app.name', 'SnapNest');

        $pages = [
            [
                'title' => 'About Us',
                'slug' => 'about-us',
                'type' => 'about',
                'description' => $appName . ' helps photographers deliver secure, branded event galleries in minutes.',
                'content' => [
                    'badge_text' => 'Our Mission',
                    'hero_title' => 'Redefining the',
                    'hero_title_highlight' => 'Moment of Delivery.',
                    'hero_description' => $appName . ' was born from a simple observation: photographers spend too much time on logistics and not enough time behind the lens. We\'re here to change that.',
                    'stats' => [
                        ['label' => 'Photos Delivered', 'value' => '50M+'],
                        ['label' => 'Active Studios', 'value' => '2,000+'],
                        ['label' => 'Countries', 'value' => '45+'],
                        ['label' => 'Uptime', 'value' => '99.99%'],
                    ],
                    'story_title' => 'How it started.',
                    'story_paragraph_1' => 'In 2022, our founders noticed a recurring problem at high-profile events: guests would wait days, even weeks, to receive their professional photos. The "hype" of the event had long passed by the time the gallery link arrived.',
                    'story_paragraph_2' => 'We built ' . $appName . ' to bridge that gap. By leveraging advanced facial recognition and automated delivery workflows, we\'ve enabled photographers to share memories while they are still fresh—creating a "wow" moment for every guest.',
                    'card_1_title' => 'Passion First',
                    'card_1_description' => 'We are photographers at heart, building for photographers.',
                    'card_2_title' => 'Pixel Perfect',
                    'card_2_description' => 'Every feature is crafted with meticulous attention to detail.',
                ],
            ],
            [
                'title' => 'Contact Us',
                'slug' => 'contact-us',
                'type' => 'contact',
                'description' => 'Reach out to our support team for product questions and onboarding help.',
                'content' => [
                    'badge_text' => '24/7 Support Available',
                    'hero_title' => 'Let\'s start a',
                    'hero_title_highlight' => 'Conversation.',
                    'hero_description' => 'Have a specific query about our platform? Our team is ready to assist you in scaling your photography business.',
                    'support_label' => 'Support',
                    'support_text' => 'Live Chat - 24/7',
                    'response_label' => 'Avg. Response',
                    'response_value' => '15m',
                    'sales_label' => 'Global Sales',
                    'sales_value' => 'Active',
                ],
            ],
            [
                'title' => 'Privacy Policy',
                'slug' => 'privacy-policy',
                'type' => 'privacy_policy',
                'description' => 'We respect your privacy and protect your data with industry-standard safeguards.',
                'content' => [
                    'title' => 'Privacy',
                    'title_highlight' => 'Matters.',
                    'subtitle' => 'We believe privacy is a fundamental human right. Here is how we protect yours and your clients\'.',
                    'last_updated' => 'Oct 2023',
                    'sections' => [
                        ['title' => 'Information We Collect', 'content' => 'We collect information you provide directly to us when you create an account, create a gallery, or communicate with us. This includes your name, email address, studio name, and any images you upload for the purpose of event delivery.'],
                        ['title' => 'How We Use Face Search', 'content' => 'Our Face Search technology is designed purely for one-to-many matching within your private galleries. We do not build global face databases. Biometric data is stored securely and is only used to help guests find their photos at specific events.'],
                        ['title' => 'Data Security', 'content' => 'We use industry-standard encryption and security protocols to protect your data and your clients\' images. All photos are stored on redundant, secure cloud servers with strict access controls.'],
                        ['title' => 'Third-Party Sharing', 'content' => 'We do not sell your data. We only share information with third-party services (like WhatsApp or Cloud Providers) as necessary to provide our core services and ensure the best experience for your clients.'],
                    ],
                    'bottom_paragraph_1' => 'At ' . $appName . ', we understand the sensitive nature of photography. Whether it is a private wedding, a high-stakes corporate gathering, or a family function, we prioritize the confidentiality and security of every image uploaded to our platform.',
                    'bottom_paragraph_2' => 'Your images are your property. We only process them to enable the features you\'ve selected, such as face matching for instant delivery. We do not use your images for marketing purposes without explicit consent, and we provide robust tools for users to request data deletion.',
                ],
            ],
            [
                'title' => 'Terms and Conditions',
                'slug' => 'terms-and-conditions',
                'type' => 'terms',
                'description' => 'By using ' . $appName . ', you agree to our service terms and responsible sharing policies.',
                'content' => [
                    'title' => 'Terms of',
                    'title_highlight' => 'Service',
                    'last_updated' => 'January 1, 2026',
                    'sections' => [
                        ['title' => 'Introduction', 'content' => 'Welcome to ' . $appName . '. These Terms of Service govern your use of the ' . $appName . ' platform, website, and associated services. By creating an account or using our platform, you agree to these terms in full.'],
                        ['title' => 'Services & Licensing', 'content' => $appName . ' provides a photography hosting and distribution platform powered by AI. We grant you a limited, non-exclusive license to use our platform for your professional photography business.'],
                        ['title' => 'User Responsibilities', 'content' => 'As a photographer on our platform, you are responsible for the content you upload. You must ensure you have the necessary rights to distribute the images and that your content does not violate any laws or third-party rights.'],
                        ['title' => 'AI Face Recognition', 'content' => 'Our AI features are designed to enhance guest experience. You agree to inform your clients and participants that AI indexing may be used for the purpose of photo delivery, in accordance with our Privacy Policy.'],
                        ['title' => 'Payment & Subscriptions', 'content' => 'Access to certain features requires a paid subscription. All fees are non-refundable unless otherwise specified. We reserve the right to change our pricing upon reasonable notice.'],
                        ['title' => 'Termination of Service', 'content' => 'We reserve the right to suspend or terminate accounts that violate these terms, engage in fraudulent activity, or disrupt the quality of service for other users.'],
                    ],
                ],
            ],
            [
                'title' => 'Cookie Policy',
                'slug' => 'cookie-policy',
                'type' => 'cookie_policy',
                'description' => 'How we use cookies and similar technologies to improve your experience.',
                'content' => [
                    'title' => 'Cookie',
                    'title_highlight' => 'Policy',
                    'last_updated' => 'January 1, 2026',
                    'intro_text' => 'At ' . $appName . ', we use cookies to provide a better experience and analyze our traffic. This policy explains what cookies are, how we use them, and your choices regarding their use.',
                    'cookie_types' => [
                        ['title' => 'Essential Cookies', 'description' => 'These are necessary for the website to function. They include session management and security cookies.', 'priority' => 'Mandatory'],
                        ['title' => 'Performance Cookies', 'description' => 'We use these to understand how visitors interact with our platform, help us identify areas for improvement.', 'priority' => 'Optional'],
                        ['title' => 'Functionality Cookies', 'description' => 'These allow the website to remember choices you make, such as your language preference or layout settings.', 'priority' => 'Optional'],
                    ],
                    'manage_title' => 'How to Manage Cookies',
                    'manage_text' => 'Most web browsers allow you to control cookies through their settings preferences. However, if you limit the ability of websites to set cookies, you may worsen your overall user experience, as it will no longer be personalized to you.',
                ],
            ],
        ];

        foreach ($pages as $page) {
            $content = $page['content'];
            unset($page['content']);

            Page::updateOrCreate(
                ['type' => $page['type']],
                [
                    'title' => $page['title'],
                    'slug' => $page['slug'],
                    'description' => $page['description'],
                    'content' => $content,
                    'is_active' => true,
                    'created_by' => $creatorId,
                ]
            );
        }

        $faqs = [
            [
                'question' => 'How do I share an event gallery?',
                'answer' => 'Use the share link or QR code from your event details and set a pin if needed.',
                'sort_order' => 1,
            ],
            [
                'question' => 'Can I upload large batches of photos?',
                'answer' => 'Yes. Chunked uploads allow thousands of images without browser timeouts.',
                'sort_order' => 2,
            ],
            [
                'question' => 'Do guests see watermarked images?',
                'answer' => 'Guests view optimized, watermarked images while originals stay private.',
                'sort_order' => 3,
            ],
        ];

        foreach ($faqs as $faq) {
            Faq::updateOrCreate(
                ['slug' => Str::slug($faq['question'])],
                [
                    'question' => $faq['question'],
                    'answer' => $faq['answer'],
                    'sort_order' => $faq['sort_order'],
                    'is_active' => true,
                    'created_by' => $creatorId,
                ]
            );
        }

        $testimonials = [
            [
                'client_name' => 'Sarah Jenkins',
                'client_position' => 'Luxury Wedding Photographer',
                'rating' => 5,
                'review' => 'This changed everything. I used to spend weeks on delivery. Now guests get their photos instantly, and my referrals have tripled.',
            ],
            [
                'client_name' => 'Mike Chen',
                'client_position' => 'Global Event Producer',
                'rating' => 5,
                'review' => 'The facial recognition is flawless. Clients are consistently blown away when seeing their personal galleries in milliseconds.',
            ],
            [
                'client_name' => 'Elena Rodriguez',
                'client_position' => 'Studio Director',
                'rating' => 5,
                'review' => 'Revenue increased by 40% because guests actually buy prints and digital downloads when the excitement is high.',
            ],
            [
                'client_name' => 'David Smith',
                'client_position' => 'Elite Sports Photographer',
                'rating' => 5,
                'review' => 'Marathons used to be an indexing nightmare. Now AI does it for me perfectly. I can focus on shooting, not sorting.',
            ],
            [
                'client_name' => 'Sara Lopez',
                'client_position' => 'Wedding Photographer',
                'rating' => 5,
                'review' => 'The white-label branding makes every gallery feel like our own product. Clients never know we use a third-party tool.',
            ],
            [
                'client_name' => 'Omar Khan',
                'client_position' => 'Event Studio Owner',
                'rating' => 4,
                'review' => 'Bulk uploads and sharing tools save us hours every week. Our team handles twice the events now.',
            ],
            [
                'client_name' => 'Priya Sharma',
                'client_position' => 'Corporate Event Photographer',
                'rating' => 5,
                'review' => 'The QR code sharing feature is brilliant. Guests scan at the venue and instantly see their photos. No more email follow-ups.',
            ],
            [
                'client_name' => 'James Wilson',
                'client_position' => 'Festival Photographer',
                'rating' => 5,
                'review' => 'Handled 15,000 photos from a three-day music festival without a hitch. The chunked upload system is rock solid.',
            ],
            [
                'client_name' => 'Aisha Patel',
                'client_position' => 'Portrait Studio Owner',
                'rating' => 5,
                'review' => 'My clients love the PIN-protected galleries. They feel secure knowing their private moments stay private.',
            ],
            [
                'client_name' => 'Carlos Mendez',
                'client_position' => 'Destination Wedding Photographer',
                'rating' => 5,
                'review' => 'Cloud storage means I can shoot abroad and deliver from anywhere. No more carrying hard drives through airports.',
            ],
            [
                'client_name' => 'Rachel Kim',
                'client_position' => 'School Event Photographer',
                'rating' => 4,
                'review' => 'Parents find their kids photos using face search in seconds. What used to take days of sorting now happens automatically.',
            ],
            [
                'client_name' => 'Thomas Berg',
                'client_position' => 'Conference Photographer',
                'rating' => 5,
                'review' => 'The guest self-upload feature is genius. Attendees contribute their own shots and everyone gets a complete event album.',
            ],
            [
                'client_name' => 'Mei Lin',
                'client_position' => 'Fashion Show Photographer',
                'rating' => 5,
                'review' => 'Real-time gallery updates during live events changed how we work. Designers see their collection shots before the show ends.',
            ],
            [
                'client_name' => 'Ryan O\'Brien',
                'client_position' => 'Sports League Photographer',
                'rating' => 4,
                'review' => 'Managing galleries for 30 teams across a season is effortless now. Each team gets their own branded gallery space.',
            ],
            [
                'client_name' => 'Fatima Al-Hassan',
                'client_position' => 'Newborn Photographer',
                'rating' => 5,
                'review' => 'The watermarking protects my work while still letting families preview beautifully. Print sales went up 60% since switching.',
            ],
            [
                'client_name' => 'Luca Rossi',
                'client_position' => 'Real Estate Photographer',
                'rating' => 5,
                'review' => 'Agents get their property galleries same day. The organized folder structure and quick sharing link saves me so much time.',
            ],
            [
                'client_name' => 'Nina Johansson',
                'client_position' => 'Wildlife Photographer',
                'rating' => 4,
                'review' => 'Even my massive RAW files upload smoothly with the chunked system. No more failed transfers on slow connections.',
            ],
            [
                'client_name' => 'Vikram Rao',
                'client_position' => 'Multi-Studio Franchise Owner',
                'rating' => 5,
                'review' => 'We run five studios on one account. The role-based access means each team only sees their own events. Perfect control.',
            ],
            [
                'client_name' => 'Sophie Dubois',
                'client_position' => 'Charity Gala Photographer',
                'rating' => 5,
                'review' => 'Donors love receiving their event photos instantly. It keeps the excitement alive and drives more engagement for our clients.',
            ],
            [
                'client_name' => 'Andre Thompson',
                'client_position' => 'Nightlife & Club Photographer',
                'rating' => 4,
                'review' => 'The Face Search works even in low-light party shots. My regulars find themselves instantly. Engagement on galleries is insane.',
            ],
            [
                'client_name' => 'Yuki Tanaka',
                'client_position' => 'Product Launch Photographer',
                'rating' => 5,
                'review' => 'Brand managers get curated galleries within hours of the launch event. The professional presentation keeps them coming back.',
            ],
            [
                'client_name' => 'Hannah Mitchell',
                'client_position' => 'Family Reunion Photographer',
                'rating' => 5,
                'review' => 'Grandma found all her photos using face search on her phone. If she can do it, anyone can. The interface is that simple.',
            ],
        ];

        foreach ($testimonials as $testimonial) {
            Testimonial::updateOrCreate(
                [
                    'client_name' => $testimonial['client_name'],
                    'client_position' => $testimonial['client_position'],
                ],
                [
                    'rating' => $testimonial['rating'],
                    'review' => $testimonial['review'],
                    'is_active' => true,
                    'created_by' => $creatorId,
                ]
            );
        }

        $facts = [
            ['fact' => 'Events Delivered', 'value' => '1,250+'],
            ['fact' => 'Photos Shared', 'value' => '2.4M'],
            ['fact' => 'Studios Onboarded', 'value' => '340'],
        ];

        foreach ($facts as $fact) {
            FactStat::updateOrCreate(
                ['fact' => $fact['fact']],
                [
                    'value' => $fact['value'],
                    'is_active' => true,
                    'created_by' => $creatorId,
                ]
            );
        }

        $categories = [
            ['name' => 'Announcements', 'slug' => 'announcements'],
            ['name' => 'Tips', 'slug' => 'tips'],
            ['name' => 'Product', 'slug' => 'product'],
            ['name' => 'Industry Trends', 'slug' => 'industry-trends'],
            ['name' => 'Tutorials', 'slug' => 'tutorials'],
            ['name' => 'Business', 'slug' => 'business'],
        ];

        foreach ($categories as $category) {
            BlogCategory::updateOrCreate(
                ['slug' => $category['slug']],
                [
                    'name' => $category['name'],
                    'is_active' => true,
                    'created_by' => $creatorId,
                ]
            );
        }

        $posts = [
            [
                'title' => 'Introducing SnapNest Galleries',
                'slug' => 'introducing-snapshare-galleries',
                'category' => 'announcements',
                'excerpt' => 'Deliver beautiful, secure galleries with watermarking and share controls.',
                'details' => "SnapNest brings pro-grade delivery to every event. Create branded galleries, manage access pins, and keep originals private.\n\nWith our platform, photographers can focus on what they do best — capturing moments — while we handle the technology behind seamless photo delivery. From chunked uploads that handle thousands of images to PIN-protected galleries that keep your work secure, every feature has been designed with working photographers in mind.\n\nWhether you shoot weddings, corporate events, or sports, SnapNest adapts to your workflow and scales with your business.",
                'tags' => ['launch', 'product'],
                'is_bookmarked' => true,
            ],
            [
                'title' => '5 Ways To Speed Up Your Event Workflow',
                'slug' => 'speed-up-event-workflow',
                'category' => 'tips',
                'excerpt' => 'From chunked uploads to quick share links, optimize your delivery in minutes.',
                'details' => "Speed is everything in event photography. The faster you deliver, the more impressed your clients will be.\n\n1. Use Chunked Uploads — Upload thousands of images without browser timeouts. Our system breaks large files into manageable pieces.\n\n2. Set Up FTP Tethering — Connect your camera via WiFi and photos auto-import every 2 minutes during the event.\n\n3. Pre-configure Gallery Templates — Save time by creating reusable gallery designs with your branding already applied.\n\n4. Use QR Code Sharing — Generate QR codes before the event so guests can access the gallery instantly.\n\n5. Enable Guest Self-Upload — Let attendees contribute their own photos, creating a more complete event album.",
                'tags' => ['workflow', 'tips'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Watermark Settings You Should Try',
                'slug' => 'watermark-settings-to-try',
                'category' => 'product',
                'excerpt' => 'Fine-tune watermark visibility to balance protection and presentation.',
                'details' => "Watermarking is a delicate balance. Too strong and it distracts from the image. Too subtle and it offers no protection.\n\nWe recommend starting with a semi-transparent logo placed in the lower-right corner at about 15-20% opacity. This protects your work while keeping the preview beautiful enough that clients want to purchase the full resolution.\n\nFor corporate events, consider text watermarks with your studio name. For weddings, a subtle logo works best. Always test your watermark on both light and dark images to ensure consistent visibility.",
                'tags' => ['watermark', 'security'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'How Face Search Is Changing Event Photography',
                'slug' => 'ai-face-search-changing-event-photography',
                'category' => 'industry-trends',
                'excerpt' => 'Facial recognition technology lets guests find their photos instantly by uploading a selfie.',
                'details' => "Gone are the days of scrolling through hundreds of event photos to find yourself. Face Search technology has fundamentally changed how guests interact with event galleries.\n\nHere is how it works: a guest uploads a selfie, and our AI engine scans every photo in the gallery, matching facial features with remarkable accuracy. Within seconds, they see every photo they appear in.\n\nFor photographers, this means higher engagement, more downloads, and happier clients. When guests can find their photos instantly, the emotional connection to the moment is still fresh — leading to more print sales and social shares.",
                'tags' => ['ai', 'face-search', 'technology'],
                'is_bookmarked' => true,
            ],
            [
                'title' => 'Complete Guide to PIN-Protected Galleries',
                'slug' => 'guide-pin-protected-galleries',
                'category' => 'tutorials',
                'excerpt' => 'Learn how to secure your event galleries with separate Admin and Guest PINs.',
                'details' => "Privacy matters, especially for private events like weddings and corporate gatherings. PIN-protected galleries give you complete control over who sees your photos.\n\nSnapNest offers two types of PINs:\n\n- Admin PIN: Full access to manage, upload, and delete photos\n- Guest PIN: View-only access for event attendees\n\nTo set up PINs, navigate to your event settings and enable PIN protection. Choose strong, memorable codes and share them only with authorized people. You can change PINs at any time without affecting gallery content.\n\nPro tip: Share the Guest PIN via QR code at the event venue for seamless access.",
                'tags' => ['security', 'tutorial', 'galleries'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Maximizing Revenue with Instant Photo Delivery',
                'slug' => 'maximizing-revenue-instant-delivery',
                'category' => 'business',
                'excerpt' => 'Studies show that photo purchases increase by 300% when delivered within hours of the event.',
                'details' => "The golden window for photo sales is the first 24 hours after an event. During this period, emotions are high, memories are fresh, and guests are most likely to purchase prints and digital downloads.\n\nTraditional photography workflows often take days or weeks for delivery, missing this crucial window entirely. With instant delivery through SnapNest, you capture the excitement while it lasts.\n\nPhotographers who switched to same-day delivery report:\n- 300% increase in digital download purchases\n- 60% increase in print orders\n- Triple the social media shares\n- 40% more repeat bookings\n\nThe math is simple: faster delivery equals more revenue.",
                'tags' => ['business', 'revenue', 'delivery'],
                'is_bookmarked' => true,
            ],
            [
                'title' => 'FTP Camera Tethering: The Ultimate Live Event Setup',
                'slug' => 'ftp-camera-tethering-live-event-setup',
                'category' => 'tutorials',
                'excerpt' => 'Connect your camera via WiFi and auto-import photos to your gallery every 2 minutes.',
                'details' => "FTP tethering transforms your camera into a live-streaming photo device. Photos are automatically imported into your event gallery as you shoot, giving guests access to images in near real-time.\n\nSetup is straightforward:\n1. Enable WiFi on your camera and configure FTP settings\n2. Point your camera FTP to your SnapNest event endpoint\n3. Set the import interval (we recommend every 2 minutes)\n4. Start shooting — photos appear in the gallery automatically\n\nThis is perfect for conferences, galas, and any event where guests want to see photos before they leave the venue.",
                'tags' => ['ftp', 'tethering', 'tutorial'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'White-Label Branding: Make It Yours',
                'slug' => 'white-label-branding-guide',
                'category' => 'product',
                'excerpt' => 'Remove platform branding and customize every gallery to match your studio identity.',
                'details' => "Your brand is your business. With white-label branding, every gallery looks like it was built from scratch by your studio.\n\nCustomize everything:\n- Upload your studio logo\n- Set custom color schemes matching your brand palette\n- Choose from multiple gallery layout templates\n- Add custom footer text and contact information\n- Use your own domain for gallery links\n\nClients never see SnapNest branding. Every touchpoint reinforces your professional identity, from the gallery page to download confirmations.\n\nThis is especially powerful for studios handling high-end weddings and corporate clients where brand consistency matters most.",
                'tags' => ['branding', 'customization', 'product'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Google Drive Import: Bulk Upload Made Easy',
                'slug' => 'google-drive-import-bulk-upload',
                'category' => 'tutorials',
                'excerpt' => 'Import entire photo albums from Google Drive with a single click.',
                'details' => "If you store your edited photos on Google Drive, importing them into SnapNest is effortless.\n\nConnect your Google account, browse your Drive folders, and select the album you want to import. SnapNest handles the rest — downloading, processing, and organizing all your photos into the event gallery.\n\nThis is perfect for photographers who edit on desktop and store finals in the cloud. No need to re-upload thousands of images manually.\n\nThe import runs in the background, so you can continue working while files transfer. You will receive a notification when the import is complete.",
                'tags' => ['google-drive', 'import', 'tutorial'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'The Rise of Guest Self-Upload at Events',
                'slug' => 'rise-of-guest-self-upload',
                'category' => 'industry-trends',
                'excerpt' => 'Why letting guests contribute photos creates a richer, more complete event album.',
                'details' => "Professional photos capture the polished moments, but guest photos capture the candid, behind-the-scenes magic that makes events memorable.\n\nGuest self-upload lets attendees contribute their own smartphone photos directly to your event gallery. The result is a more complete album that tells the full story of the event.\n\nFor photographers, this adds value to your service without extra work. Clients love seeing both professional and candid perspectives in one place.\n\nSet upload limits per guest to manage storage, and review submissions before they go live if you want quality control. Many photographers find that guest photos become some of the most-shared images from the event.",
                'tags' => ['guest-upload', 'trends', 'events'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'QR Code Sharing: From Venue to Gallery in Seconds',
                'slug' => 'qr-code-sharing-venue-to-gallery',
                'category' => 'tips',
                'excerpt' => 'Generate QR codes for your event galleries and display them at the venue for instant access.',
                'details' => "QR codes bridge the physical and digital worlds. Print them on table cards, display them on screens, or include them in event programs.\n\nGuests scan the code with their phone camera and instantly land on your gallery page. No app downloads, no typing URLs, no email sign-ups.\n\nBest practices for QR code placement:\n- Reception tables at weddings\n- Check-in desks at corporate events\n- Finish line areas at sporting events\n- Stage areas at conferences\n- Exit points (guests scan as they leave)\n\nWe generate high-resolution QR codes that work reliably even in low-light conditions. Test your QR code before the event to ensure smooth scanning.",
                'tags' => ['qr-code', 'sharing', 'tips'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Cloud Storage Options: S3, GCS, or Local?',
                'slug' => 'cloud-storage-options-comparison',
                'category' => 'product',
                'excerpt' => 'Compare storage providers and choose the best option for your photography business.',
                'details' => "SnapNest supports multiple storage backends so you can choose what works best for your budget and needs.\n\nAWS S3: The industry standard. Reliable, scalable, and cost-effective for most studios. Pay only for what you use.\n\nGoogle Cloud Storage: Great if you are already in the Google ecosystem. Similar pricing to S3 with excellent integration.\n\nLocal Storage: Store files on your own server. Best for studios with strict data residency requirements or existing infrastructure.\n\nYou can migrate between providers at any time without affecting your galleries. SnapNest handles the transfer in the background.\n\nFor most photographers, we recommend starting with S3 and its generous free tier.",
                'tags' => ['cloud', 'storage', 'comparison'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Pricing Your Photography Services in 2026',
                'slug' => 'pricing-photography-services-2026',
                'category' => 'business',
                'excerpt' => 'A data-driven guide to setting competitive rates that reflect your value.',
                'details' => "Pricing is one of the hardest decisions for photographers. Charge too little and you undervalue your work. Charge too much and you lose bookings.\n\nHere is a framework for 2026:\n\n1. Calculate Your Costs — Include gear, software, insurance, travel, and time for editing and delivery.\n\n2. Research Your Market — Check what photographers at your skill level charge in your area.\n\n3. Factor in Delivery Technology — Instant delivery through platforms like SnapNest adds measurable value. Clients pay premium prices for same-day galleries.\n\n4. Offer Tiered Packages — Basic (digital only), Standard (digital + prints), Premium (digital + prints + album + same-day delivery).\n\n5. Review Quarterly — Adjust prices based on demand, costs, and competition.",
                'tags' => ['pricing', 'business', 'guide'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Chunked Uploads: How We Handle 10,000+ Photos',
                'slug' => 'chunked-uploads-handle-thousands-photos',
                'category' => 'product',
                'excerpt' => 'Our upload system breaks large files into chunks for reliable, resumable transfers.',
                'details' => "Uploading thousands of high-resolution photos used to be a nightmare. Browser timeouts, failed transfers, and lost progress were common.\n\nSnapNest chunked uploads solve this by breaking each file into small pieces (typically 2MB each). Each chunk is uploaded independently with checksum verification.\n\nBenefits:\n- Resume interrupted uploads without starting over\n- Upload over slow or unreliable connections\n- Handle files of any size\n- Progress tracking for each file\n- Server-side integrity verification\n\nWe have tested this system with single uploads of 15,000+ photos from multi-day festivals. Every file arrived intact.",
                'tags' => ['uploads', 'technology', 'product'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Wedding Photography Trends to Watch in 2026',
                'slug' => 'wedding-photography-trends-2026',
                'category' => 'industry-trends',
                'excerpt' => 'From AI-powered delivery to drone coverage, here is what is shaping the industry.',
                'details' => "The wedding photography industry is evolving fast. Here are the trends dominating 2026:\n\n1. Same-Day Delivery — Couples expect to see at least a preview gallery before the reception ends. AI-powered sorting makes this possible.\n\n2. Guest-Contributed Albums — Combining professional and guest photos creates a more complete wedding story.\n\n3. AI Face Matching — Guests find their photos instantly via selfie search, dramatically increasing engagement.\n\n4. Cinematic Video + Photo Fusion — Hybrid shooters delivering both stills and short reels in one package.\n\n5. Sustainable Practices — Digital delivery reducing print waste, cloud storage replacing physical media.\n\nPhotographers who embrace these trends are commanding premium prices and booking out months in advance.",
                'tags' => ['wedding', 'trends', '2026'],
                'is_bookmarked' => true,
            ],
            [
                'title' => 'Setting Up Your First Event Gallery',
                'slug' => 'setting-up-first-event-gallery',
                'category' => 'tutorials',
                'excerpt' => 'A step-by-step guide for new photographers getting started with SnapNest.',
                'details' => "Welcome to SnapNest! Here is how to create your first event gallery in under 5 minutes.\n\nStep 1: Create an Event — Go to your dashboard and click 'New Event'. Enter the event name, date, and optional PIN codes.\n\nStep 2: Customize the Design — Choose a gallery template and apply your branding. Upload your logo and set color preferences.\n\nStep 3: Upload Photos — Drag and drop your images or use the bulk upload tool. Our chunked system handles large batches reliably.\n\nStep 4: Share the Gallery — Generate a share link or QR code. Send it to your client or display it at the venue.\n\nStep 5: Monitor Engagement — Track views, downloads, and face searches from your event analytics dashboard.\n\nThat is it! Your gallery is live and ready for guests.",
                'tags' => ['getting-started', 'tutorial', 'beginners'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Corporate Event Photography: A Complete Guide',
                'slug' => 'corporate-event-photography-guide',
                'category' => 'business',
                'excerpt' => 'Everything you need to know about photographing corporate events professionally.',
                'details' => "Corporate events are a lucrative niche for photographers who know how to deliver professionally and efficiently.\n\nBefore the Event:\n- Meet with the event planner to understand key moments and VIPs\n- Scout the venue for lighting conditions and backdrop options\n- Prepare your gallery with company branding and PIN protection\n\nDuring the Event:\n- Capture keynote speakers, networking moments, and candid interactions\n- Use FTP tethering for real-time gallery updates\n- Set up a QR code at the check-in desk\n\nAfter the Event:\n- Deliver the full gallery within 24 hours\n- Provide a curated highlight reel for social media\n- Send analytics reports showing engagement metrics\n\nCorporate clients value speed and professionalism above all. Instant delivery sets you apart from competitors.",
                'tags' => ['corporate', 'business', 'guide'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'Understanding Role-Based Access for Multi-Studio Teams',
                'slug' => 'role-based-access-multi-studio-teams',
                'category' => 'product',
                'excerpt' => 'Manage multiple photographers and studios with granular permission controls.',
                'details' => "As your photography business grows, you need a system that scales with your team.\n\nSnapNest role-based access lets you:\n- Create separate accounts for each photographer on your team\n- Assign specific events to specific team members\n- Control who can upload, edit, delete, and share\n- Keep financial data visible only to admins\n- Track individual photographer activity and performance\n\nThis is essential for multi-studio operations where different teams handle different events. Each photographer sees only their assigned work, maintaining organization and security.\n\nRoles are fully customizable — create as many permission levels as your business requires.",
                'tags' => ['teams', 'permissions', 'product'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'How to Photograph Low-Light Events Like a Pro',
                'slug' => 'photograph-low-light-events',
                'category' => 'tips',
                'excerpt' => 'Master the art of shooting in dark reception halls, clubs, and evening events.',
                'details' => "Low-light photography is challenging but essential for event photographers. Here are proven techniques:\n\n1. Gear Matters — Use fast lenses (f/1.4 or f/1.8) and cameras with excellent high-ISO performance.\n\n2. Off-Camera Flash — Bounce flash off ceilings or walls for natural-looking light. Avoid direct on-camera flash.\n\n3. Raise Your ISO Fearlessly — Modern cameras handle ISO 3200-6400 beautifully. Noise is better than blur.\n\n4. Use Continuous Autofocus — Moving subjects in dim light require your camera to constantly adjust focus.\n\n5. Shoot RAW — RAW files give you much more latitude to recover shadows and reduce noise in post-processing.\n\nFun fact: our Face Search works remarkably well even with low-light, high-ISO photos. The algorithm adapts to various lighting conditions.",
                'tags' => ['low-light', 'tips', 'technique'],
                'is_bookmarked' => false,
            ],
            [
                'title' => 'New Feature: Multi-Currency Payment Support',
                'slug' => 'multi-currency-payment-support',
                'category' => 'announcements',
                'excerpt' => 'Accept payments in 30+ currencies with automatic exchange rate conversion.',
                'details' => "We are excited to announce multi-currency support for all SnapNest subscription plans.\n\nPhotographers serving international clients can now:\n- Display prices in the client preferred currency\n- Accept payments through 30+ payment gateways worldwide\n- Automatic exchange rate conversion at checkout\n- Support for Stripe, PayPal, Razorpay, and 27 more gateways\n\nThis is a game-changer for destination wedding photographers and studios serving global corporate clients. No more manual currency conversion or payment friction.\n\nThe system automatically detects the user location and suggests the appropriate currency, while still allowing manual selection at checkout.",
                'tags' => ['announcement', 'payments', 'currency'],
                'is_bookmarked' => true,
            ],
        ];

        foreach ($posts as $index => $post) {
            $categoryId = BlogCategory::query()->where('slug', $post['category'])->value('id');

            if (! $categoryId) {
                continue;
            }

            BlogPost::updateOrCreate(
                ['slug' => $post['slug']],
                [
                    'title' => $post['title'],
                    'category_id' => $categoryId,
                    'author_name' => 'SnapNest Team',
                    'excerpt' => $post['excerpt'],
                    'details' => $post['details'],
                    'tags' => $post['tags'],
                    'is_bookmarked' => $post['is_bookmarked'],
                    'is_active' => true,
                    'published_at' => now()->subDays($index * 3 + 1),
                    'created_by' => $creatorId,
                ]
            );
        }
    }
}
