<?php

namespace Database\Seeders;

use App\Models\Event;
use App\Models\EventMedia;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DashboardSeeder extends Seeder
{
    public function run(): void
    {
        if (! Schema::hasTable('events') || ! Schema::hasTable('event_media')) {
            return;
        }

        if (Event::query()->exists()) {
            return;
        }

        $userId = User::query()->value('id');
        $eventSpecs = [
            [
                'name' => 'Arora Wedding',
                'type' => 'Wedding',
                'days_ago' => 2,
                'expiry_days' => 30,
                'hashtags' => ['wedding', 'classic', 'studio'],
                'media' => [3, 2, 1],
            ],
            [
                'name' => 'Maternity Studio',
                'type' => 'Maternity',
                'days_ago' => 6,
                'expiry_days' => 45,
                'hashtags' => ['maternity', 'studio'],
                'media' => [4, 0, 2],
            ],
            [
                'name' => 'Conference Summit',
                'type' => 'Conferences',
                'days_ago' => 9,
                'expiry_days' => 20,
                'hashtags' => ['conference', 'meetup'],
                'media' => [2, 1, 0],
            ],
        ];

        foreach ($eventSpecs as $index => $spec) {
            $eventDate = now()->subDays($spec['days_ago'])->toDateString();
            $expiryDate = now()->addDays($spec['expiry_days'])->toDateString();
            $appKey = (string) config('app.key');
            $guestPin = $appKey !== '' ? Crypt::encryptString('1234') : null;
            $adminPin = $appKey !== '' ? Crypt::encryptString('5678') : null;

            $event = Event::create([
                'name' => $spec['name'],
                'slug' => $this->uniqueSlug($spec['name']),
                'type' => $spec['type'],
                'details' => $spec['name'] . ' demo gallery with staged uploads.',
                'event_date' => $eventDate,
                'expiry_date' => $expiryDate,
                'notifications_enabled' => true,
                'guest_pin' => $guestPin,
                'admin_pin' => $adminPin,
                'share_token' => $this->uniqueToken(),
                'hashtags' => $spec['hashtags'],
                'is_active' => true,
                'created_by' => $userId,
            ]);

            $event->timestamps = false;
            $event->forceFill([
                'created_at' => now()->subDays($spec['days_ago']),
                'updated_at' => now()->subDays(max(0, $spec['days_ago'] - 1)),
            ])->save();

            $this->seedMedia($event, $spec['media']);
        }
    }

    private function seedMedia(Event $event, array $batches): void
    {
        $disk = Storage::disk('public');
        $basePath = 'demo/events/' . $event->id;
        $disk->makeDirectory($basePath);

        $statuses = ['ready', 'processing', 'pending'];
        $index = 1;

        foreach ($batches as $batchIndex => $count) {
            for ($i = 0; $i < $count; $i++) {
                $fileBase = Str::slug($event->name) . '-' . $index . '-' . Str::random(6);
                $originalPath = $basePath . '/' . $fileBase . '-original.svg';
                $optimizedPath = $basePath . '/' . $fileBase . '-optimized.svg';
                $thumbnailPath = $basePath . '/' . $fileBase . '-thumb.svg';

                $svg = $this->placeholderSvg($event->name, $index);
                $disk->put($originalPath, $svg);
                $disk->put($optimizedPath, $svg);
                $disk->put($thumbnailPath, $svg);

                $size = $disk->size($originalPath);

                $media = EventMedia::create([
                    'event_id' => $event->id,
                    'disk' => 'public',
                    'original_path' => $originalPath,
                    'optimized_path' => $optimizedPath,
                    'thumbnail_path' => $thumbnailPath,
                    'file_name' => $fileBase . '.svg',
                    'file_type' => 'image',
                    'mime_type' => 'image/svg+xml',
                    'size' => $size ?: 1200,
                    'width' => 1920,
                    'height' => 1080,
                    'is_featured' => $index === 1,
                    'is_cover' => $index === 1,
                    'status' => $statuses[$batchIndex] ?? 'ready',
                ]);

                $media->timestamps = false;
                $media->forceFill([
                    'created_at' => now()->subDays(max(0, $index - 1)),
                    'updated_at' => now()->subDays(max(0, $index - 1)),
                ])->save();

                $index++;
            }
        }
    }

    private function placeholderSvg(string $title, int $index): string
    {
        $safeTitle = htmlspecialchars($title, ENT_XML1 | ENT_QUOTES, 'UTF-8');

        return <<<SVG
<?xml version="1.0" encoding="UTF-8"?>
<svg xmlns="http://www.w3.org/2000/svg" width="1200" height="800" viewBox="0 0 1200 800">
  <defs>
    <linearGradient id="bg" x1="0" y1="0" x2="1" y2="1">
      <stop offset="0%" stop-color="#111827" />
      <stop offset="100%" stop-color="#374151" />
    </linearGradient>
  </defs>
  <rect width="1200" height="800" fill="url(#bg)" />
  <rect x="80" y="120" width="1040" height="560" rx="36" fill="#0f172a" opacity="0.7" />
  <text x="600" y="380" text-anchor="middle" font-family="Arial, sans-serif" font-size="46" fill="#f8fafc">
    {$safeTitle}
  </text>
  <text x="600" y="450" text-anchor="middle" font-family="Arial, sans-serif" font-size="28" fill="#cbd5f5">
    Demo Media {$index}
  </text>
</svg>
SVG;
    }

    private function uniqueSlug(string $name): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while (Event::query()->where('slug', $slug)->exists()) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function uniqueToken(): string
    {
        do {
            $token = Str::random(32);
        } while (Event::query()->where('share_token', $token)->exists());

        return $token;
    }
}
