import './bootstrap';
import Quill from 'quill';
import 'quill/dist/quill.snow.css';
import Alpine from 'alpinejs';
import intersect from '@alpinejs/intersect';
import collapse from '@alpinejs/collapse';

Alpine.plugin(intersect);
Alpine.plugin(collapse);

window.Alpine = Alpine;

Alpine.start();

const initAdminUI = () => {
    const root = document.documentElement;
    const themeStorageKey = 'admin-theme';

    const setTheme = (theme) => {
        root.classList.toggle('dark', theme === 'dark');
        root.dataset.theme = theme;
        localStorage.setItem(themeStorageKey, theme);
        document.querySelectorAll('[data-theme-toggle]').forEach((button) => {
            const isDark = theme === 'dark';
            button.setAttribute('aria-pressed', isDark ? 'true' : 'false');
            button.querySelectorAll('[data-theme-icon]').forEach((icon) => {
                const target = icon.getAttribute('data-theme-icon');
                icon.classList.toggle('hidden', target === (isDark ? 'sun' : 'moon'));
            });
        });
    };

    const storedTheme = localStorage.getItem(themeStorageKey);
    const prefersDark = window.matchMedia?.('(prefers-color-scheme: dark)').matches;
    setTheme(storedTheme ?? (prefersDark ? 'dark' : 'light'));

    document.querySelectorAll('[data-theme-toggle]').forEach((button) => {
        button.addEventListener('click', () => {
            const nextTheme = root.classList.contains('dark') ? 'light' : 'dark';
            setTheme(nextTheme);
        });
    });

    const initAdminColorThemes = () => {
        if (!document.querySelector('.admin-sidebar')) {
            return;
        }

        const colorThemeStorageKey = 'admin-color-theme';
        const availableThemes = [
            'default',
            'ocean',
            'forest',
            'sunset',
            'lavender',
            'desert',
            'aurora',
            'canyon',
            'mist',
            'ember',
            'moss',
            'cobalt',
            'rose',
            'slate',
            'ivory',
            'citrus',
            'mint',
            'plum',
            'sapphire',
            'clay',
            'dune',
            'lagoon',
            'coral',
            'copper',
            'storm',
            'glacier',
            'meadow',
            'orchid',
            'terracotta',
            'graphite',
        ];

        const applyColorTheme = (theme) => {
            const nextTheme = availableThemes.includes(theme) ? theme : 'default';
            root.setAttribute('data-admin-theme', nextTheme);
            localStorage.setItem(colorThemeStorageKey, nextTheme);

            document.querySelectorAll('[data-admin-theme-option]').forEach((button) => {
                const isActive = button.dataset.adminThemeOption === nextTheme;
                button.setAttribute('aria-pressed', isActive ? 'true' : 'false');
            });
        };

        const storedColorTheme = localStorage.getItem(colorThemeStorageKey);
        applyColorTheme(storedColorTheme || 'default');

        document.querySelectorAll('[data-admin-theme-option]').forEach((button) => {
            button.addEventListener('click', () => {
                applyColorTheme(button.dataset.adminThemeOption);
            });
        });
    };

    initAdminColorThemes();

    const updateFullscreenState = () => {
        const isFullscreen = Boolean(document.fullscreenElement);
        document.querySelectorAll('[data-fullscreen-toggle]').forEach((button) => {
            button.setAttribute('aria-pressed', isFullscreen ? 'true' : 'false');
            button.querySelectorAll('[data-fullscreen-icon]').forEach((icon) => {
                const target = icon.getAttribute('data-fullscreen-icon');
                icon.classList.toggle('hidden', target === (isFullscreen ? 'enter' : 'exit'));
            });
        });
    };

    updateFullscreenState();
    document.addEventListener('fullscreenchange', updateFullscreenState);
    document.querySelectorAll('[data-fullscreen-toggle]').forEach((button) => {
        button.addEventListener('click', () => {
            if (!document.fullscreenElement) {
                document.documentElement.requestFullscreen?.();
            } else {
                document.exitFullscreen?.();
            }
        });
    });

    const setSidebarOpen = (open) => {
        if (open) {
            root.setAttribute('data-sidebar-open', 'true');
        } else {
            root.removeAttribute('data-sidebar-open');
        }
    };

    document.querySelectorAll('[data-sidebar-toggle]').forEach((button) => {
        button.addEventListener('click', () => {
            const isOpen = root.getAttribute('data-sidebar-open') === 'true';
            setSidebarOpen(!isOpen);
        });
    });

    const sidebarOverlay = document.querySelector('[data-sidebar-overlay]');
    if (sidebarOverlay) {
        sidebarOverlay.addEventListener('click', () => setSidebarOpen(false));
    }

    // Sidebar collapse toggle (desktop) with localStorage persistence
    const collapseToggle = document.querySelector('[data-sidebar-collapse-toggle]');
    if (collapseToggle) {
        const isCollapsed = localStorage.getItem('sidebar-collapsed') === 'true';
        if (isCollapsed) {
            root.setAttribute('data-sidebar-collapsed', 'true');
        }

        collapseToggle.addEventListener('click', () => {
            const collapsed = root.getAttribute('data-sidebar-collapsed') === 'true';
            if (collapsed) {
                root.removeAttribute('data-sidebar-collapsed');
                localStorage.setItem('sidebar-collapsed', 'false');
            } else {
                root.setAttribute('data-sidebar-collapsed', 'true');
                localStorage.setItem('sidebar-collapsed', 'true');
            }
        });
    }

    const updatePasswordToggle = (button, input) => {
        const isVisible = input.type === 'text';
        button.setAttribute('aria-pressed', isVisible ? 'true' : 'false');
        button.querySelectorAll('[data-password-icon]').forEach((icon) => {
            const target = icon.getAttribute('data-password-icon');
            icon.classList.toggle('hidden', target === (isVisible ? 'show' : 'hide'));
        });
    };

    document.querySelectorAll('[data-password-toggle]').forEach((button) => {
        const targetId = button.getAttribute('data-target');
        const input = document.getElementById(targetId);
        if (!input) {
            return;
        }

        updatePasswordToggle(button, input);
        button.addEventListener('click', () => {
            input.type = input.type === 'password' ? 'text' : 'password';
            updatePasswordToggle(button, input);
        });
    });

    document.querySelectorAll('[data-bulk-scope]').forEach((scope) => {
        const items = Array.from(scope.querySelectorAll('[data-bulk-item]'));
        const master = scope.querySelector('[data-bulk-all]');
        const actions = Array.from(scope.querySelectorAll('[data-bulk-action]'));
        const counts = Array.from(scope.querySelectorAll('[data-bulk-count]'));
        const selectedLabels = Array.from(scope.querySelectorAll('[data-bulk-selected]'));

        // Select-all-pages elements
        const allPagesBanner = scope.querySelector('[data-bulk-all-pages-banner]');
        const allSelectedBanner = scope.querySelector('[data-bulk-all-selected-banner]');
        const selectAllPagesBtn = scope.querySelector('[data-bulk-select-all-pages]');
        const clearAllPagesBtn = scope.querySelector('[data-bulk-clear-all-pages]');
        const pageCountEls = scope.querySelectorAll('[data-bulk-page-count]');
        const totalItems = parseInt(scope.getAttribute('data-bulk-total') || '0', 10);
        let selectAllPages = false;

        if (!items.length) {
            return;
        }

        const updateBulkState = () => {
            const selectedCount = items.filter((item) => item.checked).length;
            const allOnPageSelected = selectedCount === items.length && items.length > 0;

            if (master) {
                master.checked = selectedCount === items.length;
                master.indeterminate = selectedCount > 0 && selectedCount < items.length;
            }

            const hasSelection = selectedCount > 0 || selectAllPages;
            actions.forEach((action) => {
                action.classList.toggle('hidden', !hasSelection);
                action.toggleAttribute('disabled', !hasSelection);
            });

            const displayCount = selectAllPages ? String(totalItems || 'All') : String(selectedCount);
            counts.forEach((count) => {
                count.textContent = displayCount;
            });

            selectedLabels.forEach((label) => {
                label.textContent = displayCount;
            });

            // Show/hide select-all-pages banners
            if (allPagesBanner) {
                allPagesBanner.classList.toggle('hidden', !allOnPageSelected || selectAllPages);
            }
            if (allSelectedBanner) {
                allSelectedBanner.classList.toggle('hidden', !selectAllPages);
            }

            pageCountEls.forEach((el) => {
                el.textContent = String(items.length);
            });
        };

        // Select all pages button
        if (selectAllPagesBtn) {
            selectAllPagesBtn.addEventListener('click', () => {
                selectAllPages = true;
                items.forEach((item) => { item.checked = true; });
                updateBulkState();
            });
        }

        // Clear all pages button
        if (clearAllPagesBtn) {
            clearAllPagesBtn.addEventListener('click', () => {
                selectAllPages = false;
                items.forEach((item) => { item.checked = false; });
                if (master) master.checked = false;
                updateBulkState();
            });
        }

        if (master) {
            master.addEventListener('change', () => {
                const shouldCheck = master.checked;
                items.forEach((item) => {
                    item.checked = shouldCheck;
                });
                if (!shouldCheck) selectAllPages = false;
                updateBulkState();
            });
        }

        items.forEach((item) => {
            item.addEventListener('change', () => {
                if (!item.checked) selectAllPages = false;
                updateBulkState();
            });
        });

        actions.forEach((action) => {
            action.addEventListener('click', () => {
                const selectedIds = items.filter((item) => item.checked).map((item) => item.value).filter(Boolean);
                const targetKey = action.getAttribute('data-bulk-target');
                const modalName = action.getAttribute('data-bulk-modal');

                // Modal is teleported to document.body by Alpine,
                // so search via data-modal-name attribute directly.
                const modalEl = modalName ? document.querySelector(`[data-modal-name="${modalName}"]`) : null;

                // FIX: Search modal first (teleported out of scope), then scope
                // excluding action buttons which also have data-bulk-target.
                let target = null;
                if (targetKey) {
                    target = (modalEl && modalEl.querySelector(`[data-bulk-target="${targetKey}"]`))
                        || scope.querySelector(`[data-bulk-target="${targetKey}"]:not([data-bulk-action])`);
                }
                if (!target) {
                    target = (modalEl && modalEl.querySelector('[data-bulk-target]'))
                        || scope.querySelector('[data-bulk-target]:not([data-bulk-action])');
                }

                if (target) {
                    target.innerHTML = '';

                    // If select-all-pages, inject flag for the controller
                    if (selectAllPages) {
                        const flag = document.createElement('input');
                        flag.type = 'hidden';
                        flag.name = 'select_all_pages';
                        flag.value = '1';
                        target.appendChild(flag);
                    }

                    selectedIds.forEach((id) => {
                        const input = document.createElement('input');
                        input.type = 'hidden';
                        input.name = 'ids[]';
                        input.value = id;
                        target.appendChild(input);
                    });
                }

                // Update selected count labels inside the modal
                if (modalEl) {
                    const countText = selectAllPages ? String(totalItems || 'all') : String(selectedIds.length);
                    modalEl.querySelectorAll('[data-bulk-selected]').forEach((label) => {
                        label.textContent = countText;
                    });
                }

                if (modalName) {
                    window.dispatchEvent(new CustomEvent('open-modal', { detail: modalName }));
                }
            });
        });

        updateBulkState();
    });

    document.querySelectorAll('[data-view-scope]').forEach((scope) => {
        const buttons = Array.from(scope.querySelectorAll('[data-view-toggle]'));
        if (!buttons.length) {
            return;
        }

        const storageKey = scope.getAttribute('data-view-storage') || 'admin-view-mode';
        const setView = (view) => {
            scope.setAttribute('data-view-mode', view);
            localStorage.setItem(storageKey, view);
            buttons.forEach((button) => {
                button.classList.toggle('btn-primary', button.dataset.viewToggle === view);
                button.classList.toggle('btn-secondary', button.dataset.viewToggle !== view);
            });
        };

        const storedView = localStorage.getItem(storageKey);
        setView(storedView || scope.getAttribute('data-view-mode') || 'list');

        buttons.forEach((button) => {
            button.addEventListener('click', () => {
                setView(button.dataset.viewToggle);
            });
        });
    });

    document.querySelectorAll('[data-tab-group]').forEach((group) => {
        const buttons = Array.from(group.querySelectorAll('[data-tab-button]'));
        const panels = Array.from(group.querySelectorAll('[data-tab-panel]'));

        if (!buttons.length || !panels.length) {
            return;
        }

        const storageKey = group.getAttribute('data-tab-storage') || 'admin-tab';
        const setTab = (tab) => {
            panels.forEach((panel) => {
                panel.classList.toggle('hidden', panel.dataset.tabPanel !== tab);
            });
            buttons.forEach((button) => {
                const isActive = button.dataset.tabButton === tab;
                button.classList.toggle('btn-primary', isActive);
                button.classList.toggle('btn-secondary', !isActive);
            });
            localStorage.setItem(storageKey, tab);
        };

        const initialTab = localStorage.getItem(storageKey) || buttons[0].dataset.tabButton;
        setTab(initialTab);

        buttons.forEach((button) => {
            button.addEventListener('click', () => {
                setTab(button.dataset.tabButton);
            });
        });
    });

    document.querySelectorAll('[data-location-scope]').forEach((scope) => {
        const countrySelect = scope.querySelector('[data-country-select]');
        const stateSelect = scope.querySelector('[data-state-select]');
        const citySelect = scope.querySelector('[data-city-select]');

        if (!countrySelect || !stateSelect) {
            return;
        }

        const filterOptions = (select, dataKey, value) => {
            Array.from(select.options).forEach((option) => {
                if (!option.value) {
                    option.hidden = false;
                    option.disabled = false;
                    return;
                }
                const matches = !value || option.dataset[dataKey] === value;
                option.hidden = !matches;
                option.disabled = !matches;
            });

            if (select.value && select.selectedOptions[0]?.hidden) {
                select.value = '';
            }
        };

        const syncStates = () => {
            filterOptions(stateSelect, 'countryId', countrySelect.value);
            if (citySelect) {
                filterOptions(citySelect, 'stateId', stateSelect.value);
            }
        };

        const syncCities = () => {
            if (citySelect) {
                filterOptions(citySelect, 'stateId', stateSelect.value);
            }
        };

        countrySelect.addEventListener('change', () => {
            stateSelect.value = '';
            if (citySelect) {
                citySelect.value = '';
            }
            syncStates();
        });

        stateSelect.addEventListener('change', () => {
            if (citySelect) {
                citySelect.value = '';
                syncCities();
            }
        });

        syncStates();
    });

    const fileUploaders = new WeakMap();
    const initOnce = (node, flag) => {
        if (!node) return false;
        if (node.dataset[flag] === 'true') {
            return false;
        }
        node.dataset[flag] = 'true';
        return true;
    };

    // --- Upload control state ---
    const uploadState = {
        abortController: null,
        xhr: null,
        paused: false,
        cancelled: false,
        reset() {
            this.paused = false;
            this.cancelled = false;
            if (this.abortController) {
                this.abortController.abort();
                this.abortController = null;
            }
            if (this.xhr) {
                this.xhr.abort();
                this.xhr = null;
            }
        },
        newAbort() {
            this.abortController = new AbortController();
            return this.abortController.signal;
        },
    };

    const ensureUploadToast = () => {
        let toast = document.querySelector('[data-upload-toast]');
        if (toast) return toast;

        toast = document.createElement('div');
        toast.dataset.uploadToast = 'true';
        toast.className = 'fixed bottom-6 right-6 z-[10000] w-80 max-w-[92vw]';
        toast.innerHTML = `
            <div class="panel p-4 space-y-3">
                <div class="flex items-start justify-between gap-3">
                    <div>
                        <p class="text-sm font-semibold text-ink" data-upload-title>Uploading...</p>
                        <p class="text-xs text-muted" data-upload-subtitle>Preparing upload</p>
                    </div>
                    <span class="h-2 w-2 rounded-full bg-brand" data-upload-dot></span>
                </div>
                <div class="w-full rounded-full bg-border/60">
                    <div class="h-1.5 rounded-full bg-brand transition-all" style="width: 12%;" data-upload-bar></div>
                </div>
                <div class="flex items-center justify-between">
                    <p class="text-xs text-muted" data-upload-footer>Working...</p>
                    <div class="flex gap-2" data-upload-controls>
                        <button type="button" data-upload-cancel class="text-xs font-medium text-danger hover:text-danger/80 transition-colors">Cancel</button>
                    </div>
                </div>
            </div>
        `;

        // Cancel button handler
        toast.querySelector('[data-upload-cancel]').addEventListener('click', () => {
            if (uploadState.cancelled) {
                // Already cancelled — act as "Resume": reload to restart
                window.location.reload();
                return;
            }
            uploadState.cancelled = true;
            uploadState.reset();
            updateUploadToast({
                title: 'Upload cancelled',
                subtitle: `Cancelled by user`,
                percent: 0,
                state: 'error',
                footer: 'Partially uploaded files are saved',
            });
            const cancelBtn = toast.querySelector('[data-upload-cancel]');
            if (cancelBtn) {
                cancelBtn.textContent = 'Resume';
                cancelBtn.classList.remove('text-danger', 'hover:text-danger/80');
                cancelBtn.classList.add('text-brand', 'hover:text-brand/80');
            }
        });

        document.body.appendChild(toast);
        return toast;
    };

    const updateUploadToast = ({ title, subtitle, percent, footer, state }) => {
        const toast = ensureUploadToast();
        const titleEl = toast.querySelector('[data-upload-title]');
        const subtitleEl = toast.querySelector('[data-upload-subtitle]');
        const footerEl = toast.querySelector('[data-upload-footer]');
        const bar = toast.querySelector('[data-upload-bar]');
        const dot = toast.querySelector('[data-upload-dot]');
        const controls = toast.querySelector('[data-upload-controls]');

        if (titleEl && title) titleEl.textContent = title;
        if (subtitleEl && subtitle) subtitleEl.textContent = subtitle;
        if (footerEl && footer) footerEl.textContent = footer;
        if (typeof percent === 'number' && bar) {
            bar.style.width = `${Math.min(100, Math.max(0, percent))}%`;
        }

        if (dot) {
            dot.className = 'h-2 w-2 rounded-full';
            if (state === 'error') {
                dot.classList.add('bg-danger');
            } else if (state === 'success') {
                dot.classList.add('bg-success');
                // Hide cancel when complete
                if (controls) controls.classList.add('hidden');
            } else {
                dot.classList.add('bg-brand');
                if (controls) controls.classList.remove('hidden');
            }
        }
    };

    const showProcessingOverlay = (message) => {
        const overlay = document.getElementById('media-processing-overlay');
        if (overlay) {
            overlay.classList.remove('hidden');
            overlay.classList.add('flex');
            document.body.style.overflow = 'hidden';
        }
        if (message) {
            const statusEl = document.getElementById('media-processing-status');
            if (statusEl) statusEl.textContent = message;
        }
    };

    const triggerProcessing = () => {
        if (typeof window.startMediaProcessing === 'function') {
            window.startMediaProcessing();
        } else {
            setTimeout(() => window.location.reload(), 1500);
        }
    };

    const hideUploadToast = (delay = 2000) => {
        const toast = document.querySelector('[data-upload-toast]');
        if (!toast) return;
        setTimeout(() => toast.remove(), delay);
    };

    const refreshMediaWrapper = async () => {
        const currentWrapper = document.querySelector('[data-media-wrapper]');
        if (!currentWrapper) return;

        try {
            const response = await fetch(window.location.href, {
                headers: { 'X-Requested-With': 'XMLHttpRequest' },
            });
            if (!response.ok) return;
            const html = await response.text();
            const temp = document.createElement('div');
            temp.innerHTML = html;
            const nextWrapper = temp.querySelector('[data-media-wrapper]');
            if (!nextWrapper) return;
            currentWrapper.replaceWith(nextWrapper);
            initFileUploaders();
            initChunkUploaders();
            initGoogleDriveImport();
        } catch (error) {
            // keep current view
        }
    };

    const escapeSelector = (value) => {
        if (window.CSS?.escape) {
            return CSS.escape(value);
        }
        return value.replace(/[^a-zA-Z0-9_-]/g, '\\$&');
    };

    const formatFileSize = (bytes) => {
        if (!Number.isFinite(bytes)) {
            return '0 KB';
        }
        if (bytes < 1024) {
            return `${bytes} B`;
        }
        const kb = bytes / 1024;
        if (kb < 1024) {
            return `${kb.toFixed(1)} KB`;
        }
        const mb = kb / 1024;
        return `${mb.toFixed(1)} MB`;
    };

    const setFileUploaderProgress = (input, fileName, percent, forceLabel) => {
        const entry = fileUploaders.get(input);
        if (!entry) {
            return;
        }

        const key = encodeURIComponent(fileName);
        const selector = `[data-file-key="${escapeSelector(key)}"]`;
        const item = entry.list.querySelector(selector);
        if (!item) {
            return;
        }

        const bar = item.querySelector('[data-file-progress]');
        const status = item.querySelector('[data-file-status]');

        if (bar) {
            bar.style.width = `${percent}%`;
        }

        if (status) {
            if (forceLabel) {
                status.textContent = forceLabel;
            } else if (percent >= 100) {
                status.textContent = 'Uploaded';
            } else {
                status.textContent = `${percent}%`;
            }
        }

        item.classList.toggle('is-uploading', percent > 0 && percent < 100);
        item.classList.toggle('is-complete', percent >= 100);
    };

    const initFileUploaders = () => {
        if (!document.querySelector('.admin-sidebar')) {
            return;
        }

        document.querySelectorAll('input[type="file"]').forEach((input) => {
            if (input.dataset.fileUploaderReady === 'true') {
                return;
            }

            const wrapper = document.createElement('div');
            wrapper.className = 'file-uploader';
            wrapper.dataset.fileUploader = 'true';

            const list = document.createElement('div');
            list.className = 'file-uploader-list hidden';

            const inputId = input.id || `file-${Math.random().toString(36).slice(2, 10)}`;
            input.id = inputId;
            input.classList.add('file-uploader-input');

            Array.from(input.classList).forEach((name) => {
                if (name.startsWith('mt-') || name.startsWith('mb-') || name.startsWith('my-')) {
                    wrapper.classList.add(name);
                }
            });

            const label = document.createElement('label');
            label.className = 'file-uploader-drop';
            label.setAttribute('for', inputId);
            label.innerHTML = `
                <span class="file-uploader-cloud">
                    <svg viewBox="0 0 24 24" aria-hidden="true" focusable="false">
                        <path d="M8 18H6.5a4.5 4.5 0 0 1 0-9 5.5 5.5 0 0 1 10.5 1.6A4 4 0 0 1 17.5 18H16" />
                        <path d="M12 8v7" />
                        <path d="M9 11l3-3 3 3" />
                    </svg>
                </span>
                <span class="file-uploader-title">${input.multiple ? 'Browse files to upload' : 'Browse file to upload'}</span>
                <span class="file-uploader-subtitle">or drag & drop</span>
            `;

            const parent = input.parentNode;
            if (!parent) {
                return;
            }

            parent.insertBefore(wrapper, input);
            wrapper.appendChild(input);
            wrapper.appendChild(label);
            wrapper.appendChild(list);

            const renderList = () => {
                const files = Array.from(input.files || []);
                list.innerHTML = '';

                if (!files.length) {
                    list.classList.add('hidden');
                    return;
                }

                const totalSize = files.reduce((sum, file) => sum + (file.size || 0), 0);
                const summary = document.createElement('div');
                summary.className = 'file-uploader-summary';
                summary.innerHTML = `
                    <div class="file-uploader-item">
                        <span class="file-uploader-file">
                            <svg viewBox="0 0 24 24" aria-hidden="true" focusable="false">
                                <path d="M6 3h8l4 4v14H6z" />
                                <path d="M14 3v5h5" />
                            </svg>
                        </span>
                        <div class="file-uploader-meta">
                            <span class="file-uploader-name">${files.length} file${files.length > 1 ? 's' : ''} selected</span>
                            <span class="file-uploader-info">${formatFileSize(totalSize)} total</span>
                        </div>
                        <span class="file-uploader-status" data-file-status>Ready</span>
                    </div>
                `;

                list.classList.remove('hidden');
                list.appendChild(summary);
            };

            const setDragState = (isActive) => {
                wrapper.classList.toggle('is-dragover', isActive);
            };

            wrapper.addEventListener('dragover', (event) => {
                event.preventDefault();
                setDragState(true);
            });

            wrapper.addEventListener('dragleave', () => {
                setDragState(false);
            });

            wrapper.addEventListener('drop', (event) => {
                event.preventDefault();
                setDragState(false);
                if (!event.dataTransfer?.files?.length) {
                    return;
                }
                const transfer = new DataTransfer();
                Array.from(event.dataTransfer.files).forEach((file) => transfer.items.add(file));
                input.files = transfer.files;
                input.dispatchEvent(new Event('change', { bubbles: true }));
            });

            input.addEventListener('change', renderList);
            renderList();

            fileUploaders.set(input, { list, renderList });
        });
    };

    initFileUploaders();

    const initChunkUploaders = () => {
        document.querySelectorAll('[data-chunk-upload-form]').forEach((form) => {
            if (!initOnce(form, 'chunkUploadInit')) {
                return;
            }

            const input = form.querySelector('[data-chunk-upload-input]');
            const toggle = form.querySelector('[data-chunk-toggle]');
            const progress = form.querySelector('[data-chunk-progress]');
            const endpoint = form.getAttribute('data-chunk-endpoint');
            const chunkSize = Number(form.getAttribute('data-chunk-size')) || 5242880;
            const submitButton = form.querySelector('button[type=\"submit\"]');

            if (!input || !toggle || !endpoint) {
                return;
            }

            const updateProgress = (text) => {
                if (progress) {
                    progress.textContent = text;
                }
            };

            const uploadFileInChunks = async (file, csrfToken) => {
                const totalChunks = Math.ceil(file.size / chunkSize);
                const uploadId = `${Date.now()}-${Math.random().toString(36).slice(2, 10)}`;

                for (let index = 0; index < totalChunks; index += 1) {
                    const start = index * chunkSize;
                    const chunk = file.slice(start, start + chunkSize);

                    // Skip SHA256 for single-chunk files (no integrity risk, saves ~50ms/file)
                    let hashHex = '';
                    if (crypto.subtle && totalChunks > 1) {
                        const chunkBuffer = await chunk.arrayBuffer();
                        const hashBuffer = await crypto.subtle.digest('SHA-256', chunkBuffer);
                        const hashArray = Array.from(new Uint8Array(hashBuffer));
                        hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
                    }

                    const formData = new FormData();
                    formData.append('upload_id', uploadId);
                    formData.append('chunk_index', String(index));
                    formData.append('total_chunks', String(totalChunks));
                    formData.append('file_name', file.name);
                    formData.append('file_type', file.type || 'application/octet-stream');
                    formData.append('file_size', String(file.size));
                    formData.append('chunk_checksum', hashHex);
                    formData.append('chunk', chunk, file.name);

                    const response = await fetch(endpoint, {
                        method: 'POST',
                        headers: { 'X-CSRF-TOKEN': csrfToken },
                        body: formData,
                    });

                    if (!response.ok) {
                        const error = await response.json().catch(() => ({}));
                        throw new Error(error.error || 'Chunk upload failed');
                    }

                    const percent = Math.round(((index + 1) / totalChunks) * 100);
                    setFileUploaderProgress(input, file.name, percent);
                }
            };

            form.addEventListener('submit', async (event) => {
                event.preventDefault();

                const files = Array.from(input.files || []);
                if (!files.length) {
                    updateProgress('Select at least one file.');
                    return;
                }

                const csrfToken = document.querySelector('meta[name=\"csrf-token\"]')?.getAttribute('content');
                if (!csrfToken) {
                    updateProgress('Missing CSRF token.');
                    return;
                }

                if (submitButton) {
                    submitButton.disabled = true;
                }

                const action = form.getAttribute('action') || window.location.href;
                const totalFiles = files.length;
                let uploadedCount = 0;
                let failedCount = 0;

                try {
                    // Reset upload state and create new abort controller
                    uploadState.cancelled = false;
                    uploadState.paused = false;
                    const signal = uploadState.newAbort();

                    // Smart auto-detection: use batch mode for small files (much faster)
                    const CHUNK_THRESHOLD = 2 * 1024 * 1024; // 2MB
                    const maxFileSize = Math.max(...files.map(f => f.size));
                    const useChunked = toggle.checked && maxFileSize >= CHUNK_THRESHOLD;

                    // Throttled UI update helper (max 5 updates/sec for large uploads)
                    let lastUIUpdate = 0;
                    const updateOverallStatus = (force = false) => {
                        const now = Date.now();
                        if (!force && now - lastUIUpdate < 200) return;
                        lastUIUpdate = now;
                        const percent = Math.round(((uploadedCount + failedCount) / totalFiles) * 100);
                        const statusText = `${uploadedCount}/${totalFiles} files uploaded` + (failedCount > 0 ? ` (${failedCount} failed)` : '');
                        updateProgress(statusText);
                        updateUploadToast({
                            title: 'Uploading media...',
                            subtitle: statusText,
                            percent,
                            footer: useChunked
                                ? `6 parallel uploads \u2022 Please keep this tab open`
                                : `5 parallel streams \u2022 Please keep this tab open`,
                        });
                    };

                    if (useChunked) {
                        // --- Chunked upload path (for large files >= 2MB) ---
                        const CONCURRENCY = 6;

                        updateOverallStatus(true);

                        let nextIndex = 0;
                        const worker = async () => {
                            while (nextIndex < files.length) {
                                if (uploadState.cancelled) break;
                                const fi = nextIndex++;
                                const file = files[fi];
                                setFileUploaderProgress(input, file.name, 0, 'Uploading');
                                try {
                                    await uploadFileInChunks(file, csrfToken);
                                    setFileUploaderProgress(input, file.name, 100);
                                    uploadedCount++;
                                } catch (err) {
                                    if (uploadState.cancelled) break;
                                    failedCount++;
                                    setFileUploaderProgress(input, file.name, 0, 'Failed');
                                }
                                updateOverallStatus();
                            }
                        };

                        await Promise.all(
                            Array.from({ length: Math.min(CONCURRENCY, files.length) }, () => worker())
                        );

                        if (uploadedCount === 0 && failedCount > 0) {
                            throw new Error(`All ${totalFiles} files failed to upload.`);
                        }
                    } else {
                        // --- Parallel batch upload path (for small files or non-chunked) ---
                        const BATCH_SIZE = 20; // PHP max_file_uploads defaults to 20
                        const PARALLEL_STREAMS = 5;
                        const batches = [];
                        for (let i = 0; i < files.length; i += BATCH_SIZE) {
                            batches.push(files.slice(i, i + BATCH_SIZE));
                        }
                        const totalBatches = batches.length;

                        updateOverallStatus(true);

                        let nextBatchIndex = 0;
                        const batchWorker = async () => {
                            while (nextBatchIndex < totalBatches) {
                                if (uploadState.cancelled) break;
                                const bi = nextBatchIndex++;
                                const batch = batches[bi];

                                const batchFormData = new FormData();
                                batchFormData.append('_token', csrfToken);
                                batch.forEach((file) => batchFormData.append('media[]', file));

                                try {
                                    const response = await fetch(action, {
                                        method: 'POST',
                                        headers: {
                                            'X-CSRF-TOKEN': csrfToken,
                                            Accept: 'application/json',
                                        },
                                        body: batchFormData,
                                        signal,
                                    });

                                    if (!response.ok) {
                                        failedCount += batch.length;
                                        batch.forEach((file) => setFileUploaderProgress(input, file.name, 0, 'Failed'));
                                    } else {
                                        uploadedCount += batch.length;
                                        batch.forEach((file) => setFileUploaderProgress(input, file.name, 100));
                                    }
                                } catch (batchError) {
                                    if (uploadState.cancelled) break;
                                    failedCount += batch.length;
                                    batch.forEach((file) => setFileUploaderProgress(input, file.name, 0, 'Failed'));
                                }
                                updateOverallStatus();
                            }
                        };

                        await Promise.all(
                            Array.from({ length: Math.min(PARALLEL_STREAMS, totalBatches) }, () => batchWorker())
                        );

                        if (uploadedCount === 0) {
                            throw new Error(`All ${totalFiles} files failed to upload.`);
                        }
                    }

                    // Force final UI update with accurate count
                    updateOverallStatus(true);

                    if (uploadState.cancelled) {
                        const cancelText = `${uploadedCount}/${totalFiles} uploaded before cancel`;
                        updateProgress(cancelText);
                        // Toast already updated by cancel handler
                        if (uploadedCount > 0) {
                            await refreshMediaWrapper();
                            hideUploadToast(0);
                            showProcessingOverlay(`${uploadedCount} files uploaded. Optimizing now...`);
                            triggerProcessing();
                        }
                    } else {
                        const doneText = `${uploadedCount}/${totalFiles} files uploaded` + (failedCount > 0 ? ` (${failedCount} failed)` : '');
                        updateProgress(doneText + ' \u2013 Processing...');
                        await refreshMediaWrapper();
                        hideUploadToast(0);
                        showProcessingOverlay(`${uploadedCount} files uploaded. Optimizing now...`);
                        triggerProcessing();
                    }
                } catch (error) {
                    if (uploadState.cancelled) {
                        const cancelText = `${uploadedCount}/${totalFiles} uploaded before cancel`;
                        updateProgress(cancelText);
                        if (uploadedCount > 0) {
                            await refreshMediaWrapper();
                            hideUploadToast(0);
                            showProcessingOverlay(`${uploadedCount} files uploaded. Optimizing now...`);
                            triggerProcessing();
                        }
                    } else {
                        const failText = uploadedCount > 0
                            ? `${uploadedCount}/${totalFiles} uploaded, then failed: ${error?.message || 'Unknown error'}`
                            : (error?.message || 'Upload failed. Please try again.');
                        updateProgress(failText);
                        updateUploadToast({
                            title: 'Upload failed',
                            subtitle: failText,
                            percent: 0,
                            state: 'error',
                            footer: uploadedCount > 0 ? `${uploadedCount} files saved` : 'No files were added',
                        });
                        if (uploadedCount > 0) {
                            hideUploadToast(0);
                            showProcessingOverlay(`${uploadedCount} files uploaded. Optimizing now...`);
                            triggerProcessing();
                        }
                    }
                    files.forEach((file) => {
                        setFileUploaderProgress(input, file.name, 0, 'Failed');
                    });
                } finally {
                    if (submitButton) {
                        submitButton.disabled = false;
                    }
                }
            });
        });
    };

    initChunkUploaders();

    // --- ZIP Upload Handler ---
    const zipForm = document.getElementById('zip-upload-form');
    if (zipForm) {
        zipForm.addEventListener('submit', async (event) => {
            event.preventDefault();

            const fileInput = zipForm.querySelector('input[type="file"]');
            const btn = document.getElementById('zip-upload-btn');
            const progressEl = document.getElementById('zip-upload-progress');
            const file = fileInput?.files?.[0];

            if (!file) {
                if (progressEl) progressEl.textContent = 'Please select a ZIP file.';
                return;
            }

            if (!file.name.toLowerCase().endsWith('.zip')) {
                if (progressEl) progressEl.textContent = 'Only .zip files are accepted.';
                return;
            }

            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
            if (!csrfToken) return;

            if (btn) btn.disabled = true;
            if (progressEl) progressEl.textContent = 'Uploading ZIP...';

            // Reset upload state for ZIP
            uploadState.cancelled = false;
            uploadState.paused = false;

            updateUploadToast({
                title: 'Uploading ZIP...',
                subtitle: `${(file.size / 1024 / 1024).toFixed(1)} MB`,
                percent: 0,
                footer: 'Please keep this tab open',
            });

            const formData = new FormData();
            formData.append('_token', csrfToken);
            formData.append('zip', file);
            const zipStartTime = performance.now();

            try {
                const xhr = new XMLHttpRequest();
                uploadState.xhr = xhr;
                const action = zipForm.getAttribute('action') || window.location.href;
                let uploadDoneTime = null;

                xhr.upload.addEventListener('progress', (e) => {
                    if (e.lengthComputable) {
                        const pct = Math.round((e.loaded / e.total) * 100);
                        if (pct >= 100) {
                            uploadDoneTime = performance.now();
                            const uploadSec = ((uploadDoneTime - zipStartTime) / 1000).toFixed(1);
                            if (progressEl) progressEl.textContent = 'Server processing ZIP...';
                            updateUploadToast({
                                title: 'Server processing...',
                                subtitle: `Upload done in ${uploadSec}s — extracting on server`,
                                percent: 100,
                                footer: 'Almost done, please wait',
                            });
                        } else {
                            if (progressEl) progressEl.textContent = `Uploading ZIP... ${pct}%`;
                            updateUploadToast({
                                title: 'Uploading ZIP...',
                                subtitle: `${pct}% uploaded`,
                                percent: pct,
                                footer: 'Please keep this tab open',
                            });
                        }
                    }
                });

                const result = await new Promise((resolve, reject) => {
                    xhr.onload = () => {
                        if (xhr.status >= 200 && xhr.status < 300) {
                            resolve(JSON.parse(xhr.responseText));
                        } else {
                            const err = JSON.parse(xhr.responseText || '{}');
                            reject(new Error(err.error || err.message || 'Upload failed'));
                        }
                    };
                    xhr.onerror = () => reject(new Error('Network error'));
                    xhr.onabort = () => reject(new Error('Upload cancelled'));
                    xhr.open('POST', action);
                    xhr.setRequestHeader('Accept', 'application/json');
                    xhr.setRequestHeader('X-CSRF-TOKEN', csrfToken);
                    xhr.send(formData);
                });

                const totalSec = ((performance.now() - zipStartTime) / 1000).toFixed(1);
                const count = result.count || 0;
                if (progressEl) progressEl.textContent = `Done! ${count} photos extracted. Processing now...`;
                hideUploadToast(0);
                showProcessingOverlay(`${count} photos extracted. Optimizing now...`);
                triggerProcessing();
            } catch (err) {
                if (uploadState.cancelled) {
                    if (progressEl) progressEl.textContent = 'Upload cancelled.';
                    // Toast already updated by cancel handler
                } else {
                    if (progressEl) progressEl.textContent = err.message || 'Upload failed.';
                    updateUploadToast({
                        title: 'ZIP upload failed',
                        subtitle: err.message || 'Unknown error',
                        percent: 100,
                        state: 'error',
                    });
                }
            } finally {
                uploadState.xhr = null;
                if (btn) btn.disabled = false;
            }
        });
    }

    const initGoogleDriveImport = () => {
        document.querySelectorAll('[data-google-drive-form]').forEach((form) => {
            if (!initOnce(form, 'googleDriveInit')) {
                return;
            }

            const errorEl = form.querySelector('[data-google-drive-error]');
            const action = form.getAttribute('action') || window.location.href;
            const modalName = form.getAttribute('data-modal-name') || null;

            form.addEventListener('submit', async (event) => {
                event.preventDefault();
                if (errorEl) {
                    errorEl.classList.add('hidden');
                    errorEl.textContent = '';
                }

                const formData = new FormData(form);
                const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
                if (!csrfToken) {
                    if (errorEl) {
                        errorEl.textContent = 'Missing CSRF token.';
                        errorEl.classList.remove('hidden');
                    }
                    return;
                }

                try {
                    updateUploadToast({
                        title: 'Importing from Google Drive...',
                        subtitle: 'Fetching files',
                        percent: 10,
                    });

                    const response = await fetch(action, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': csrfToken,
                            Accept: 'application/json',
                        },
                        body: formData,
                    });

                    if (!response.ok) {
                        const data = await response.json().catch(() => ({}));
                        const message = Array.isArray(data.error) ? data.error.join(', ') : (data.error || data.message || 'Import failed.');
                        throw new Error(message);
                    }

                    updateUploadToast({
                        title: 'Import complete',
                        subtitle: 'Refreshing media list',
                        percent: 100,
                        state: 'success',
                    });

                    if (modalName) {
                        window.dispatchEvent(new CustomEvent('close-modal', { detail: modalName }));
                    }

                    await refreshMediaWrapper();
                    hideUploadToast();
                } catch (error) {
                    const message = error?.message || 'Import failed.';
                    updateUploadToast({
                        title: 'Import failed',
                        subtitle: message,
                        percent: 0,
                        state: 'error',
                    });
                    if (errorEl) {
                        errorEl.textContent = message;
                        errorEl.classList.remove('hidden');
                    }
                }
            });
        });
    };

    initGoogleDriveImport();

    document.querySelectorAll('[data-feature-list]').forEach((scope) => {
        const items = scope.querySelector('[data-feature-items]');
        const addButton = scope.querySelector('[data-feature-add]');
        const template = scope.querySelector('[data-feature-template]');

        if (!items || !addButton || !template) {
            return;
        }

        const addItem = () => {
            const fragment = template.content.cloneNode(true);
            items.appendChild(fragment);
        };

        addButton.addEventListener('click', () => {
            addItem();
        });

        items.addEventListener('click', (event) => {
            const button = event.target.closest('[data-feature-remove]');
            if (!button) {
                return;
            }

            const item = button.closest('[data-feature-item]');
            if (!item) {
                return;
            }

            const currentItems = items.querySelectorAll('[data-feature-item]');
            if (currentItems.length > 1) {
                item.remove();
            } else {
                const input = item.querySelector('input');
                if (input) {
                    input.value = '';
                }
            }
        });
    });

    const initNotifications = () => {
        document.querySelectorAll('[data-notification-scope]').forEach((scope) => {
            const endpoint = scope.getAttribute('data-notification-endpoint');
            const channelName = scope.getAttribute('data-notification-channel');
            const list = scope.querySelector('[data-notification-list]');
            const emptyState = scope.querySelector('[data-notification-empty]');
            const countBadge = scope.querySelector('[data-notification-count]');
            const toggle = scope.querySelector('[data-notification-toggle]');

            if (!endpoint || !list) {
                return;
            }

            let latestId = 0;
            let unread = 0;
            let loaded = false;
            const maxItems = 6;

            const updateBadge = () => {
                if (!countBadge) {
                    return;
                }
                if (unread > 0) {
                    countBadge.textContent = unread > 9 ? '9+' : String(unread);
                    countBadge.classList.remove('hidden');
                    countBadge.classList.add('flex');
                } else {
                    countBadge.textContent = '';
                    countBadge.classList.add('hidden');
                    countBadge.classList.remove('flex');
                }
            };

            const buildItem = (item) => {
                const wrapper = document.createElement('div');
                wrapper.className = 'border-t border-border/60 px-4 py-3 text-sm first:border-t-0';
                wrapper.dataset.notificationId = String(item.id ?? '');

                const header = document.createElement('div');
                header.className = 'flex items-center justify-between gap-3';

                const title = document.createElement('p');
                title.className = 'font-semibold text-ink';
                title.textContent = item.summary || 'Activity update';

                const time = document.createElement('span');
                time.className = 'text-xs text-muted';
                time.textContent = item.logged_at_label || '';

                header.appendChild(title);
                header.appendChild(time);

                const meta = document.createElement('p');
                meta.className = 'mt-1 text-xs text-muted';
                meta.textContent = item.user ? `By ${item.user}` : 'System update';

                wrapper.appendChild(header);
                wrapper.appendChild(meta);

                return wrapper;
            };

            const renderList = (items) => {
                list.innerHTML = '';
                const validItems = items.filter((item) => item && item.id);
                if (!validItems.length) {
                    if (emptyState) {
                        emptyState.classList.remove('hidden');
                        list.appendChild(emptyState);
                    }
                    return;
                }

                if (emptyState) {
                    emptyState.classList.add('hidden');
                }

                validItems.forEach((item) => {
                    list.appendChild(buildItem(item));
                });
            };

            const prependItems = (items) => {
                const existingIds = new Set(
                    Array.from(list.querySelectorAll('[data-notification-id]')).map((node) => Number(node.dataset.notificationId)),
                );
                const freshItems = items.filter((item) => item && item.id && !existingIds.has(item.id));
                if (!freshItems.length) {
                    return;
                }

                if (emptyState) {
                    emptyState.classList.add('hidden');
                }

                [...freshItems].reverse().forEach((item) => {
                    list.insertBefore(buildItem(item), list.firstChild);
                });

                const nodes = Array.from(list.querySelectorAll('[data-notification-id]'));
                if (nodes.length > maxItems) {
                    nodes.slice(maxItems).forEach((node) => node.remove());
                }
            };

            const fetchNotifications = async (after) => {
                const url = new URL(endpoint, window.location.origin);
                if (after) {
                    url.searchParams.set('after', String(after));
                }
                url.searchParams.set('limit', String(maxItems));

                try {
                    const response = await fetch(url.toString(), { headers: { Accept: 'application/json' } });
                    if (!response.ok) {
                        return null;
                    }
                    return await response.json();
                } catch (error) {
                    return null;
                }
            };

            const loadInitial = async () => {
                const data = await fetchNotifications(0);
                if (!data) {
                    return;
                }
                latestId = Number(data.latest_id || latestId);
                loaded = true;
                renderList(Array.isArray(data.items) ? data.items : []);
            };

            const pollUpdates = async () => {
                if (!loaded) {
                    await loadInitial();
                    return;
                }

                if (!latestId) {
                    await loadInitial();
                    return;
                }

                const data = await fetchNotifications(latestId);
                if (!data || !Array.isArray(data.items) || data.items.length === 0) {
                    return;
                }

                latestId = Number(data.latest_id || latestId);
                unread += data.items.length;
                updateBadge();
                prependItems(data.items);
            };

            if (toggle) {
                toggle.addEventListener('click', () => {
                    unread = 0;
                    updateBadge();
                });
            }

            loadInitial();
            updateBadge();
            setInterval(pollUpdates, 25000);

            if (window.Echo && channelName) {
                window.Echo.private(channelName).listen('.admin.notification', (payload) => {
                    if (!payload || !payload.item) {
                        return;
                    }

                    latestId = Number(payload.latest_id || payload.item.id || latestId);
                    unread += 1;
                    updateBadge();
                    prependItems([payload.item]);
                });
            }
        });
    };

    initNotifications();

    const initAdminSearch = () => {
        document.querySelectorAll('[data-admin-search]').forEach((scope) => {
            const endpoint = scope.getAttribute('data-admin-search-endpoint');
            const input = scope.querySelector('[data-admin-search-input]');
            const results = scope.querySelector('[data-admin-search-results]');

            if (!endpoint || !input || !results) {
                return;
            }

            let timer = null;
            let lastQuery = '';
            let latestItems = [];
            let abortController = null;

            const closeResults = () => {
                results.classList.add('hidden');
                results.innerHTML = '';
                latestItems = [];
            };

            const openResults = () => {
                results.classList.remove('hidden');
            };

            const renderResults = (items) => {
                results.innerHTML = '';
                latestItems = items;

                if (!items.length) {
                    const empty = document.createElement('div');
                    empty.className = 'px-4 py-2 text-sm text-muted';
                    empty.textContent = 'No results found.';
                    results.appendChild(empty);
                    openResults();
                    return;
                }

                const groups = items.reduce((acc, item) => {
                    const key = item.type || 'Results';
                    acc[key] = acc[key] || [];
                    acc[key].push(item);
                    return acc;
                }, {});

                Object.entries(groups).forEach(([group, groupItems]) => {
                    const header = document.createElement('p');
                    header.className = 'px-4 pt-3 text-xs font-semibold uppercase tracking-widest text-muted';
                    header.textContent = group;
                    results.appendChild(header);

                    groupItems.forEach((item) => {
                        const link = document.createElement('a');
                        link.href = item.url;
                        link.className = 'flex flex-col gap-1 px-4 py-2 text-sm text-ink/80 transition hover:bg-surface-2/80 hover:text-ink';

                        const title = document.createElement('span');
                        title.className = 'font-semibold text-ink';
                        title.textContent = item.title || 'Result';
                        link.appendChild(title);

                        if (item.subtitle) {
                            const meta = document.createElement('span');
                            meta.className = 'text-xs text-muted';
                            meta.textContent = item.subtitle;
                            link.appendChild(meta);
                        }

                        results.appendChild(link);
                    });
                });

                openResults();
            };

            const fetchResults = async (query) => {
                if (abortController) {
                    abortController.abort();
                }
                abortController = new AbortController();

                const url = new URL(endpoint, window.location.origin);
                url.searchParams.set('q', query);

                try {
                    const response = await fetch(url.toString(), {
                        headers: { Accept: 'application/json' },
                        signal: abortController.signal,
                    });
                    if (!response.ok) {
                        return null;
                    }
                    return await response.json();
                } catch (error) {
                    if (error?.name === 'AbortError') {
                        return null;
                    }
                    return null;
                }
            };

            const scheduleSearch = () => {
                const query = input.value.trim();
                if (query.length < 2) {
                    lastQuery = query;
                    closeResults();
                    return;
                }

                if (query === lastQuery) {
                    return;
                }

                lastQuery = query;

                if (timer) {
                    clearTimeout(timer);
                }

                timer = setTimeout(async () => {
                    const data = await fetchResults(query);
                    if (!data || input.value.trim() !== query) {
                        return;
                    }
                    renderResults(Array.isArray(data.items) ? data.items : []);
                }, 250);
            };

            input.addEventListener('input', scheduleSearch);
            input.addEventListener('focus', scheduleSearch);
            input.addEventListener('keydown', (event) => {
                if (event.key === 'Escape') {
                    closeResults();
                    input.blur();
                    return;
                }
                if (event.key === 'Enter' && latestItems.length) {
                    event.preventDefault();
                    window.location.assign(latestItems[0].url);
                }
            });

            document.addEventListener('click', (event) => {
                if (!scope.contains(event.target)) {
                    closeResults();
                }
            });
        });
    };

    initAdminSearch();

    const initSwitchLabels = () => {
        const pairs = new Map([
            ['Enabled', 'Disabled'],
            ['Disabled', 'Enabled'],
            ['Active', 'Inactive'],
            ['Inactive', 'Active'],
            ['Verified', 'Pending'],
            ['Pending', 'Verified'],
            ['On', 'Off'],
            ['Off', 'On'],
            ['Yes', 'No'],
            ['No', 'Yes'],
        ]);

        document.querySelectorAll('input.peer.sr-only[type="checkbox"]').forEach((input) => {
            const label = input.closest('label');
            const spanChildren = label ? Array.from(label.children).filter((child) => child.tagName === 'SPAN') : [];
            const textSpan = spanChildren.at(-1) || null;
            const updateLabel = () => {
                if (!textSpan) {
                    return;
                }

                const current = textSpan.dataset.switchOn || textSpan.textContent.trim();
                const alternate = textSpan.dataset.switchOff || pairs.get(current);
                if (!alternate) {
                    return;
                }

                if (!textSpan.dataset.switchOn || !textSpan.dataset.switchOff) {
                    const onText = input.checked ? current : alternate;
                    const offText = input.checked ? alternate : current;
                    textSpan.dataset.switchOn = onText;
                    textSpan.dataset.switchOff = offText;
                }

                textSpan.textContent = input.checked ? textSpan.dataset.switchOn : textSpan.dataset.switchOff;
            };

            updateLabel();
            input.addEventListener('change', () => {
                updateLabel();
            });
        });
    };

    initSwitchLabels();

    const initRequiredStars = () => {
        const escapeId = (value) => {
            if (window.CSS?.escape) {
                return CSS.escape(value);
            }
            return value.replace(/[^a-zA-Z0-9_-]/g, '\\$&');
        };

        const findLabel = (field) => {
            const id = field.getAttribute('id');
            if (id) {
                const byFor = document.querySelector(`label[for="${escapeId(id)}"]`);
                if (byFor) {
                    return byFor;
                }
            }

            const wrappedLabel = field.closest('label');
            if (wrappedLabel) {
                return wrappedLabel;
            }

            const container = field.closest('div');
            if (!container) {
                return null;
            }

            const directLabel = Array.from(container.children).find((child) => child.tagName === 'LABEL');
            if (directLabel) {
                return directLabel;
            }

            const parent = container.parentElement;
            if (!parent) {
                return null;
            }

            return Array.from(parent.children).find((child) => child.tagName === 'LABEL') || null;
        };

        const addStar = (label) => {
            if (label.querySelector('[data-required-star]')) {
                return;
            }

            const star = document.createElement('span');
            star.textContent = ' *';
            star.className = 'text-danger';
            star.setAttribute('data-required-star', 'true');
            label.appendChild(star);
        };

        document.querySelectorAll('input[required]:not([type="hidden"]), select[required], textarea[required]').forEach((field) => {
            const label = findLabel(field);
            if (!label) {
                return;
            }

            addStar(label);
        });
    };

    initRequiredStars();

    const initPhoneInputs = () => {
        document.querySelectorAll('[data-phone-input]').forEach((input) => {
            const sanitize = () => {
                let value = input.value.replace(/[^0-9+]/g, '');
                value = value.replace(/(?!^)\+/g, '');
                if (value.length > 15) {
                    value = value.slice(0, 15);
                }
                input.value = value;
            };

            input.addEventListener('input', sanitize);
            input.addEventListener('blur', sanitize);
            sanitize();
        });
    };

    initPhoneInputs();

    const initWysiwyg = () => {
        const isAdminShell = Boolean(document.querySelector('.admin-sidebar'));
        if (!isAdminShell) {
            return;
        }

        const textareas = Array.from(document.querySelectorAll('textarea:not([data-no-wysiwyg])'));
        if (!textareas.length) {
            return;
        }

        textareas.forEach((textarea) => {
            if (textarea.dataset.wysiwygReady === 'true') {
                return;
            }

            const shell = document.createElement('div');
            shell.className = 'quill-shell';

            const editor = document.createElement('div');
            editor.className = 'quill-editor';

            shell.appendChild(editor);
            textarea.insertAdjacentElement('afterend', shell);

            const quill = new Quill(editor, {
                theme: 'snow',
                placeholder: textarea.getAttribute('placeholder') || '',
                modules: {
                    toolbar: [
                        ['bold', 'italic', 'underline'],
                        [{ list: 'ordered' }, { list: 'bullet' }],
                        ['link'],
                        ['clean'],
                    ],
                },
            });

            if (textarea.value.trim()) {
                quill.clipboard.dangerouslyPasteHTML(textarea.value);
            }

            const syncValue = () => {
                const text = quill.getText().trim();
                textarea.value = text.length ? quill.root.innerHTML : '';
            };

            quill.on('text-change', syncValue);
            textarea.form?.addEventListener('submit', syncValue);
            syncValue();

            textarea.hidden = true;
            textarea.setAttribute('aria-hidden', 'true');
            textarea.dataset.wysiwygReady = 'true';
        });
    };

    initWysiwyg();

    const initPinGenerators = () => {
        document.querySelectorAll('[data-pin-generate]').forEach((button) => {
            const targetId = button.getAttribute('data-target');
            const input = targetId ? document.getElementById(targetId) : null;
            if (!input) {
                return;
            }

            const length = Number(button.getAttribute('data-length')) || 6;

            button.addEventListener('click', () => {
                const pin = Array.from({ length }, () => Math.floor(Math.random() * 10)).join('');
                input.value = pin;
                input.dispatchEvent(new Event('input', { bubbles: true }));
                input.dispatchEvent(new Event('change', { bubbles: true }));
                input.focus();
            });
        });
    };

    initPinGenerators();

    const initCloudStorageSettings = () => {
        const providerSelect = document.querySelector('[data-cloud-storage-provider]');
        if (!providerSelect) {
            return;
        }

        const panels = Array.from(document.querySelectorAll('[data-provider-panel]'));
        const updatePanels = () => {
            const selected = providerSelect.value;
            panels.forEach((panel) => {
                panel.classList.toggle('hidden', panel.dataset.providerPanel !== selected);
            });
        };

        providerSelect.addEventListener('change', updatePanels);
        updatePanels();
    };

    initCloudStorageSettings();
};

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initAdminUI);
} else {
    initAdminUI();
}
