@php
    $brandName = trim((string) ($siteSettings['site_name'] ?? ''));
    $brandName = $brandName !== '' ? $brandName : config('app.name', 'SnapNest');
    $adminLogo = $siteSettings['admin_logo'] ?? null;
@endphp
<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}" class="h-full bg-slate-50">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>{{ $event->name }} - Design - {{ $brandName }}</title>
    <!-- Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&family=Outfit:wght@400;500;600;700&family=Poppins:wght@400;500;600;700&family=Playfair+Display:wght@400;700&family=Montserrat:wght@400;600&family=Lora:wght@400;700&display=swap" rel="stylesheet">
    
    @vite(['resources/css/app.css', 'resources/js/app.js'])
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <style>
        [x-cloak] { display: none !important; }
        .no-scrollbar::-webkit-scrollbar { display: none; }
        .no-scrollbar { -ms-overflow-style: none; scrollbar-width: none; }
        /* Range Slider Styling */
        input[type=range] {
            -webkit-appearance: none;
            width: 100%;
            background: transparent;
            cursor: pointer;
        }
        input[type=range]::-webkit-slider-thumb {
            -webkit-appearance: none;
            height: 18px;
            width: 18px;
            border-radius: 50%;
            background: #2563EB;
            border: 3px solid #fff;
            box-shadow: 0 1px 3px rgba(0,0,0,0.2);
            margin-top: -7px;
        }
        input[type=range]::-webkit-slider-runnable-track {
            width: 100%;
            height: 4px;
            background: #E2E8F0;
            border-radius: 2px;
        }
        .canvas-bg {
            background-image: radial-gradient(#cbd5e1 1px, transparent 1px);
            background-size: 24px 24px;
        }
    </style>
</head>
<body class="h-full overflow-hidden flex flex-col font-sans text-slate-800" 
      x-data="designApp()" x-init="init()">
      
    <!-- Header -->
    <header class="h-16 bg-white border-b flex items-center justify-between px-6 z-20 shrink-0 shadow-sm relative">
        <div class="flex items-center gap-4">
             <a href="{{ route('admin.events.show', $event) }}" class="p-2 hover:bg-slate-100 rounded-lg transition-colors">
                 <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><line x1="19" y1="12" x2="5" y2="12"/><polyline points="12 19 5 12 12 5"/></svg>
             </a>
             <div class="h-6 w-px bg-slate-200 mx-1"></div>
             <h1 class="font-bold text-slate-800 tracking-tight text-lg">{{ $event->name }}</h1>
             <span class="bg-blue-50 text-blue-600 text-[10px] font-black uppercase tracking-widest px-2 py-0.5 rounded-md">{{ __('Live Editor') }}</span>
        </div>
        
        <div class="flex items-center gap-4">
             <!-- Viewport Toggle (Desktop / Mobile) -->
             <div class="flex gap-1 bg-slate-100 p-1 rounded-xl">
                 <button @click="viewMode = 'desktop'" class="flex items-center gap-1.5 px-3 py-1.5 rounded-lg text-xs font-bold uppercase tracking-wider transition-all" :class="viewMode === 'desktop' ? 'bg-white text-blue-600 shadow-sm' : 'text-slate-400 hover:text-slate-600'">
                     <svg xmlns="http://www.w3.org/2000/svg" width="15" height="15" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="2" y="3" width="20" height="14" rx="2" ry="2"/><line x1="8" y1="21" x2="16" y2="21"/><line x1="12" y1="17" x2="12" y2="21"/></svg>
                     {{ __('Desktop') }}
                 </button>
                 <button @click="viewMode = 'mobile'" class="flex items-center gap-1.5 px-3 py-1.5 rounded-lg text-xs font-bold uppercase tracking-wider transition-all" :class="viewMode === 'mobile' ? 'bg-white text-blue-600 shadow-sm' : 'text-slate-400 hover:text-slate-600'">
                     <svg xmlns="http://www.w3.org/2000/svg" width="15" height="15" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="5" y="2" width="14" height="20" rx="3" ry="3"/><line x1="12" y1="18" x2="12.01" y2="18"/></svg>
                     {{ __('Mobile') }}
                 </button>
             </div>
             <div class="h-6 w-px bg-slate-200"></div>
             <button @click="save()" class="flex items-center gap-2 px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-sm font-bold transition-all shadow-md active:scale-95"
                :class="{'opacity-75 cursor-wait': saving}" :disabled="saving">
                 <svg x-show="!saving" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M20 6 9 17l-5-5"/></svg>
                 <svg x-show="saving" class="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>
                 <span x-text="saving ? 'Saving...' : 'Apply Changes'"></span>
             </button>
        </div>
    </header>

    <!-- Main Content Area -->
    <div class="flex-1 flex overflow-hidden">

        <!-- Left Configuration Sidebar -->
        <aside class="w-[420px] bg-white z-20 flex shrink-0 border-r shadow-2xl relative transition-all duration-300 ease-in-out">
             
             <!-- Navigation Strip -->
             <div class="w-20 bg-slate-50 border-r flex flex-col items-center py-8 gap-8 z-30">
                 <button @click="activeTab = 'typography'" class="group flex flex-col items-center gap-2 relative w-full px-2" :class="activeTab === 'typography' ? 'text-blue-600' : 'text-slate-400 hover:text-slate-600'">
                     <div class="p-3 rounded-2xl transition-all" :class="activeTab === 'typography' ? 'bg-blue-100' : 'group-hover:bg-slate-200'">
                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M4 7V4h16v3"/><path d="M9 20h6"/><path d="M12 4v16"/></svg>
                     </div>
                     <span class="text-[10px] font-bold uppercase tracking-tight">{{ __('Fonts') }}</span>
                     <div class="absolute right-0 top-1/2 -translate-y-1/2 w-1 h-8 bg-blue-600 rounded-l-full" x-show="activeTab === 'typography'"></div>
                 </button>

                 <button @click="activeTab = 'grid'" class="group flex flex-col items-center gap-2 relative w-full px-2" :class="activeTab === 'grid' ? 'text-blue-600' : 'text-slate-400 hover:text-slate-600'">
                     <div class="p-3 rounded-2xl transition-all" :class="activeTab === 'grid' ? 'bg-blue-100' : 'group-hover:bg-slate-200'">
                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="3" width="7" height="7"/><rect x="14" y="3" width="7" height="7"/><rect x="14" y="14" width="7" height="7"/><rect x="3" y="14" width="7" height="7"/></svg>
                     </div>
                     <span class="text-[10px] font-bold uppercase tracking-tight">{{ __('Layout') }}</span>
                     <div class="absolute right-0 top-1/2 -translate-y-1/2 w-1 h-8 bg-blue-600 rounded-l-full" x-show="activeTab === 'grid'"></div>
                 </button>

                 <button @click="activeTab = 'colors'" class="group flex flex-col items-center gap-2 relative w-full px-2" :class="activeTab === 'colors' ? 'text-blue-600' : 'text-slate-400 hover:text-slate-600'">
                     <div class="p-3 rounded-2xl transition-all" :class="activeTab === 'colors' ? 'bg-blue-100' : 'group-hover:bg-slate-200'">
                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 2.69l5.66 5.66a8 8 0 1 1-11.31 0z"/></svg>
                     </div>
                     <span class="text-[10px] font-bold uppercase tracking-tight">{{ __('Colors') }}</span>
                     <div class="absolute right-0 top-1/2 -translate-y-1/2 w-1 h-8 bg-blue-600 rounded-l-full" x-show="activeTab === 'colors'"></div>
                 </button>

                 <button @click="activeTab = 'templates'" class="group flex flex-col items-center gap-2 relative w-full px-2" :class="activeTab === 'templates' ? 'text-blue-600' : 'text-slate-400 hover:text-slate-600'">
                     <div class="p-3 rounded-2xl transition-all" :class="activeTab === 'templates' ? 'bg-blue-100' : 'group-hover:bg-slate-200'">
                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="3" width="7" height="7"/><rect x="14" y="3" width="7" height="7"/><rect x="3" y="14" width="7" height="7"/><rect x="14" y="14" width="7" height="7"/></svg>
                     </div>
                     <span class="text-[10px] font-bold uppercase tracking-tight">{{ __('Templates') }}</span>
                     <div class="absolute right-0 top-1/2 -translate-y-1/2 w-1 h-8 bg-blue-600 rounded-l-full" x-show="activeTab === 'templates'"></div>
                 </button>

                 <button @click="activeTab = 'cover'" class="group flex flex-col items-center gap-2 relative w-full px-2" :class="activeTab === 'cover' ? 'text-blue-600' : 'text-slate-400 hover:text-slate-600'">
                     <div class="p-3 rounded-2xl transition-all" :class="activeTab === 'cover' ? 'bg-blue-100' : 'group-hover:bg-slate-200'">
                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>
                     </div>
                     <span class="text-[10px] font-bold uppercase tracking-tight">{{ __('Cover') }}</span>
                     <div class="absolute right-0 top-1/2 -translate-y-1/2 w-1 h-8 bg-blue-600 rounded-l-full" x-show="activeTab === 'cover'"></div>
                 </button>
             </div>

             <!-- Tab Content Area -->
             <div class="flex-1 flex flex-col overflow-hidden bg-white">
                 <div class="h-16 border-b flex items-center px-8 font-bold text-slate-800 bg-white shrink-0 capitalize text-lg" x-text="activeTabLabel"></div>
                 
                 <div class="flex-1 overflow-y-auto p-8 space-y-10">
                     
                     <!-- TYPOGRAPHY -->
                     <div x-show="activeTab === 'typography'" x-transition class="space-y-8">
                         <div>
                             <label class="block text-xs font-bold text-slate-400 uppercase tracking-widest mb-4">{{ __('Title Font') }}</label>
                             <div class="relative">
                                 <select x-model="settings.font_family" class="w-full pl-4 pr-10 py-4 bg-slate-50 border-2 border-slate-100 rounded-2xl appearance-none font-medium text-slate-700 outline-none focus:border-blue-500 transition-all cursor-pointer">
                                     <template x-for="font in fonts" :key="font">
                                         <option :value="font" x-text="font" :style="{ fontFamily: font }"></option>
                                     </template>
                                 </select>
                                 <div class="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none">
                                     <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2.5" viewBox="0 0 24 24"><path d="m6 9 6 6 6-6"/></svg>
                                 </div>
                             </div>
                         </div>

                         <div>
                             <div class="flex justify-between mb-4">
                                <label class="text-sm font-bold text-slate-700 uppercase tracking-wider">{{ __('Font Size') }}</label>
                                <span class="bg-slate-100 px-2 py-0.5 rounded text-xs font-bold text-slate-600" x-text="settings.font_size + 'px'"></span>
                             </div>
                             <input type="range" min="16" max="64" step="2" x-model="settings.font_size">
                         </div>
                     </div>

                     <!-- GRID SETTINGS -->
                     <div x-show="activeTab === 'grid'" x-transition class="space-y-12">
                         <!-- Hero Style -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Header Display') }}</p>
                             <div class="grid grid-cols-3 gap-3">
                                 <button @click="settings.hero_style = 'standard'" 
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.hero_style === 'standard' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><path d="M3 9h18"/></svg>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Full') }}</span>
                                 </button>
                                 <button @click="settings.hero_style = 'split'"
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.hero_style === 'split' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><path d="M12 3v18"/></svg>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Split') }}</span>
                                 </button>
                                 <button @click="settings.hero_style = 'minimal'"
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.hero_style === 'minimal' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="8" rx="2"/></svg>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Slim') }}</span>
                                 </button>
                             </div>
                         </div>

                         <!-- Grid Spacing & Layout -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Media Grid') }}</p>
                             <div class="grid grid-cols-3 gap-3 mb-6">
                                 <button @click="settings.grid_layout = 'masonry'" 
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.grid_layout === 'masonry' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <div class="flex gap-1 items-end h-5">
                                         <div class="w-1.5 h-5 bg-current opacity-40 rounded-sm"></div>
                                         <div class="w-1.5 h-3 bg-current opacity-40 rounded-sm"></div>
                                         <div class="w-1.5 h-4 bg-current opacity-40 rounded-sm"></div>
                                     </div>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Masonry') }}</span>
                                 </button>
                                 <button @click="settings.grid_layout = 'square'"
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.grid_layout === 'square' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <div class="grid grid-cols-2 gap-1 h-5">
                                         <div class="w-1.5 h-1.5 bg-current opacity-40 rounded-sm"></div>
                                         <div class="w-1.5 h-1.5 bg-current opacity-40 rounded-sm"></div>
                                     </div>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Square') }}</span>
                                 </button>
                                 <button @click="settings.grid_layout = 'collage'"
                                    class="p-4 border-2 rounded-2xl flex flex-col items-center gap-2 transition-all active:scale-95"
                                    :class="settings.grid_layout === 'collage' ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                     <div class="grid grid-cols-2 gap-1 h-5">
                                         <div class="w-4 h-4 bg-current opacity-40 rounded-sm col-span-2"></div>
                                     </div>
                                     <span class="text-[10px] font-black uppercase tracking-widest">{{ __('Collage') }}</span>
                                 </button>
                             </div>
                             
                             <div class="space-y-6">
                                 <div>
                                     <div class="flex justify-between mb-3 text-[10px] font-black uppercase text-slate-400">
                                         <span>{{ __('Spacing') }}</span>
                                         <span class="text-blue-600" x-text="settings.grid_gap + 'px'"></span>
                                     </div>
                                     <input type="range" min="0" max="64" step="4" x-model="settings.grid_gap">
                                 </div>
                                 <div>
                                     <div class="flex justify-between mb-3 text-[10px] font-black uppercase text-slate-400">
                                         <span>{{ __('Corners') }}</span>
                                         <span class="text-blue-600" x-text="settings.image_radius + 'px'"></span>
                                     </div>
                                     <input type="range" min="0" max="48" step="4" x-model="settings.image_radius">
                                 </div>
                             </div>
                         </div>

                         <!-- Styling & Patterns -->
                         <div class="space-y-8">
                             <div>
                                 <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Aesthetics') }}</p>
                                 <div class="grid grid-cols-5 gap-2">
                                     <template x-for="style in ['default', 'elevated', 'glass', 'minimal', 'bordered']" :key="style">
                                         <button @click="settings.card_style = style" 
                                              class="p-2 border-2 rounded-xl text-[8px] font-black uppercase tracking-widest transition-all h-12 flex items-center justify-center text-center leading-tight"
                                              :class="settings.card_style === style ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400'">
                                             <span x-text="style"></span>
                                         </button>
                                     </template>
                                 </div>
                             </div>
                             <div>
                                 <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Background Pattern') }}</p>
                                 <div class="flex gap-4">
                                     <button @click="settings.pattern = 'none'" class="flex-1 p-3 border-2 rounded-2xl transition-all" :class="settings.pattern === 'none' ? 'border-blue-600 bg-blue-50' : 'border-slate-100 bg-slate-50'">
                                         <div class="w-full h-8 bg-white border border-slate-200 rounded-lg"></div>
                                         <span class="text-[9px] font-black mt-2 block text-slate-400 uppercase tracking-widest">{{ __('None') }}</span>
                                     </button>
                                     <button @click="settings.pattern = 'dots'" class="flex-1 p-3 border-2 rounded-2xl transition-all" :class="settings.pattern === 'dots' ? 'border-blue-600 bg-blue-50' : 'border-slate-100 bg-slate-50'">
                                         <div class="w-full h-8 bg-white border border-slate-200 rounded-lg bg-dot-grid"></div>
                                         <span class="text-[9px] font-black mt-2 block text-slate-400 uppercase tracking-widest">{{ __('Dots') }}</span>
                                     </button>
                                     <button @click="settings.pattern = 'lines'" class="flex-1 p-3 border-2 rounded-2xl transition-all" :class="settings.pattern === 'lines' ? 'border-blue-600 bg-blue-50' : 'border-slate-100 bg-slate-50'">
                                         <div class="w-full h-8 bg-white border border-slate-200 rounded-lg bg-line-grid"></div>
                                         <span class="text-[9px] font-black mt-2 block text-slate-400 uppercase tracking-widest">{{ __('Grid') }}</span>
                                     </button>
                                 </div>
                             </div>
                         </div>

                         <!-- Tab Navigation Style -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Tab Style') }}</p>
                             <div class="grid grid-cols-4 gap-2">
                                 <template x-for="style in [
                                     {id:'underline', label:'Underline'},
                                     {id:'thick-underline', label:'Thick Line'},
                                     {id:'pill', label:'Pill'},
                                     {id:'pill-outline', label:'Pill Border'},
                                     {id:'block', label:'Block'},
                                     {id:'rounded-block', label:'Round Block'},
                                     {id:'soft-fill', label:'Soft Fill'},
                                     {id:'dot', label:'Dot'},
                                     {id:'double-line', label:'Double'},
                                     {id:'dash', label:'Dashed'},
                                     {id:'glow', label:'Glow'},
                                     {id:'lifted', label:'Lifted'},
                                     {id:'gradient-fill', label:'Gradient'},
                                     {id:'bracket', label:'Bracket'},
                                     {id:'highlight', label:'Highlight'},
                                     {id:'slide-up', label:'Slide Up'},
                                     {id:'boxed', label:'Boxed'},
                                     {id:'rounded-box', label:'Round Box'},
                                     {id:'fade-bg', label:'Fade'},
                                     {id:'minimal', label:'Minimal'}
                                 ]" :key="style.id">
                                     <button @click="settings.tab_style = style.id"
                                          class="p-2 border-2 rounded-xl text-[8px] font-black uppercase tracking-widest transition-all h-10 flex items-center justify-center text-center leading-tight"
                                          :class="settings.tab_style === style.id ? 'border-blue-600 bg-blue-50 text-blue-700' : 'border-slate-100 bg-slate-50 text-slate-400 hover:border-slate-200'">
                                         <span x-text="style.label"></span>
                                     </button>
                                 </template>
                             </div>
                         </div>
                     </div>

                     <!-- COLORS -->
                     <div x-show="activeTab === 'colors'" x-transition class="space-y-10">

                         <!-- Live Preview Strip -->
                         <div class="rounded-2xl overflow-hidden border-2 border-slate-100 shadow-sm">
                             <div class="h-10 flex items-center px-4 gap-2" :style="{ backgroundColor: settings.navbar_bg_color }">
                                 <div class="w-3 h-3 rounded-sm" :style="{ backgroundColor: settings.primary_color }"></div>
                                 <div class="h-1.5 w-12 rounded-full opacity-70" :style="{ backgroundColor: settings.navbar_text_color }"></div>
                                 <div class="flex-1"></div>
                                 <div class="h-1.5 w-8 rounded-full opacity-50" :style="{ backgroundColor: settings.navbar_text_color }"></div>
                                 <div class="h-1.5 w-8 rounded-full opacity-50" :style="{ backgroundColor: settings.navbar_text_color }"></div>
                             </div>
                             <div class="h-16 flex items-center justify-center" :style="{ background: settings.hero_gradient }">
                                 <div class="h-2 w-24 rounded-full opacity-60" :style="{ backgroundColor: settings.text_color }"></div>
                             </div>
                             <div class="p-3 flex gap-2" :style="{ backgroundColor: settings.background_color }">
                                 <div class="flex-1 h-10 rounded-lg" :style="{ backgroundColor: settings.surface_color, border: '1px solid ' + settings.primary_color + '15' }"></div>
                                 <div class="flex-1 h-10 rounded-lg" :style="{ backgroundColor: settings.surface_color, border: '1px solid ' + settings.primary_color + '15' }"></div>
                                 <div class="flex-1 h-10 rounded-lg" :style="{ backgroundColor: settings.surface_color, border: '1px solid ' + settings.primary_color + '15' }"></div>
                             </div>
                             <div class="h-8 flex items-center justify-center" :style="{ backgroundColor: settings.background_color }">
                                 <div class="px-4 py-1 rounded-full text-[8px] font-bold text-white" :style="{ backgroundColor: settings.primary_color }">{{ __('Button') }}</div>
                             </div>
                         </div>

                         <!-- Quick Palettes -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Quick Palettes') }}</p>
                             <div class="grid grid-cols-5 gap-2">
                                 <button @click="settings.primary_color='#2563EB'; settings.background_color='#FFFFFF'; settings.text_color='#0F172A'; settings.surface_color='#FFFFFF'; settings.navbar_bg_color='#FFFFFF'; settings.navbar_text_color='#1E293B'"
                                     class="group flex flex-col items-center gap-1.5 p-2 rounded-xl border-2 transition-all hover:border-blue-300" :class="settings.primary_color==='#2563EB' && settings.background_color==='#FFFFFF' ? 'border-blue-600 bg-blue-50' : 'border-slate-100'">
                                     <div class="flex gap-0.5"><div class="w-3 h-3 rounded-full bg-[#2563EB]"></div><div class="w-3 h-3 rounded-full bg-white border border-slate-200"></div><div class="w-3 h-3 rounded-full bg-[#0F172A]"></div></div>
                                     <span class="text-[8px] font-bold text-slate-500 uppercase">{{ __('Default') }}</span>
                                 </button>
                                 <button @click="settings.primary_color='#DC2626'; settings.background_color='#FFFFFF'; settings.text_color='#0F172A'; settings.surface_color='#FFFFFF'; settings.navbar_bg_color='#FFFFFF'; settings.navbar_text_color='#1E293B'"
                                     class="group flex flex-col items-center gap-1.5 p-2 rounded-xl border-2 transition-all hover:border-red-300" :class="settings.primary_color==='#DC2626' ? 'border-red-500 bg-red-50' : 'border-slate-100'">
                                     <div class="flex gap-0.5"><div class="w-3 h-3 rounded-full bg-[#DC2626]"></div><div class="w-3 h-3 rounded-full bg-white border border-slate-200"></div><div class="w-3 h-3 rounded-full bg-[#0F172A]"></div></div>
                                     <span class="text-[8px] font-bold text-slate-500 uppercase">{{ __('Bold') }}</span>
                                 </button>
                                 <button @click="settings.primary_color='#B76E79'; settings.background_color='#FFF8F6'; settings.text_color='#3D2C2C'; settings.surface_color='#FFFFFF'; settings.navbar_bg_color='#FFFFFF'; settings.navbar_text_color='#4A3333'"
                                     class="group flex flex-col items-center gap-1.5 p-2 rounded-xl border-2 transition-all hover:border-pink-300" :class="settings.primary_color==='#B76E79' ? 'border-pink-400 bg-pink-50' : 'border-slate-100'">
                                     <div class="flex gap-0.5"><div class="w-3 h-3 rounded-full bg-[#B76E79]"></div><div class="w-3 h-3 rounded-full bg-[#FFF8F6] border border-slate-200"></div><div class="w-3 h-3 rounded-full bg-[#3D2C2C]"></div></div>
                                     <span class="text-[8px] font-bold text-slate-500 uppercase">{{ __('Rose') }}</span>
                                 </button>
                                 <button @click="settings.primary_color='#D4AF37'; settings.background_color='#0A0A14'; settings.text_color='#F0E6D3'; settings.surface_color='#141425'; settings.navbar_bg_color='#0F0F1A'; settings.navbar_text_color='#E8DCC8'"
                                     class="group flex flex-col items-center gap-1.5 p-2 rounded-xl border-2 transition-all hover:border-yellow-400" :class="settings.primary_color==='#D4AF37' && settings.background_color==='#0A0A14' ? 'border-yellow-500 bg-yellow-50' : 'border-slate-100'">
                                     <div class="flex gap-0.5"><div class="w-3 h-3 rounded-full bg-[#D4AF37]"></div><div class="w-3 h-3 rounded-full bg-[#0A0A14] border border-slate-600"></div><div class="w-3 h-3 rounded-full bg-[#F0E6D3]"></div></div>
                                     <span class="text-[8px] font-bold text-slate-500 uppercase">{{ __('Luxury') }}</span>
                                 </button>
                                 <button @click="settings.primary_color='#059669'; settings.background_color='#F0FDF4'; settings.text_color='#022C22'; settings.surface_color='#FFFFFF'; settings.navbar_bg_color='#FFFFFF'; settings.navbar_text_color='#064E3B'"
                                     class="group flex flex-col items-center gap-1.5 p-2 rounded-xl border-2 transition-all hover:border-emerald-300" :class="settings.primary_color==='#059669' ? 'border-emerald-500 bg-emerald-50' : 'border-slate-100'">
                                     <div class="flex gap-0.5"><div class="w-3 h-3 rounded-full bg-[#059669]"></div><div class="w-3 h-3 rounded-full bg-[#F0FDF4] border border-slate-200"></div><div class="w-3 h-3 rounded-full bg-[#022C22]"></div></div>
                                     <span class="text-[8px] font-bold text-slate-500 uppercase">{{ __('Nature') }}</span>
                                 </button>
                             </div>
                         </div>

                         <!-- Brand & Accent -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Brand & Accent') }}</p>
                             <div class="space-y-3">
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.primary_color }">
                                         <input type="color" x-model="settings.primary_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Primary Color') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Buttons, links, active tabs') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.primary_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.text_color }">
                                         <input type="color" x-model="settings.text_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Text Color') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Headings & body text') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.text_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                             </div>
                         </div>

                         <!-- Page & Surface -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Page & Surface') }}</p>
                             <div class="space-y-3">
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.background_color }">
                                         <input type="color" x-model="settings.background_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Page Background') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Gallery page background') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.background_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.surface_color }">
                                         <input type="color" x-model="settings.surface_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Card Surface') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Photo cards & panels') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.surface_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                             </div>
                         </div>

                         <!-- Navigation -->
                         <div>
                             <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-4">{{ __('Navigation Bar') }}</p>
                             <div class="space-y-3">
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.navbar_bg_color }">
                                         <input type="color" x-model="settings.navbar_bg_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Nav Background') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Top bar fill color') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.navbar_bg_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                                 <div class="flex items-center gap-3 p-3 bg-slate-50 rounded-2xl border border-slate-100">
                                     <label class="relative w-10 h-10 rounded-xl cursor-pointer overflow-hidden shrink-0 shadow-inner border border-slate-200" :style="{ backgroundColor: settings.navbar_text_color }">
                                         <input type="color" x-model="settings.navbar_text_color" class="absolute inset-0 opacity-0 w-full h-full cursor-pointer">
                                     </label>
                                     <div class="flex-1 min-w-0">
                                         <p class="text-sm font-bold text-slate-800">{{ __('Nav Text & Icons') }}</p>
                                         <p class="text-[10px] text-slate-400 uppercase tracking-wider">{{ __('Brand name & nav links') }}</p>
                                     </div>
                                     <input type="text" x-model="settings.navbar_text_color" class="w-20 h-8 text-[11px] font-mono font-bold text-center text-slate-600 bg-white border border-slate-200 rounded-lg uppercase">
                                 </div>
                             </div>
                         </div>

                     </div>

                    <!-- TEMPLATES -->
                    <div x-show="activeTab === 'templates'" x-transition class="space-y-8">
                        <div class="space-y-4">
                            <div>
                                <p class="text-[10px] font-black uppercase tracking-[0.2em] text-blue-600 mb-1">{{ __('Theme Presets') }}</p>
                                <p class="text-sm font-semibold text-slate-700">{{ __('Template Library') }}</p>
                                <p class="text-xs text-slate-500 mt-1">{{ __('Pick an occasion-ready template or reset to defaults.') }}</p>
                            </div>
                            <div class="grid grid-cols-2 gap-3">
                                <button type="button" class="h-12 flex items-center justify-center rounded-2xl border-2 border-slate-100 text-slate-600 text-xs font-black uppercase tracking-widest hover:bg-slate-50 hover:border-slate-200 transition-all active:scale-95" @click="resetTemplate()">{{ __('Reset to Default') }}</button>
                                <button type="button" class="h-12 flex items-center justify-center rounded-2xl bg-blue-600 text-white text-xs font-black uppercase tracking-widest hover:bg-blue-700 shadow-lg shadow-blue-100 active:scale-95 transition-all" @click="templateModalOpen = true">{{ __('Choose Template') }}</button>
                            </div>
                        </div>

                        <div class="border border-slate-100 rounded-2xl bg-slate-50/70 overflow-hidden">
                            <!-- Mini preview of applied template -->
                            <div class="h-20 relative" :style="{ background: selectedTemplate ? selectedTemplate.hero_gradient : 'linear-gradient(135deg,#F1F5F9,#E2E8F0)' }">
                                <div class="absolute bottom-2 left-3 right-3 flex gap-1">
                                    <template x-for="n in 4" :key="n">
                                        <div class="flex-1 h-4 rounded-sm" :style="{ backgroundColor: selectedTemplate ? selectedTemplate.primary+'20' : '#CBD5E1' }"></div>
                                    </template>
                                </div>
                            </div>
                            <div class="p-4 flex items-center gap-3">
                                <div class="min-w-0 flex-1">
                                    <p class="text-sm font-bold text-slate-800" x-text="selectedTemplate ? selectedTemplate.name : 'No template selected'"></p>
                                    <p class="text-xs text-slate-500 mt-0.5" x-text="selectedTemplate ? selectedTemplate.tagline : 'Pick a template to instantly style your gallery'"></p>
                                </div>
                                <div x-show="selectedTemplate" class="flex gap-1 shrink-0">
                                    <div class="w-4 h-4 rounded-full border border-white shadow-sm" :style="{ backgroundColor: selectedTemplate?.primary }"></div>
                                    <div class="w-4 h-4 rounded-full border border-slate-200 shadow-sm" :style="{ backgroundColor: selectedTemplate?.background }"></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- COVER IMAGE -->
                    <div x-show="activeTab === 'cover'" x-transition class="space-y-6">
                          <label class="block text-xs font-bold text-slate-400 uppercase tracking-widest mb-4">{{ __('Event Cover') }}</label>
                          <div class="border-2 border-dashed border-slate-200 rounded-2xl p-10 text-center hover:bg-slate-50 hover:border-blue-300 transition-all cursor-pointer relative group" @click="$refs.coverInput.click()">
                              <div class="relative mx-auto h-48 w-full max-w-sm rounded-xl overflow-hidden shadow-lg group-hover:scale-[1.02] transition-transform">
                                  <img :src="coverPreviewUrl" class="w-full h-full object-cover" x-show="coverPreviewUrl">
                                  <div class="absolute inset-0 bg-slate-900/40 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity">
                                      <svg xmlns="http://www.w3.org/2000/svg" class="h-10 w-10 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 9a2 2 0 012-2h.93a2 2 0 001.664-.89l.812-1.22A2 2 0 0110.07 4h3.86a2 2 0 011.664.89l.812 1.22A2 2 0 0018.07 7H19a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9z" /><circle cx="12" cy="13" r="3" /></svg>
                                  </div>
                              </div>
                              <p class="text-sm font-bold text-slate-600 mt-6 tracking-wide">{{ __('CLICK TO REPLACE COVER') }}</p>
                              <p class="text-xs text-slate-400 mt-1 uppercase font-medium">{{ __('JPG, PNG, WEBP (MAX 10MB)') }}</p>
                              <input x-ref="coverInput" type="file" class="hidden" @change="handleCoverUpload">
                          </div>
                     </div>

                 </div>
             </div>
        </aside>

        <!-- Canvas / Preview Area -->
        <main class="flex-1 canvas-bg flex flex-col relative overflow-hidden transition-all bg-slate-50">
             <!-- Iframe Wrapper -->
             <div class="flex-1 flex items-center justify-center p-8 lg:p-12 overflow-hidden">
                 <div class="transition-all duration-700 ease-[cubic-bezier(0.34,1.56,0.64,1)] shrink-0"
                      :class="viewMode === 'mobile' ? 'w-[375px] h-[812px] rounded-[3rem] ring-[12px] ring-slate-900 shadow-[0_0_100px_rgba(0,0,0,0.3)] overflow-hidden relative scale-[0.75] flex flex-col' : 'w-full h-full rounded-2xl shadow-2xl border border-slate-200 overflow-hidden'">
                      
                      <!-- Mobile Status Bar -->
                      <div x-show="viewMode === 'mobile'" class="h-12 shrink-0 bg-white z-20 relative flex justify-between items-end px-8 pb-3">
                          <span class="text-xs font-black text-slate-900">9:41</span>
                          <div class="flex gap-1.5 items-center">
                              <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M12 21c-4.97 0-9-4.03-9-9s4.03-9 9-9 9 4.03 9 9-4.03 9-9 9z"/></svg>
                              <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6zm-2 12H4V8h16v10z"/></svg>
                          </div>
                      </div>

                      <!-- Notch for mobile -->
                      <div x-show="viewMode === 'mobile'" class="absolute top-0 left-1/2 -translate-x-1/2 w-40 h-7 bg-slate-900 rounded-b-3xl z-30"></div>

                      <iframe x-ref="previewFrame"
                        class="w-full bg-white transition-opacity duration-300"
                        :class="(viewMode === 'mobile' ? 'flex-1 min-h-0' : 'h-full') + ' ' + (iframeLoading ? 'opacity-0' : 'opacity-100')"
                        :src="'{{ route('events.share', ['event' => $event->slug, 'token' => $event->share_token]) }}' + '&design_preview=true'"
                        @load="iframeLoading = false; sendSettings()"></iframe>
                 </div>
             </div>
        </main>
    </div>

    <script>
        axios.defaults.headers.common['X-CSRF-TOKEN'] = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

        function designApp() {
            return {
                viewMode: 'desktop',
                activeTab: 'typography',
                iframeLoading: true,
                saving: false,
                showToast: false,
                coverPreviewUrl: @json($event->coverMedia ? $event->coverMedia->preview_url : asset('images/default-cover.jpg')),
                
                fonts: ['Outfit', 'Inter', 'Poppins', 'Playfair Display', 'Montserrat', 'Lora', 'Roboto', 'Open Sans'],
                templateModalOpen: false,
                templateSearch: '',
                templateSort: 'newest',
                templateStyle: 'All',
                templateCategory: 'All',
                templatePreview: null,
                templates: [],
                templateStyles: ['All', 'Modern', 'Minimal', 'Elegant', 'Bold', 'Glass', 'Dark', 'Retro', 'Editorial', 'Playful'],
                templateCategories: ['All', 'Wedding', 'Portrait', 'Corporate', 'Sports', 'Party', 'Travel', 'Birthday', 'Baby Shower', 'Graduation', 'Festival', 'General'],

                settings: {
                    font_family: 'Outfit',
                    font_size: 32,
                    grid_gap: 16,
                    image_radius: 0,
                    grid_layout: 'masonry',
                    hero_style: 'standard',
                    card_style: 'default',
                    pattern: 'none',
                    primary_color: '#2563EB',
                    background_color: '#FFFFFF',
                    navbar_bg_color: '#FFFFFF',
                    navbar_text_color: '#1E293B',
                    surface_color: '#FFFFFF',
                    text_color: '#0F172A',
                    hero_gradient: 'linear-gradient(135deg,#EEF2FF 0%,#E0F2FE 50%,#FDF2F8 100%)',
                    card_shadow: '0 12px 40px rgba(15,23,42,0.08)',
                    card_radius: 18,
                    button_radius: 14,
                    tab_style: 'underline',
                    template_key: null,
                    template_name: null,
                    template_category: null,
                    template_style: null,
                    ...(@json($event->design_settings) || {})
                },
                selectedTemplate: null,
                
                get activeTabLabel() {
                    const map = {
                        'typography': 'Typography & Style',
                        'grid': 'Layout & Spacing',
                        'colors': 'Brand Colors',
                        'templates': 'Templates',
                        'cover': 'Header Image'
                    };
                    return map[this.activeTab] || this.activeTab;
                },
                
                init() {
                   this.buildTemplateLibrary();
                   this.syncSelectedTemplate();
                   this.$watch('settings', (val) => {
                       this.sendSettings();
                   }, { deep: true });
                },

                buildTemplateLibrary() {
                    this.templates = [
                        // ── Wedding ──────────────────────────────────
                        { id:'tpl-1', name:'Eternal Bloom', tagline:'Soft rose gold elegance', category:'Wedding', style:'Elegant',
                          font_family:'Playfair Display', grid:'masonry', hero_style:'split', card_style:'glass', pattern:'none',
                          grid_gap:20, radius:24, primary:'#B76E79', background:'#FFF8F6', nav_bg:'#FFFFFF', nav_text:'#4A3333',
                          surface:'#FFFFFF', text:'#3D2C2C', hero_gradient:'linear-gradient(135deg,#FFEEF0 0%,#FFF0F5 50%,#FCE4EC 100%)',
                          shadow:'0 20px 60px rgba(183,110,121,0.12)', font_size:36, card_radius:24, button_radius:20 },
                        { id:'tpl-2', name:'Ivory & Lace', tagline:'Timeless minimal charm', category:'Wedding', style:'Minimal',
                          font_family:'Lora', grid:'square', hero_style:'minimal', card_style:'minimal', pattern:'none',
                          grid_gap:40, radius:0, primary:'#8B7355', background:'#FFFEF9', nav_bg:'#FFFDF5', nav_text:'#5C4A32',
                          surface:'#FFFEF9', text:'#3E3225', hero_gradient:'linear-gradient(180deg,#FFF8E7 0%,#FFFEF9 100%)',
                          shadow:'none', font_size:40, card_radius:0, button_radius:4 },
                        { id:'tpl-3', name:'Midnight Vows', tagline:'Dark luxury with gold accents', category:'Wedding', style:'Dark',
                          font_family:'Playfair Display', grid:'collage', hero_style:'standard', card_style:'elevated', pattern:'dots',
                          grid_gap:12, radius:16, primary:'#D4AF37', background:'#0A0A14', nav_bg:'#0F0F1A', nav_text:'#E8DCC8',
                          surface:'#141425', text:'#F0E6D3', hero_gradient:'linear-gradient(135deg,#1A1A2E 0%,#16213E 50%,#0F3460 100%)',
                          shadow:'0 25px 50px rgba(212,175,55,0.15)', font_size:38, card_radius:16, button_radius:8 },
                        { id:'tpl-4', name:'Garden Romance', tagline:'Sage green botanical feel', category:'Wedding', style:'Elegant',
                          font_family:'Lora', grid:'masonry', hero_style:'split', card_style:'default', pattern:'none',
                          grid_gap:24, radius:20, primary:'#6B8F71', background:'#F7FAF7', nav_bg:'#FFFFFF', nav_text:'#2D3B2D',
                          surface:'#FFFFFF', text:'#1E2B1E', hero_gradient:'linear-gradient(135deg,#E8F5E9 0%,#F1F8E9 50%,#F7FAF7 100%)',
                          shadow:'0 16px 40px rgba(107,143,113,0.1)', font_size:34, card_radius:20, button_radius:16 },

                        // ── Portrait ─────────────────────────────────
                        { id:'tpl-5', name:'Studio Noir', tagline:'Pure black & white drama', category:'Portrait', style:'Minimal',
                          font_family:'Inter', grid:'square', hero_style:'minimal', card_style:'bordered', pattern:'none',
                          grid_gap:4, radius:0, primary:'#000000', background:'#FFFFFF', nav_bg:'#000000', nav_text:'#FFFFFF',
                          surface:'#FFFFFF', text:'#000000', hero_gradient:'linear-gradient(180deg,#F5F5F5 0%,#FFFFFF 100%)',
                          shadow:'none', font_size:48, card_radius:0, button_radius:0 },
                        { id:'tpl-6', name:'Golden Hour', tagline:'Warm amber sunset tones', category:'Portrait', style:'Elegant',
                          font_family:'Lora', grid:'masonry', hero_style:'standard', card_style:'elevated', pattern:'none',
                          grid_gap:16, radius:12, primary:'#C8852D', background:'#FFFBF0', nav_bg:'#FFFAF0', nav_text:'#5C3D1A',
                          surface:'#FFFFFF', text:'#3A2710', hero_gradient:'linear-gradient(135deg,#FFF3D6 0%,#FFECD2 50%,#FFFBF0 100%)',
                          shadow:'0 20px 50px rgba(200,133,45,0.12)', font_size:36, card_radius:12, button_radius:10 },
                        { id:'tpl-7', name:'Fresh Lens', tagline:'Clean teal modern portfolio', category:'Portrait', style:'Modern',
                          font_family:'Outfit', grid:'masonry', hero_style:'split', card_style:'default', pattern:'none',
                          grid_gap:16, radius:16, primary:'#0D9488', background:'#F0FDFA', nav_bg:'#FFFFFF', nav_text:'#134E4A',
                          surface:'#FFFFFF', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#CCFBF1 0%,#F0FDFA 100%)',
                          shadow:'0 12px 40px rgba(13,148,136,0.08)', font_size:32, card_radius:16, button_radius:14 },

                        // ── Corporate ────────────────────────────────
                        { id:'tpl-8', name:'Executive Suite', tagline:'Professional navy authority', category:'Corporate', style:'Modern',
                          font_family:'Inter', grid:'square', hero_style:'standard', card_style:'bordered', pattern:'none',
                          grid_gap:8, radius:8, primary:'#1E3A5F', background:'#F8FAFC', nav_bg:'#1E3A5F', nav_text:'#FFFFFF',
                          surface:'#FFFFFF', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#EEF2FF 0%,#E0F2FE 100%)',
                          shadow:'0 4px 16px rgba(30,58,95,0.08)', font_size:32, card_radius:8, button_radius:6 },
                        { id:'tpl-9', name:'Innovation Lab', tagline:'Electric blue tech vibe', category:'Corporate', style:'Glass',
                          font_family:'Roboto', grid:'masonry', hero_style:'minimal', card_style:'glass', pattern:'dots',
                          grid_gap:20, radius:20, primary:'#2563EB', background:'#F1F5FF', nav_bg:'#FFFFFF', nav_text:'#1E293B',
                          surface:'#FFFFFF', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#DBEAFE 0%,#EDE9FE 50%,#F1F5FF 100%)',
                          shadow:'0 20px 50px rgba(37,99,235,0.1)', font_size:30, card_radius:20, button_radius:14 },
                        { id:'tpl-10', name:'Summit', tagline:'Charcoal with orange spark', category:'Corporate', style:'Bold',
                          font_family:'Montserrat', grid:'square', hero_style:'standard', card_style:'minimal', pattern:'lines',
                          grid_gap:0, radius:0, primary:'#EA580C', background:'#FAFAFA', nav_bg:'#18181B', nav_text:'#FAFAFA',
                          surface:'#FFFFFF', text:'#18181B', hero_gradient:'linear-gradient(135deg,#FFF7ED 0%,#FAFAFA 100%)',
                          shadow:'none', font_size:34, card_radius:0, button_radius:4 },

                        // ── Sports ───────────────────────────────────
                        { id:'tpl-11', name:'Game Day', tagline:'Bold red arena energy', category:'Sports', style:'Bold',
                          font_family:'Outfit', grid:'square', hero_style:'standard', card_style:'bordered', pattern:'lines',
                          grid_gap:4, radius:0, primary:'#DC2626', background:'#FAFAFA', nav_bg:'#0A0A0A', nav_text:'#FFFFFF',
                          surface:'#FFFFFF', text:'#0A0A0A', hero_gradient:'linear-gradient(135deg,#FEE2E2 0%,#FAFAFA 100%)',
                          shadow:'none', font_size:40, card_radius:0, button_radius:0 },
                        { id:'tpl-12', name:'Athletic Edge', tagline:'Neon green dark power', category:'Sports', style:'Dark',
                          font_family:'Inter', grid:'collage', hero_style:'minimal', card_style:'glass', pattern:'none',
                          grid_gap:8, radius:12, primary:'#22C55E', background:'#0A0F0A', nav_bg:'#0A120A', nav_text:'#D1FAE5',
                          surface:'#111B11', text:'#ECFDF5', hero_gradient:'linear-gradient(135deg,#052E16 0%,#0A0F0A 100%)',
                          shadow:'0 20px 50px rgba(34,197,94,0.15)', font_size:36, card_radius:12, button_radius:8 },
                        { id:'tpl-13', name:'Victory Lane', tagline:'Royal blue championship feel', category:'Sports', style:'Modern',
                          font_family:'Montserrat', grid:'masonry', hero_style:'split', card_style:'elevated', pattern:'none',
                          grid_gap:16, radius:16, primary:'#2563EB', background:'#F8FAFF', nav_bg:'#FFFFFF', nav_text:'#1E3A5F',
                          surface:'#FFFFFF', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#DBEAFE 0%,#EEF2FF 100%)',
                          shadow:'0 16px 40px rgba(37,99,235,0.12)', font_size:34, card_radius:16, button_radius:12 },

                        // ── Party ────────────────────────────────────
                        { id:'tpl-14', name:'Neon Nights', tagline:'Electric pink purple glow', category:'Party', style:'Dark',
                          font_family:'Poppins', grid:'collage', hero_style:'minimal', card_style:'glass', pattern:'dots',
                          grid_gap:12, radius:20, primary:'#EC4899', background:'#0C0014', nav_bg:'#120020', nav_text:'#F9A8D4',
                          surface:'#1A0030', text:'#FDF2F8', hero_gradient:'linear-gradient(135deg,#4A044E 0%,#1E1B4B 50%,#0C0014 100%)',
                          shadow:'0 20px 60px rgba(236,72,153,0.2)', font_size:38, card_radius:20, button_radius:24 },
                        { id:'tpl-15', name:'Confetti Pop', tagline:'Bright joyful celebration', category:'Party', style:'Playful',
                          font_family:'Outfit', grid:'masonry', hero_style:'standard', card_style:'default', pattern:'dots',
                          grid_gap:24, radius:32, primary:'#F59E0B', background:'#FFFCF0', nav_bg:'#FFFFFF', nav_text:'#92400E',
                          surface:'#FFFFFF', text:'#451A03', hero_gradient:'linear-gradient(135deg,#FEF3C7 0%,#FCE7F3 50%,#FFFCF0 100%)',
                          shadow:'0 12px 30px rgba(245,158,11,0.1)', font_size:36, card_radius:32, button_radius:24 },
                        { id:'tpl-16', name:'Velvet Lounge', tagline:'Deep purple & gold luxury', category:'Party', style:'Elegant',
                          font_family:'Playfair Display', grid:'masonry', hero_style:'split', card_style:'elevated', pattern:'none',
                          grid_gap:20, radius:24, primary:'#A855F7', background:'#FAF5FF', nav_bg:'#FFFFFF', nav_text:'#3B0764',
                          surface:'#FFFFFF', text:'#1E0A3C', hero_gradient:'linear-gradient(135deg,#F3E8FF 0%,#EDE9FE 50%,#FAF5FF 100%)',
                          shadow:'0 25px 50px rgba(168,85,247,0.12)', font_size:36, card_radius:24, button_radius:16 },

                        // ── Travel ───────────────────────────────────
                        { id:'tpl-17', name:'Wanderlust', tagline:'Earth tones adventure diary', category:'Travel', style:'Elegant',
                          font_family:'Lora', grid:'masonry', hero_style:'split', card_style:'elevated', pattern:'none',
                          grid_gap:20, radius:16, primary:'#B45309', background:'#FFFBEB', nav_bg:'#FFFBEB', nav_text:'#78350F',
                          surface:'#FFFFFF', text:'#451A03', hero_gradient:'linear-gradient(135deg,#FEF3C7 0%,#FFFBEB 100%)',
                          shadow:'0 20px 50px rgba(180,83,9,0.1)', font_size:34, card_radius:16, button_radius:12 },
                        { id:'tpl-18', name:'Passport Stamps', tagline:'Vintage blue retro explorer', category:'Travel', style:'Retro',
                          font_family:'Roboto', grid:'collage', hero_style:'standard', card_style:'bordered', pattern:'lines',
                          grid_gap:8, radius:4, primary:'#1D4ED8', background:'#FEFCE8', nav_bg:'#FEF9C3', nav_text:'#1E3A8A',
                          surface:'#FFFEF5', text:'#1E3A8A', hero_gradient:'linear-gradient(135deg,#FEF9C3 0%,#FEFCE8 100%)',
                          shadow:'none', font_size:30, card_radius:4, button_radius:4 },
                        { id:'tpl-19', name:'Horizon', tagline:'Ocean blue calm serenity', category:'Travel', style:'Minimal',
                          font_family:'Outfit', grid:'masonry', hero_style:'minimal', card_style:'minimal', pattern:'none',
                          grid_gap:32, radius:8, primary:'#0284C7', background:'#F0F9FF', nav_bg:'#FFFFFF', nav_text:'#0C4A6E',
                          surface:'#FFFFFF', text:'#0C4A6E', hero_gradient:'linear-gradient(180deg,#E0F2FE 0%,#F0F9FF 100%)',
                          shadow:'none', font_size:38, card_radius:8, button_radius:6 },

                        // ── Birthday ─────────────────────────────────
                        { id:'tpl-20', name:'Sweet Sixteen', tagline:'Pastel pink lavender dream', category:'Birthday', style:'Playful',
                          font_family:'Poppins', grid:'masonry', hero_style:'standard', card_style:'default', pattern:'dots',
                          grid_gap:20, radius:28, primary:'#E879A8', background:'#FFF5F9', nav_bg:'#FFFFFF', nav_text:'#831843',
                          surface:'#FFFFFF', text:'#4A0825', hero_gradient:'linear-gradient(135deg,#FCE7F3 0%,#F3E8FF 50%,#FFF5F9 100%)',
                          shadow:'0 12px 30px rgba(232,121,168,0.1)', font_size:34, card_radius:28, button_radius:20 },
                        { id:'tpl-21', name:'Milestone', tagline:'Sophisticated black & gold', category:'Birthday', style:'Dark',
                          font_family:'Montserrat', grid:'square', hero_style:'minimal', card_style:'elevated', pattern:'none',
                          grid_gap:8, radius:12, primary:'#EAB308', background:'#09090B', nav_bg:'#09090B', nav_text:'#FDE68A',
                          surface:'#18181B', text:'#FAFAF9', hero_gradient:'linear-gradient(135deg,#1C1917 0%,#09090B 100%)',
                          shadow:'0 25px 50px rgba(234,179,8,0.15)', font_size:42, card_radius:12, button_radius:8 },
                        { id:'tpl-22', name:'Fiesta', tagline:'Vibrant orange celebration', category:'Birthday', style:'Bold',
                          font_family:'Poppins', grid:'collage', hero_style:'standard', card_style:'default', pattern:'dots',
                          grid_gap:16, radius:20, primary:'#F97316', background:'#FFF7ED', nav_bg:'#FFFFFF', nav_text:'#9A3412',
                          surface:'#FFFFFF', text:'#431407', hero_gradient:'linear-gradient(135deg,#FFEDD5 0%,#FEF3C7 50%,#FFF7ED 100%)',
                          shadow:'0 16px 40px rgba(249,115,22,0.1)', font_size:36, card_radius:20, button_radius:16 },

                        // ── Baby Shower ──────────────────────────────
                        { id:'tpl-23', name:'Little Star', tagline:'Soft sky blue nursery', category:'Baby Shower', style:'Playful',
                          font_family:'Lora', grid:'masonry', hero_style:'split', card_style:'default', pattern:'dots',
                          grid_gap:24, radius:24, primary:'#60A5FA', background:'#F0F9FF', nav_bg:'#FFFFFF', nav_text:'#1E40AF',
                          surface:'#FFFFFF', text:'#1E3A5F', hero_gradient:'linear-gradient(135deg,#DBEAFE 0%,#E0F2FE 50%,#F0F9FF 100%)',
                          shadow:'0 12px 30px rgba(96,165,250,0.1)', font_size:34, card_radius:24, button_radius:20 },
                        { id:'tpl-24', name:'Blossom', tagline:'Peach & sage gentle warmth', category:'Baby Shower', style:'Elegant',
                          font_family:'Playfair Display', grid:'masonry', hero_style:'standard', card_style:'elevated', pattern:'none',
                          grid_gap:20, radius:20, primary:'#F4845F', background:'#FFF7F3', nav_bg:'#FFFFFF', nav_text:'#7C2D12',
                          surface:'#FFFFFF', text:'#431407', hero_gradient:'linear-gradient(135deg,#FFEDD5 0%,#FFF7F3 100%)',
                          shadow:'0 20px 50px rgba(244,132,95,0.1)', font_size:36, card_radius:20, button_radius:16 },
                        { id:'tpl-25', name:'Cloud Nine', tagline:'Dreamy lavender whisper', category:'Baby Shower', style:'Glass',
                          font_family:'Outfit', grid:'square', hero_style:'minimal', card_style:'glass', pattern:'dots',
                          grid_gap:16, radius:28, primary:'#A78BFA', background:'#FAF5FF', nav_bg:'#FFFFFF', nav_text:'#5B21B6',
                          surface:'#FFFFFF', text:'#3B0764', hero_gradient:'linear-gradient(135deg,#EDE9FE 0%,#F3E8FF 50%,#FAF5FF 100%)',
                          shadow:'0 16px 40px rgba(167,139,250,0.1)', font_size:32, card_radius:28, button_radius:20 },

                        // ── Graduation ───────────────────────────────
                        { id:'tpl-26', name:'Magna Carta', tagline:'Deep navy academic prestige', category:'Graduation', style:'Bold',
                          font_family:'Montserrat', grid:'square', hero_style:'standard', card_style:'bordered', pattern:'none',
                          grid_gap:8, radius:4, primary:'#D4AF37', background:'#F8FAFC', nav_bg:'#0F172A', nav_text:'#F1F5F9',
                          surface:'#FFFFFF', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#E2E8F0 0%,#F8FAFC 100%)',
                          shadow:'0 4px 16px rgba(15,23,42,0.08)', font_size:36, card_radius:4, button_radius:4 },
                        { id:'tpl-27', name:'New Chapter', tagline:'Fresh green optimistic start', category:'Graduation', style:'Modern',
                          font_family:'Lora', grid:'masonry', hero_style:'split', card_style:'minimal', pattern:'none',
                          grid_gap:24, radius:12, primary:'#059669', background:'#F0FDF4', nav_bg:'#FFFFFF', nav_text:'#064E3B',
                          surface:'#FFFFFF', text:'#022C22', hero_gradient:'linear-gradient(135deg,#D1FAE5 0%,#F0FDF4 100%)',
                          shadow:'none', font_size:34, card_radius:12, button_radius:10 },
                        { id:'tpl-28', name:'Class Act', tagline:'Sharp monochrome with red', category:'Graduation', style:'Minimal',
                          font_family:'Inter', grid:'masonry', hero_style:'minimal', card_style:'elevated', pattern:'none',
                          grid_gap:12, radius:8, primary:'#DC2626', background:'#FFFFFF', nav_bg:'#FFFFFF', nav_text:'#171717',
                          surface:'#FFFFFF', text:'#171717', hero_gradient:'linear-gradient(180deg,#FAFAFA 0%,#FFFFFF 100%)',
                          shadow:'0 20px 50px rgba(0,0,0,0.06)', font_size:32, card_radius:8, button_radius:6 },

                        // ── Festival ─────────────────────────────────
                        { id:'tpl-29', name:'Carnival', tagline:'Vivid multi-color spectacle', category:'Festival', style:'Bold',
                          font_family:'Poppins', grid:'collage', hero_style:'standard', card_style:'glass', pattern:'dots',
                          grid_gap:12, radius:20, primary:'#E11D48', background:'#0F0218', nav_bg:'#1A0328', nav_text:'#FDF2F8',
                          surface:'#1F0A32', text:'#FDF2F8', hero_gradient:'linear-gradient(135deg,#4C1D95 0%,#831843 50%,#0F0218 100%)',
                          shadow:'0 20px 60px rgba(225,29,72,0.2)', font_size:38, card_radius:20, button_radius:16 },
                        { id:'tpl-30', name:'Harvest Moon', tagline:'Warm autumn golden hues', category:'Festival', style:'Retro',
                          font_family:'Lora', grid:'masonry', hero_style:'split', card_style:'elevated', pattern:'none',
                          grid_gap:20, radius:12, primary:'#D97706', background:'#FFFBEB', nav_bg:'#FFFBEB', nav_text:'#92400E',
                          surface:'#FFFFFF', text:'#451A03', hero_gradient:'linear-gradient(135deg,#FDE68A 0%,#FEF3C7 50%,#FFFBEB 100%)',
                          shadow:'0 20px 50px rgba(217,119,6,0.1)', font_size:34, card_radius:12, button_radius:10 },
                        { id:'tpl-31', name:'Winter Glow', tagline:'Icy blue festive shimmer', category:'Festival', style:'Glass',
                          font_family:'Outfit', grid:'masonry', hero_style:'standard', card_style:'glass', pattern:'dots',
                          grid_gap:16, radius:24, primary:'#38BDF8', background:'#F0F9FF', nav_bg:'#FFFFFF', nav_text:'#0C4A6E',
                          surface:'#FFFFFF', text:'#0C4A6E', hero_gradient:'linear-gradient(135deg,#BAE6FD 0%,#E0F2FE 50%,#F0F9FF 100%)',
                          shadow:'0 16px 40px rgba(56,189,248,0.1)', font_size:34, card_radius:24, button_radius:16 },

                        // ── General / Multipurpose ───────────────────
                        { id:'tpl-32', name:'Minimalist Pro', tagline:'Pure clean zero distraction', category:'General', style:'Minimal',
                          font_family:'Inter', grid:'square', hero_style:'minimal', card_style:'minimal', pattern:'none',
                          grid_gap:48, radius:0, primary:'#171717', background:'#FFFFFF', nav_bg:'#FFFFFF', nav_text:'#171717',
                          surface:'#FFFFFF', text:'#171717', hero_gradient:'linear-gradient(180deg,#FAFAFA 0%,#FFFFFF 100%)',
                          shadow:'none', font_size:44, card_radius:0, button_radius:2 },
                        { id:'tpl-33', name:'Aurora Glass', tagline:'Purple translucent futurism', category:'General', style:'Glass',
                          font_family:'Outfit', grid:'collage', hero_style:'minimal', card_style:'glass', pattern:'dots',
                          grid_gap:16, radius:24, primary:'#8B5CF6', background:'#FAF5FF', nav_bg:'#FFFFFF', nav_text:'#5B21B6',
                          surface:'#FFFFFF', text:'#1E0A3C', hero_gradient:'linear-gradient(135deg,#EDE9FE 0%,#E0E7FF 50%,#FAF5FF 100%)',
                          shadow:'0 20px 60px rgba(139,92,246,0.12)', font_size:34, card_radius:24, button_radius:16 },
                        { id:'tpl-34', name:'Editorial Serif', tagline:'Magazine-style type focus', category:'General', style:'Editorial',
                          font_family:'Playfair Display', grid:'masonry', hero_style:'split', card_style:'minimal', pattern:'none',
                          grid_gap:48, radius:4, primary:'#0F172A', background:'#FEFDFB', nav_bg:'#FEFDFB', nav_text:'#0F172A',
                          surface:'#FEFDFB', text:'#0F172A', hero_gradient:'linear-gradient(135deg,#F5F0EB 0%,#FEFDFB 100%)',
                          shadow:'none', font_size:48, card_radius:4, button_radius:2 },
                        { id:'tpl-35', name:'Retro Sunset', tagline:'Warm coral & mustard vibes', category:'General', style:'Retro',
                          font_family:'Poppins', grid:'masonry', hero_style:'standard', card_style:'elevated', pattern:'lines',
                          grid_gap:12, radius:8, primary:'#E85D2A', background:'#FEF7EE', nav_bg:'#FEF7EE', nav_text:'#7C2D12',
                          surface:'#FFFFFF', text:'#431407', hero_gradient:'linear-gradient(135deg,#FFEDD5 0%,#FDE68A 50%,#FEF7EE 100%)',
                          shadow:'0 16px 40px rgba(232,93,42,0.1)', font_size:32, card_radius:8, button_radius:8 },
                        { id:'tpl-36', name:'Nordic Frost', tagline:'Cool grey Scandinavian calm', category:'General', style:'Modern',
                          font_family:'Outfit', grid:'masonry', hero_style:'minimal', card_style:'default', pattern:'none',
                          grid_gap:20, radius:16, primary:'#6366F1', background:'#F8FAFC', nav_bg:'#FFFFFF', nav_text:'#334155',
                          surface:'#FFFFFF', text:'#1E293B', hero_gradient:'linear-gradient(135deg,#E0E7FF 0%,#F1F5F9 100%)',
                          shadow:'0 12px 40px rgba(99,102,241,0.08)', font_size:32, card_radius:16, button_radius:12 },
                    ];
                },

                syncSelectedTemplate() {
                    const key = this.settings.template_key;
                    if (!key) {
                        this.selectedTemplate = null;
                        return;
                    }
                    this.selectedTemplate = this.templates.find(t => t.id === key) || null;
                },

                filteredTemplates() {
                    let items = [...this.templates];
                    const search = this.templateSearch.trim().toLowerCase();
                    const category = this.templateCategory;
                    const style = this.templateStyle;

                    if (category && category !== 'All') {
                        items = items.filter(t => t.category === category);
                    }
                    if (style && style !== 'All') {
                        items = items.filter(t => t.style === style);
                    }
                    if (search) {
                        items = items.filter(t => (t.name.toLowerCase().includes(search) || (t.tagline || '').toLowerCase().includes(search) || t.category.toLowerCase().includes(search)));
                    }

                    switch (this.templateSort) {
                        case 'popular':
                            items = items.sort((a, b) => (parseFloat(a.id.split('-')[1]) % 17) - (parseFloat(b.id.split('-')[1]) % 17));
                            break;
                        case 'az':
                            items = items.sort((a, b) => a.name.localeCompare(b.name));
                            break;
                        default:
                            items = items.sort((a, b) => b.order - a.order);
                    }
                    return items;
                },

                tplMiniGrid(tpl) {
                    const layout = tpl.grid || 'masonry';
                    const gap = Math.min(tpl.grid_gap || 16, 6) + 'px';
                    const rad = Math.min(tpl.radius || 0, 6) + 'px';
                    if (layout === 'square') return { display:'grid', gridTemplateColumns:'1fr 1fr 1fr', gap, children: [1,1,1,1,1,1], ratio:'1/1', rad };
                    if (layout === 'collage') return { display:'grid', gridTemplateColumns:'1fr 1fr', gap, children: [2,1,1,1,1,2], ratio:'auto', rad };
                    return { display:'grid', gridTemplateColumns:'1fr 1fr 1fr', gap, children: [1,1,1,1,1,1], ratio:'3/4', rad };
                },

                selectTemplate(tpl) {
                    this.selectedTemplate = tpl;
                    this.settings = {
                        ...this.settings,
                        template_key: tpl.id,
                        template_name: tpl.name,
                        template_category: tpl.category,
                        template_style: tpl.style,
                        font_family: tpl.font_family || 'Outfit',
                        primary_color: tpl.primary,
                        background_color: tpl.background,
                        navbar_bg_color: tpl.nav_bg,
                        navbar_text_color: tpl.nav_text,
                        surface_color: tpl.surface,
                        text_color: tpl.text,
                        hero_gradient: tpl.hero_gradient,
                        card_shadow: tpl.shadow,
                        card_radius: tpl.radius || 18,
                        image_radius: tpl.radius || 0,
                        grid_gap: tpl.grid_gap || 16,
                        button_radius: tpl.style === 'Minimalist' ? 4 : (tpl.style === 'Luxe' ? 30 : 14),
                        grid_layout: tpl.grid,
                        hero_style: tpl.hero_style || 'standard',
                        card_style: tpl.card_style || 'default',
                        pattern: tpl.pattern || 'none',
                        tab_style: tpl.tab_style || 'underline',
                    };
                    this.templateModalOpen = false;
                },

                resetTemplate() {
                    this.selectedTemplate = null;
                    this.settings = {
                        ...this.settings,
                        template_key: null,
                        template_name: null,
                        template_category: null,
                        template_style: null,
                        font_family: 'Outfit',
                        font_size: 32,
                        grid_gap: 16,
                        image_radius: 0,
                        grid_layout: 'masonry',
                        hero_style: 'standard',
                        card_style: 'default',
                        pattern: 'none',
                        primary_color: '#2563EB',
                        background_color: '#FFFFFF',
                        navbar_bg_color: '#FFFFFF',
                        navbar_text_color: '#1E293B',
                        surface_color: '#FFFFFF',
                        text_color: '#0F172A',
                        hero_gradient: 'linear-gradient(135deg,#EEF2FF 0%,#E0F2FE 50%,#FDF2F8 100%)',
                        card_shadow: '0 12px 40px rgba(15,23,42,0.08)',
                        card_radius: 18,
                        button_radius: 14,
                        tab_style: 'underline',
                    };
                },

                sendSettings() {
                    const win = this.$refs.previewFrame.contentWindow;
                    if(win) {
                        win.postMessage({ type: 'design-update', settings: JSON.parse(JSON.stringify(this.settings)) }, '*');
                    }
                },
                
                save() {
                    this.saving = true;
                    axios.post('{{ route('admin.events.design.update', $event) }}', { design_settings: this.settings })
                        .then(() => {
                            setTimeout(() => {
                                this.saving = false;
                                this.showToast = true;
                                setTimeout(() => this.showToast = false, 3000);
                            }, 500);
                        })
                        .catch(err => {
                            console.error(err);
                            this.saving = false;
                            alert('Error saving settings');
                        });
                },
                
                handleCoverUpload(e) {
                    alert('Please use the central "Edit Event" page to update the cover image.');
                }
            }
        }
    </script>

    <!-- Template Modal -->
    <div x-show="templateModalOpen" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-6">
        <div class="absolute inset-0 bg-slate-900/70 backdrop-blur" @click="templateModalOpen = false"></div>
        <div class="relative bg-white rounded-3xl shadow-2xl w-full max-w-7xl h-[90vh] overflow-hidden flex flex-col">
            <div class="px-6 py-4 border-b flex items-center justify-between shrink-0">
                <div>
                    <p class="text-[10px] uppercase tracking-[0.2em] font-black text-blue-600 mb-1">{{ __('Theme Catalog') }}</p>
                    <p class="text-xl font-bold text-slate-800">{{ __('Choose a Template') }}</p>
                </div>
                <button class="p-2 rounded-full hover:bg-slate-100 transition-colors" @click="templateModalOpen = false">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
                </button>
            </div>

            <div class="flex-1 flex overflow-hidden min-h-0 bg-white">
                <!-- Sidebar -->
                <div class="w-72 border-r bg-slate-50 p-6 overflow-y-auto shrink-0 space-y-2">
                    <p class="text-[10px] font-black uppercase tracking-[0.2em] text-slate-400 mb-6 px-2">{{ __('Occasion') }}</p>
                    <template x-for="cat in templateCategories" :key="cat">
                        <button class="w-full text-left px-5 py-3 rounded-2xl text-sm font-bold transition-all flex items-center justify-between group"
                                :class="templateCategory === cat ? 'bg-blue-600 text-white shadow-xl shadow-blue-100' : 'text-slate-600 hover:bg-white hover:text-slate-900'"
                                @click="templateCategory = cat">
                            <span x-text="cat"></span>
                            <div class="w-1.5 h-1.5 rounded-full bg-white opacity-0 group-hover:opacity-40" :class="templateCategory === cat ? 'opacity-100' : ''"></div>
                        </button>
                    </template>
                </div>

                <!-- Main Grid Area -->
                <div class="flex-1 flex flex-col overflow-hidden min-h-0">
                    <div class="px-8 py-6 border-b flex flex-wrap items-center gap-6 bg-white shrink-0">
                        <div class="relative flex-1 min-w-[320px]">
                            <input type="text" x-model="templateSearch" placeholder="{{ __('Search templates...') }}" class="w-full h-12 pl-12 pr-4 rounded-2xl border-2 border-slate-100 bg-slate-50 focus:bg-white focus:border-blue-500 transition-all outline-none font-medium text-slate-700">
                            <svg class="absolute left-4 top-1/2 -translate-y-1/2 text-slate-400" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="11" cy="11" r="8"></circle><line x1="21" y1="21" x2="16.65" y2="16.65"></line></svg>
                        </div>
                        <div class="flex items-center gap-3">
                            <select x-model="templateStyle" class="h-12 px-5 rounded-2xl border-2 border-slate-100 bg-slate-50 text-sm font-bold outline-none focus:border-blue-500 transition-all text-slate-700">
                                <template x-for="style in templateStyles" :key="style">
                                    <option :value="style" x-text="style"></option>
                                </template>
                            </select>
                            <select x-model="templateSort" class="h-12 px-5 rounded-2xl border-2 border-slate-100 bg-slate-50 text-sm font-bold outline-none focus:border-blue-500 transition-all text-slate-700">
                                <option value="newest">{{ __('Newest First') }}</option>
                                <option value="popular">{{ __('Most Popular') }}</option>
                                <option value="az">{{ __('Alphabetical') }}</option>
                            </select>
                        </div>
                    </div>

                    <div class="flex-1 overflow-y-auto px-10 py-10 bg-slate-50/20">
                        <div class="grid gap-6 grid-cols-1 md:grid-cols-2 xl:grid-cols-3">
                            <template x-for="tpl in filteredTemplates()" :key="tpl.id">
                                <div class="group bg-white rounded-2xl overflow-hidden flex flex-col shadow-sm hover:shadow-xl hover:-translate-y-1 transition-all duration-300 border border-slate-100 cursor-pointer" @click="selectTemplate(tpl)">
                                    <!-- Mini Visual Preview -->
                                    <div class="h-48 relative overflow-hidden" :style="{ background: tpl.background }">
                                        <!-- Hero gradient bar -->
                                        <div class="absolute top-0 left-0 right-0" :style="{ height: tpl.hero_style==='minimal'?'32%':'45%', background: tpl.hero_gradient }">
                                            <!-- Mini nav bar -->
                                            <div class="flex items-center justify-between px-3 py-2">
                                                <div class="flex items-center gap-1.5">
                                                    <div class="w-3 h-3 rounded-sm" :style="{ backgroundColor: tpl.primary }"></div>
                                                    <div class="h-1 w-8 rounded-full opacity-60" :style="{ backgroundColor: tpl.nav_text }"></div>
                                                </div>
                                                <div class="flex gap-1">
                                                    <div class="h-1 w-4 rounded-full opacity-40" :style="{ backgroundColor: tpl.nav_text }"></div>
                                                    <div class="h-1 w-4 rounded-full opacity-40" :style="{ backgroundColor: tpl.nav_text }"></div>
                                                </div>
                                            </div>
                                            <!-- Mini title text -->
                                            <div class="px-3 mt-1">
                                                <div class="h-1.5 w-16 rounded-full opacity-70" :style="{ backgroundColor: tpl.text }"></div>
                                                <div class="h-1 w-10 rounded-full opacity-40 mt-1" :style="{ backgroundColor: tpl.text }"></div>
                                            </div>
                                        </div>
                                        <!-- Mini photo grid -->
                                        <div class="absolute bottom-2 left-3 right-3" :style="{ display:'grid', gridTemplateColumns: tpl.grid==='collage'?'1fr 1fr':tpl.grid==='square'?'1fr 1fr 1fr':'1fr 1fr 1fr', gap: Math.min(tpl.grid_gap,4)+'px' }">
                                            <template x-for="(n, i) in (tpl.grid==='collage'?5:6)" :key="i">
                                                <div :style="{ backgroundColor: tpl.primary+'18', borderRadius: Math.min(tpl.radius,6)+'px', aspectRatio: tpl.grid==='square'?'1/1':'3/4', border: tpl.card_style==='bordered'?'1.5px solid '+tpl.primary+'40':'none', boxShadow: tpl.card_style==='elevated'?'0 2px 8px '+tpl.primary+'15':'none', backdropFilter: tpl.card_style==='glass'?'blur(4px)':'none', gridColumn: (tpl.grid==='collage'&&(i===0||i===4))?'span 2':'span 1' }">
                                                    <div class="w-full h-full" :style="{ backgroundColor: tpl.card_style==='glass'?tpl.primary+'10':tpl.surface, borderRadius: Math.min(tpl.radius,6)+'px', opacity: tpl.card_style==='glass'?0.6:0.5, mixBlendMode:'multiply' }"></div>
                                                </div>
                                            </template>
                                        </div>
                                        <!-- Pattern overlay -->
                                        <div x-show="tpl.pattern==='dots'" class="absolute inset-0 opacity-[0.04]" style="background-image:radial-gradient(circle,currentColor 1px,transparent 1px);background-size:8px 8px" :style="{ color: tpl.text }"></div>
                                        <div x-show="tpl.pattern==='lines'" class="absolute inset-0 opacity-[0.03]" style="background-image:repeating-linear-gradient(45deg,currentColor 0,currentColor 1px,transparent 0,transparent 6px)" :style="{ color: tpl.text }"></div>
                                        <!-- Active indicator -->
                                        <div x-show="settings.template_key === tpl.id" class="absolute top-3 right-3 w-6 h-6 bg-blue-600 rounded-full flex items-center justify-center shadow-lg">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="3"><path d="M20 6 9 17l-5-5"/></svg>
                                        </div>
                                    </div>
                                    <!-- Info -->
                                    <div class="p-4 flex-1 flex flex-col">
                                        <div class="flex items-start justify-between gap-2">
                                            <div class="min-w-0 flex-1">
                                                <p class="font-bold text-slate-900 text-sm leading-tight" x-text="tpl.name"></p>
                                                <p class="text-xs text-slate-500 mt-0.5" x-text="tpl.tagline"></p>
                                            </div>
                                            <!-- Color dots -->
                                            <div class="flex gap-1 shrink-0 mt-0.5">
                                                <div class="w-3.5 h-3.5 rounded-full border border-white shadow-sm" :style="{ backgroundColor: tpl.primary }"></div>
                                                <div class="w-3.5 h-3.5 rounded-full border border-slate-200 shadow-sm" :style="{ backgroundColor: tpl.background }"></div>
                                                <div class="w-3.5 h-3.5 rounded-full border border-slate-200 shadow-sm" :style="{ backgroundColor: tpl.nav_bg }"></div>
                                            </div>
                                        </div>
                                        <div class="flex gap-2 items-center mt-2.5">
                                            <span class="px-2 py-0.5 rounded-md text-[10px] font-bold uppercase tracking-wider" :style="{ backgroundColor: tpl.primary+'12', color: tpl.primary }" x-text="tpl.category"></span>
                                            <span class="px-2 py-0.5 rounded-md bg-slate-100 text-[10px] font-bold uppercase tracking-wider text-slate-500" x-text="tpl.style"></span>
                                            <span class="px-2 py-0.5 rounded-md bg-slate-50 text-[10px] font-medium text-slate-400" x-text="tpl.grid"></span>
                                        </div>
                                    </div>
                                </div>
                            </template>
                        </div>
                        <p x-show="filteredTemplates().length === 0" class="text-center text-slate-400 py-20 text-sm">{{ __('No templates match your filters.') }}</p>
                    </div>
                </div>
            </div>

            <!-- Bottom bar: count + close -->
            <div class="border-t px-6 py-3 bg-white flex items-center justify-between shrink-0">
                <p class="text-xs text-slate-500"><span class="font-bold text-slate-700" x-text="filteredTemplates().length"></span> {{ __('templates available') }}</p>
                <button class="px-5 py-2 rounded-xl border border-slate-200 text-sm font-semibold text-slate-700 hover:bg-slate-100 transition-colors" @click="templateModalOpen = false">{{ __('Close') }}</button>
            </div>
        </div>
    </div>

    <!-- Success Toast -->
    <div x-show="showToast" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0 transform translate-y-4"
         x-transition:enter-end="opacity-100 transform translate-y-0"
         x-transition:leave="transition ease-in duration-200"
         x-transition:leave-start="opacity-100 transform translate-y-0"
         x-transition:leave-end="opacity-0 transform translate-y-4"
         class="fixed bottom-8 left-1/2 -translate-x-1/2 z-50 bg-slate-900 text-white px-6 py-3 rounded-2xl shadow-2xl flex items-center gap-3 border border-slate-700 font-medium"
         x-cloak>
        <div class="w-2 h-2 rounded-full bg-green-500 animate-pulse"></div>
        <span>{{ __('Design settings applied successfully!') }}</span>
    </div>

</body>
</html>
