<x-admin-layout :title="__('Event Details')" :subtitle="__('Review event privacy, sharing, and uploads.')">
    @php
        $shareUrl = $event->share_url;
        $chunkSize = $chunkSize ?? (int) config('events.media.chunk_size', 5242880);
        $ftpBasePath = $ftpBasePath ?? rtrim((string) config('events.media.ftp_path'), DIRECTORY_SEPARATOR);
    @endphp

    <div class="space-y-6">
        <!-- Header -->
        <div class="flex flex-wrap items-center justify-between gap-4">
             <div>
                 <div class="flex items-center gap-2">
                     <h2 class="text-xl font-semibold text-ink">{{ $event->name }}</h2>
                     <a href="{{ route('admin.events.edit', $event) }}" class="text-muted hover:text-ink">
                         <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 3a2.828 2.828 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5L17 3z"/></svg>
                     </a>
                 </div>
                 <p class="mt-1 text-sm text-muted">{{ __('Photos:') }} {{ $mediaCounts['total'] }} &middot; {{ __('Date:') }} {{ \App\Support\FormatSettings::date($event->event_date) }}</p>
            </div>
            <div class="flex flex-wrap items-center gap-3">
                 <span class="badge {{ $event->is_active ? 'badge-success' : 'badge-danger' }}">{{ $event->is_active ? __('Published') : __('Inactive') }}</span>

                 <a href="{{ $shareUrl }}" target="_blank" class="btn-primary flex items-center gap-2">
                     <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M4 12v8a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2v-8"/><polyline points="16 6 12 2 8 6"/><line x1="12" x2="12" y1="2" y2="15"/></svg>
                     {{ __('Share') }}
                </a>
            </div>
        </div>

        <div class="panel p-6" data-tab-group data-tab-storage="event-view-{{ $event->id }}">
            <div class="flex flex-wrap gap-2 pb-4 mb-6">
                <!-- Tabs -->
                <button type="button" class="btn-secondary" data-tab-button="photos">{{ __('Photos') }}</button>
                @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_guest_upload'))
                <button type="button" class="btn-secondary" data-tab-button="guest_upload">
                    {{ __('Guest Upload') }}
                    @if($event->allow_guest_upload)
                        <span class="ml-2 inline-block h-2 w-2 rounded-full bg-green-500"></span>
                    @endif
                </button>
                @endif
                @if (auth()->user()->hasRole(['Super Admin', 'Admin']) || (auth()->user()->plan && auth()->user()->plan->has_design_editor))
                    <a href="{{ route('admin.events.design', $event) }}" target="_blank" class="btn-secondary">{{ __('Design') }}</a>
                @endif
                <button type="button" class="btn-secondary" data-tab-button="details">{{ __('Share') }}</button>
                <button type="button" class="btn-secondary" data-tab-button="settings">{{ __('Settings') }}</button>
                <button type="button" class="btn-secondary" data-tab-button="selections">
                    {{ __('Selections') }}
                    @if(($selectionsCount ?? 0) > 0)
                        <span class="ml-1 inline-flex h-5 min-w-[1.25rem] items-center justify-center rounded-full bg-brand/20 px-1.5 text-xs font-semibold text-brand">{{ $selectionsCount }}</span>
                    @endif
                </button>
                <button type="button" class="btn-secondary" data-tab-button="analytics">{{ __('Analytics') }}</button>
            </div>

            <div class="mt-6 space-y-8">
                <!-- PHOTOS TAB -->
                <div data-tab-panel="photos">
                    <div data-media-wrapper data-bulk-scope data-view-scope data-view-mode="grid" data-view-storage="event-media-view-{{ $event->id }}" data-bulk-total="{{ $media->total() }}">
                        <div class="flex flex-wrap items-center justify-between gap-4">
                            <div>
                                <div class="flex items-center gap-3">
                                    <p class="panel-heading">{{ __('Media Uploads') }}</p>
                                    <button type="button" class="text-xs font-medium text-brand hover:underline" x-data="" x-on:click.prevent="$dispatch('open-modal', 'upload-how-it-works')">{{ __('How it works?') }}</button>
                                </div>
                                <p class="mt-2 text-sm text-muted">{{ __('Upload images or videos, then monitor processing status.') }}</p>
                            </div>
                            <div class="flex flex-wrap items-center gap-3 text-xs text-muted">
                                <span>{{ __('Images:') }} {{ $mediaCounts['images'] }}</span>
                                <span>{{ __('Videos:') }} {{ $mediaCounts['videos'] }}</span>
                            </div>
                        </div>

                        <!-- Upload Forms -->
                        <div class="mt-6 grid gap-4 lg:grid-cols-2">
                            <!-- Regular Upload -->
                            <form method="POST" action="{{ route('admin.events.media.store', $event) }}" enctype="multipart/form-data" class="rounded-2xl border border-border/70 bg-surface-2/40 p-4 space-y-3" data-chunk-upload-form data-chunk-endpoint="{{ route('admin.events.media.chunk', $event) }}" data-chunk-size="{{ $chunkSize }}">
                                @csrf
                                <div>
                                    <h3 class="text-sm font-semibold text-ink">{{ __('Upload Files') }}</h3>
                                    <p class="mt-1 text-xs text-muted">{{ __('Best for small batches (up to 50 files).') }}</p>
                                </div>
                                <input id="media" name="media[]" type="file" accept=".jpg,.jpeg,.png,.webp,.mp4" multiple class="text-sm text-muted" data-chunk-upload-input>
                                <x-input-error :messages="array_merge($errors->get('media'), $errors->get('media.*'))" class="mt-1" />
                                <div class="flex items-center gap-3">
                                    <input id="chunked" type="checkbox" class="h-4 w-4 rounded border-border text-brand focus:ring-brand/40" data-chunk-toggle>
                                    <label for="chunked" class="text-xs text-muted">{{ __('Chunked mode (for files > 50 MB)') }}</label>
                                </div>
                                <div class="flex flex-wrap items-center gap-3">
                                    <button type="submit" class="btn-primary text-sm">{{ __('Upload') }}</button>
                                    <span class="text-xs text-muted" data-chunk-progress></span>
                                </div>
                            </form>

                            <!-- ZIP Upload (recommended for bulk) -->
                            <form method="POST" action="{{ route('admin.events.media.zip', $event) }}" enctype="multipart/form-data" class="rounded-2xl border-2 border-brand/30 bg-brand/5 p-4 space-y-3" id="zip-upload-form">
                                @csrf
                                <div>
                                    <h3 class="text-sm font-semibold text-ink flex items-center gap-2">
                                        {{ __('ZIP Upload') }}
                                        <span class="rounded-full bg-brand/10 px-2 py-0.5 text-[10px] font-bold uppercase tracking-wider text-brand">{{ __('Recommended for bulk') }}</span>
                                    </h3>
                                    <p class="mt-1 text-xs text-muted">{{ __('Upload 100-5000+ photos in one go. Just zip your folder and upload. Much faster than individual files.') }}</p>
                                </div>
                                <input id="zip-file" name="zip" type="file" accept=".zip" class="text-sm text-muted">
                                <x-input-error :messages="$errors->get('zip')" class="mt-1" />
                                <div class="flex flex-wrap items-center gap-3">
                                    <button type="submit" class="btn-primary text-sm" id="zip-upload-btn">{{ __('Upload ZIP') }}</button>
                                    <span class="text-xs text-muted" id="zip-upload-progress"></span>
                                </div>
                            </form>
                        </div>



                        <!-- Google Drive Panel -->
                        @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_google_drive_import'))
                        <div class="mt-6 flex flex-wrap items-center justify-between gap-4 rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                            <div>
                                <p class="text-sm font-semibold text-ink">{{ __('Google Drive Import') }}</p>
                                <p class="mt-1 text-xs text-muted">{{ __('Paste a public folder link to import images and MP4s.') }}</p>
                            </div>
                            <button type="button" class="btn-secondary" x-data="" x-on:click.prevent="$dispatch('open-modal', 'import-google-drive-{{ $event->id }}')">{{ __('Import from Google Drive') }}</button>
                        </div>
                        @endif

                        @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_google_drive_import'))
                        @if(($driveImports ?? collect())->isNotEmpty())
                            <div class="mt-4 rounded-2xl border border-border/70 bg-surface-2/40 p-4" id="drive-imports-panel" data-status-url="{{ route('admin.events.media.google-drive.status', $event) }}">
                                <div class="flex items-center justify-between gap-2">
                                    <div>
                                        <p class="text-sm font-semibold text-ink">{{ __('Google Drive Import Progress') }}</p>
                                        <p class="text-xs text-muted">{{ __('Auto-refreshes while this page is open.') }}</p>
                                    </div>
                                    <button type="button" class="btn-secondary text-xs" id="drive-import-refresh">{{ __('Refresh') }}</button>
                                </div>
                                <div class="mt-3 space-y-3" id="drive-import-list">
                                    @foreach ($driveImports as $import)
                                        @php
                                            $total = $import->total_files ?? 0;
                                            $done = $import->processed_files ?? 0;
                                            $failed = $import->failed_files ?? 0;
                                            $remaining = max(0, $total - $done - $failed);
                                            $percent = $total > 0 ? (int) round(($done / $total) * 100) : 0;
                                        @endphp
                                        <div class="rounded-xl border border-border/60 bg-surface-2/70 p-3 drive-import-row" data-import-id="{{ $import->id }}">
                                            <div class="flex items-center justify-between text-sm">
                                                <div class="flex items-center gap-2">
                                                    <span class="text-xs font-semibold text-muted">#{{ $import->id }}</span>
                                                    <span class="badge">{{ ucfirst($import->status) }}</span>
                                                </div>
                                                <span class="text-xs text-muted">{{ $done }} / {{ $total }} {{ __('processed') }} &middot; {{ $failed }} {{ __('failed') }}</span>
                                            </div>
                                            <div class="mt-2 h-2 rounded-full bg-border/70 overflow-hidden">
                                                <div class="h-2 bg-brand rounded-full transition-all" style="width: {{ $percent }}%;"></div>
                                            </div>
                                            <div class="mt-2 text-xs text-muted">
                                                {{ __('Remaining:') }} <span data-import-remaining>{{ $remaining }}</span>
                                                @if($import->last_error)
                                                    <span class="text-danger ml-2">{{ $import->last_error }}</span>
                                                @endif
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                            </div>
                        @endif

                        <!-- Google Drive Modal -->
                         <x-modal name="import-google-drive-{{ $event->id }}" :show="session('open_google_drive_modal') || $errors->has('google_drive_link')" focusable>
                            <form method="POST" action="{{ route('admin.events.media.google-drive', $event) }}" class="p-6 space-y-4" data-google-drive-form data-modal-name="import-google-drive-{{ $event->id }}">
                                @csrf
                                <div>
                                    <p class="text-lg font-semibold text-ink">{{ __('Import from Google Drive') }}</p>
                                    <p class="mt-2 text-sm text-muted">{{ __('Use a public Google Drive folder link. Supported types: JPG, JPEG, PNG, WEBP, MP4.') }}</p>
                                </div>
                                <div>
                                    <x-input-label for="google_drive_link" :value="__('Google Drive Folder Link')" />
                                    <x-text-input id="google_drive_link" name="google_drive_link" type="text" class="mt-1" :value="old('google_drive_link')" placeholder="https://drive.google.com/drive/folders/..." required />
                                    <x-input-error :messages="$errors->get('google_drive_link')" class="mt-2" />
                                </div>
                                <div class="flex justify-end gap-3">
                                    <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                    <x-primary-button type="submit">{{ __('Import') }}</x-primary-button>
                                </div>
                            </form>
                        </x-modal>
                        @endif

                        <!-- Filters -->
                        <form method="get" class="filter-form mt-6 grid gap-4 lg:grid-cols-[1fr,1fr,auto] md:grid-cols-[1fr,1fr,auto] sm:grid-cols-2">
                            <div>
                                <label class="sr-only" for="media_type">{{ __('Media Type') }}</label>
                                <select id="media_type" name="media_type" class="h-9 rounded-lg border-border bg-surface-2 px-3 py-0 text-sm font-medium text-ink focus:border-brand focus:ring-brand/20 w-full transition-colors">
                                    <option value="">{{ __('All media') }}</option>
                                    <option value="image" @selected($mediaType === 'image')>{{ __('Images') }}</option>
                                    <option value="video" @selected($mediaType === 'video')>{{ __('Videos') }}</option>
                                </select>
                            </div>
                            <div>
                                <label class="sr-only" for="media_sort">{{ __('Sort') }}</label>
                                <select id="media_sort" name="media_sort" class="h-9 rounded-lg border-border bg-surface-2 px-3 py-0 text-sm font-medium text-ink focus:border-brand focus:ring-brand/20 w-full transition-colors">
                                    <option value="newest" @selected($mediaSort === 'newest')>{{ __('Newest') }}</option>
                                    <option value="oldest" @selected($mediaSort === 'oldest')>{{ __('Oldest') }}</option>
                                    <option value="size_desc" @selected($mediaSort === 'size_desc')>{{ __('File Size (High to Low)') }}</option>
                                    <option value="size_asc" @selected($mediaSort === 'size_asc')>{{ __('File Size (Low to High)') }}</option>
                                    <option value="name_asc" @selected($mediaSort === 'name_asc')>{{ __('File Name (A-Z)') }}</option>
                                    <option value="name_desc" @selected($mediaSort === 'name_desc')>{{ __('File Name (Z-A)') }}</option>
                                </select>
                            </div>
                            <div class="flex items-center gap-3">
                                <button class="btn-secondary h-9" type="submit">{{ __('Filter') }}</button>
                                <a href="{{ route('admin.events.show', $event) }}" class="btn-secondary h-9 flex items-center">{{ __('Reset') }}</a>
                            </div>
                        </form>

                        <div class="mt-6 flex flex-wrap items-center justify-between gap-4">
                            <div class="flex items-center gap-2">
                                <button type="button" class="btn-secondary h-9" data-view-toggle="list">{{ __('List') }}</button>
                                <button type="button" class="btn-secondary h-9" data-view-toggle="grid">{{ __('Grid') }}</button>
                            </div>
                            @php
                                $exportQuery = array_merge(request()->query(), ['format' => 'csv']);
                            @endphp
                            <div class="flex flex-wrap items-center gap-2">
                                <button type="button" class="btn-secondary h-9 hidden" data-bulk-action data-bulk-modal="confirm-media-bulk-download" data-bulk-target="media-download">
                                    {{ __('Bulk Download') }} (<span data-bulk-count>0</span>)
                                </button>
                                <x-dropdown align="right" width="48" content-classes="p-2">
                                    <x-slot name="trigger">
                                        <button type="button" class="btn-secondary h-9">{{ __('Export') }}</button>
                                    </x-slot>
                                    <x-slot name="content">
                                        <div class="media-dropdown-list">
                                            <a href="{{ route('admin.events.media.export', array_merge(['event' => $event->id], $exportQuery)) }}" class="media-dropdown-item whitespace-nowrap">{{ __('Export CSV') }}</a>
                                            <a href="{{ route('admin.events.media.export', array_merge(['event' => $event->id], array_merge(request()->query(), ['format' => 'xlsx']))) }}" class="media-dropdown-item whitespace-nowrap">{{ __('Export Excel') }}</a>
                                            <a href="{{ route('admin.events.media.export', array_merge(['event' => $event->id], array_merge(request()->query(), ['format' => 'pdf']))) }}" class="media-dropdown-item whitespace-nowrap">{{ __('Export PDF') }}</a>
                                        </div>
                                    </x-slot>
                                </x-dropdown>
                                <a href="{{ route('admin.events.invoice.download', $event) }}" class="btn-secondary h-9">{{ __('Invoice') }}</a>
                                <button type="button" class="btn-danger h-9 hidden" data-bulk-action data-bulk-modal="confirm-media-bulk-delete" data-bulk-target="media-delete">
                                    {{ __('Bulk Delete') }} (<span data-bulk-count>0</span>)
                                </button>
                            </div>
                        </div>

                        <!-- Select All Pages Banner -->
                        <div class="mt-4 hidden rounded-xl border border-brand/30 bg-brand/5 px-4 py-3 text-sm text-ink" data-bulk-all-pages-banner>
                            <span>{!! __('All <strong data-bulk-page-count>0</strong> items on this page are selected.') !!}</span>
                            <button type="button" class="ml-2 font-semibold text-brand hover:underline" data-bulk-select-all-pages>
                                {!! __('Select all <strong>:total</strong> items across all pages', ['total' => $media->total() ?? $mediaCounts['total']]) !!}
                            </button>
                            <input type="hidden" name="select_all_pages" value="0" data-bulk-all-pages-flag>
                        </div>
                        <div class="mt-2 hidden rounded-xl border border-brand/30 bg-brand/10 px-4 py-3 text-sm font-semibold text-brand" data-bulk-all-selected-banner>
                            {!! __('All <strong>:total</strong> items across all pages are selected.', ['total' => $media->total() ?? $mediaCounts['total']]) !!}
                            <button type="button" class="ml-2 font-normal underline" data-bulk-clear-all-pages>{{ __('Clear selection') }}</button>
                        </div>

                        <!-- VIEW LIST -->
                        <div class="mt-6 overflow-x-auto view-list">
                            <table class="w-full text-left text-sm">
                                <thead class="text-xs uppercase tracking-[0.2em] text-muted border-b border-border/60">
                                    <tr>
                                        <th class="px-3 pb-3 w-10"><input type="checkbox" class="h-4 w-4 rounded border-border bg-surface-2 text-brand focus:ring-brand/40" data-bulk-all></th>
                                        <th class="px-3 pb-3 w-20">{{ __('Preview') }}</th>
                                        <th class="px-3 pb-3 w-20">{{ __('Type') }}</th>
                                        <th class="px-3 pb-3 w-20">{{ __('Size') }}</th>
                                        <th class="px-3 pb-3 w-20">{{ __('Status') }}</th>
                                        <th class="px-3 pb-3 w-12 text-right">{{ __('Actions') }}</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-border/60">
                                    @forelse ($media as $item)
                                        @php
                                            $previewUrl = $item->thumbnail_url ?: $item->preview_url;
                                            $sizeLabel = $item->size ? number_format($item->size / 1024, 1) . ' KB' : 'N/A';
                                        @endphp
                                        <tr class="hover:bg-surface-2/40 transition-colors">
                                            <td class="px-3 py-4"><input type="checkbox" value="{{ $item->id }}" class="h-4 w-4 rounded border-border bg-surface-2 text-brand focus:ring-brand/40" data-bulk-item></td>
                                            <td class="px-3 py-4">
                                                @if ($item->file_type === 'video')
                                                    <video src="{{ $previewUrl }}" class="h-14 w-20 rounded-lg border border-border object-cover" muted></video>
                                                @else
                                                    <img src="{{ $previewUrl }}" class="h-14 w-20 rounded-lg border border-border object-cover">
                                                @endif
                                            </td>
                                            <td class="px-3 py-4 text-sm text-ink whitespace-nowrap">{{ strtoupper($item->file_type) }}</td>
                                            <td class="px-3 py-4 text-sm text-ink whitespace-nowrap">{{ $sizeLabel }}</td>
                                            <td class="px-3 py-4 text-sm text-ink whitespace-nowrap">{{ ucfirst($item->status) }}</td>
                                            <td class="px-3 py-4 text-right">
                                                <x-dropdown align="right" width="56" content-classes="media-dropdown">
                                                    <x-slot name="trigger">
                                                        <button type="button" class="media-icon-button" aria-label="{{ __('More actions') }}">
                                                            <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                                <circle cx="12" cy="5" r="1"></circle>
                                                                <circle cx="12" cy="12" r="1"></circle>
                                                                <circle cx="12" cy="19" r="1"></circle>
                                                            </svg>
                                                        </button>
                                                    </x-slot>
                                                    <x-slot name="content">
                                                        <div class="media-dropdown-list">
                                                            <button type="button" class="media-dropdown-item" x-data="" x-on:click.prevent="$dispatch('open-modal', 'preview-media-{{ $item->id }}')" title="{{ __('Preview') }}">
                                                                <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                    <path d="M1 12s4-7 11-7 11 7 11 7-4 7-11 7-11-7-11-7z"></path>
                                                                    <circle cx="12" cy="12" r="3"></circle>
                                                                </svg>
                                                                <span>{{ __('Preview') }}</span>
                                                            </button>
                                                            <a href="{{ route('admin.events.media.download', [$event, $item]) }}" class="media-dropdown-item" title="{{ __('Download') }}">
                                                                <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                    <path d="M12 3v14"></path>
                                                                    <path d="M6 11l6 6 6-6"></path>
                                                                    <path d="M4 21h16"></path>
                                                                </svg>
                                                                <span>{{ __('Download') }}</span>
                                                            </a>
                                                            <a href="{{ route('admin.events.media.download', [$event, $item, 'type' => 'original']) }}" class="media-dropdown-item" title="{{ __('Download original') }}">
                                                                <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                    <path d="M12 3v14"></path>
                                                                    <path d="M6 11l6 6 6-6"></path>
                                                                    <path d="M4 21h16"></path>
                                                                </svg>
                                                                <span>{{ __('Original') }}</span>
                                                            </a>
                                                            @if ($item->file_type === 'image')
                                                                <form method="POST" action="{{ route('admin.events.media.cover', [$event, $item]) }}">
                                                                    @csrf
                                                                    <button type="submit" class="media-dropdown-item" title="{{ __('Set as cover') }}">
                                                                        <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                            <path d="M3 5h18v14H3z"></path>
                                                                            <path d="m3 15 5-5 4 4 5-5 4 4"></path>
                                                                        </svg>
                                                                        <span>{{ __('Make Cover') }}</span>
                                                                    </button>
                                                                </form>
                                                            @endif
                                                            <form method="POST" action="{{ route('admin.events.media.featured', [$event, $item]) }}">
                                                                @csrf
                                                                <button type="submit" class="media-dropdown-item" title="{{ $item->is_featured ? __('Unfeature') : __('Feature') }}">
                                                                    <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                        <path d="M12 17.27 18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z"></path>
                                                                    </svg>
                                                                    <span>{{ $item->is_featured ? __('Unfeature') : __('Feature') }}</span>
                                                                </button>
                                                            </form>
                                                            <button type="button" class="media-dropdown-item media-dropdown-item-danger" x-data="" x-on:click.prevent="$dispatch('open-modal', 'confirm-media-delete-{{ $item->id }}')" title="{{ __('Delete') }}">
                                                                <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                    <polyline points="3 6 5 6 21 6"></polyline>
                                                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6"></path>
                                                                    <path d="M10 11v6"></path>
                                                                    <path d="M14 11v6"></path>
                                                                    <path d="M9 6V4a1 1 0 0 1 1-1h4a1 1 0 0 1 1 1v2"></path>
                                                                </svg>
                                                                <span>{{ __('Delete') }}</span>
                                                            </button>
                                                        </div>
                                                    </x-slot>
                                                </x-dropdown>
                                                <!-- Modals -->
                                                <x-modal name="confirm-media-delete-{{ $item->id }}" :show="false" focusable>
                                                    <form method="POST" action="{{ route('admin.events.media.destroy', [$event, $item]) }}" class="p-6 space-y-4">
                                                        @csrf
                                                        @method('DELETE')
                                                        <div>
                                                            <p class="text-lg font-semibold text-ink">{{ __('Delete this file?') }}</p>
                                                            <p class="mt-2 text-sm text-muted">{!! __('This file will be <strong class="text-danger">permanently deleted</strong> from the server. This action cannot be undone.') !!}</p>
                                                        </div>
                                                        <div class="flex justify-end gap-3">
                                                            <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                                            <x-danger-button type="submit">{{ __('Delete Permanently') }}</x-danger-button>
                                                        </div>
                                                    </form>
                                                </x-modal>
                                                <x-modal name="preview-media-{{ $item->id }}" :show="false" focusable>
                                                    <div class="p-6 space-y-4">
                                                        <div class="flex items-start justify-between gap-4">
                                                            <div>
                                                                <p class="text-sm text-muted">{{ strtoupper($item->file_type) }} &middot; {{ $sizeLabel }}</p>
                                                            </div>
                                                            <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Close') }}</x-secondary-button>
                                                        </div>
                                                        @if ($item->file_type === 'video')
                                                            <video src="{{ $item->preview_url }}" controls class="w-full rounded-2xl border border-border"></video>
                                                        @else
                                                            <img src="{{ $item->preview_url }}" alt="{{ $item->file_name }}" class="w-full rounded-2xl border border-border">
                                                        @endif
                                                    </div>
                                                </x-modal>
                                            </td>
                                        </tr>
                                    @empty
                                        <tr><td colspan="6" class="py-6 text-center text-sm text-muted">{{ __('No media uploaded yet.') }}</td></tr>
                                    @endforelse
                                </tbody>
                            </table>
                        </div>

                        <!-- VIEW GRID -->
                        <div class="mt-6 grid gap-5 sm:grid-cols-2 lg:grid-cols-4 view-grid">
                            @forelse ($media as $item)
                                @php
                                    $previewUrl = $item->thumbnail_url ?: $item->preview_url;
                                    $sizeLabel = $item->size ? number_format($item->size / 1024, 1) . ' KB' : 'N/A';
                                @endphp
                                <div class="media-card">
                                    <div class="overflow-hidden rounded-t-[22px] bg-muted/5">
                                        @if ($item->file_type === 'video')
                                            <video src="{{ $previewUrl }}" class="h-48 w-full object-cover" muted></video>
                                        @else
                                            <img src="{{ $previewUrl }}" alt="{{ $item->file_name }}" class="h-48 w-full object-cover">
                                        @endif
                                    </div>
                                    <div class="media-card-body">
                                        <div class="media-meta">
                                            <span class="inline-flex items-center gap-1">
                                                <span class="h-1.5 w-1.5 rounded-full bg-ink/70"></span>
                                                <span class="tracking-[0.18em] text-[11px] uppercase">{{ $item->file_type }}</span>
                                            </span>
                                            <span class="text-muted/80">&middot; {{ $sizeLabel }}</span>
                                        </div>
                                        <div class="flex items-center gap-2 text-xs text-muted">
                                            <span class="inline-flex items-center rounded-full bg-muted/10 px-3 py-1 text-[11px] font-semibold uppercase tracking-[0.2em] text-muted">
                                                {{ ucfirst($item->status) }}
                                            </span>
                                            @if ($item->is_cover)
                                                <span class="badge badge-soft">{{ __('Cover') }}</span>
                                            @endif
                                            @if ($item->is_featured)
                                                <span class="badge badge-soft">{{ __('Featured') }}</span>
                                            @endif
                                        </div>
                                    </div>
                                    <div class="media-actions">
                                        <a href="{{ route('admin.events.media.download', [$event, $item]) }}" class="media-icon-button" title="{{ __('Download') }}">
                                            <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M12 3v14"></path>
                                                <path d="M6 11l6 6 6-6"></path>
                                                <path d="M4 21h16"></path>
                                            </svg>
                                        </a>
                                        <x-dropdown align="right" width="56" content-classes="media-dropdown">
                                            <x-slot name="trigger">
                                                <button type="button" class="media-icon-button" aria-label="{{ __('More actions') }}">
                                                    <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                        <circle cx="12" cy="5" r="1"></circle>
                                                        <circle cx="12" cy="12" r="1"></circle>
                                                        <circle cx="12" cy="19" r="1"></circle>
                                                    </svg>
                                                </button>
                                            </x-slot>
                                            <x-slot name="content">
                                                <div class="media-dropdown-list">
                                                    <button type="button" class="media-dropdown-item" x-data="" x-on:click.prevent="$dispatch('open-modal', 'preview-media-{{ $item->id }}')" title="{{ __('Preview') }}">
                                                        <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                            <path d="M1 12s4-7 11-7 11 7 11 7-4 7-11 7-11-7-11-7z"></path>
                                                            <circle cx="12" cy="12" r="3"></circle>
                                                        </svg>
                                                        <span>{{ __('Preview') }}</span>
                                                    </button>
                                                    <a href="{{ route('admin.events.media.download', [$event, $item, 'type' => 'original']) }}" class="media-dropdown-item" title="{{ __('Download original') }}">
                                                        <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                            <path d="M12 3v14"></path>
                                                            <path d="M6 11l6 6 6-6"></path>
                                                            <path d="M4 21h16"></path>
                                                        </svg>
                                                        <span>{{ __('Original') }}</span>
                                                    </a>
                                                    @if ($item->file_type === 'image')
                                                        <form method="POST" action="{{ route('admin.events.media.cover', [$event, $item]) }}">
                                                            @csrf
                                                            <button type="submit" class="media-dropdown-item" title="{{ __('Set as cover') }}">
                                                                <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                    <path d="M3 5h18v14H3z"></path>
                                                                    <path d="m3 15 5-5 4 4 5-5 4 4"></path>
                                                                </svg>
                                                                <span>{{ __('Make Cover') }}</span>
                                                            </button>
                                                        </form>
                                                    @endif
                                                    <form method="POST" action="{{ route('admin.events.media.featured', [$event, $item]) }}">
                                                        @csrf
                                                        <button type="submit" class="media-dropdown-item" title="{{ $item->is_featured ? __('Unfeature') : __('Feature') }}">
                                                            <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                                <path d="M12 17.27 18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z"></path>
                                                            </svg>
                                                            <span>{{ $item->is_featured ? __('Unfeature') : __('Feature') }}</span>
                                                        </button>
                                                    </form>
                                                    <button type="button" class="media-dropdown-item media-dropdown-item-danger" x-data="" x-on:click.prevent="$dispatch('open-modal', 'confirm-media-delete-{{ $item->id }}')" title="{{ __('Delete') }}">
                                                        <svg viewBox="0 0 24 24" class="h-5 w-5" fill="none" stroke="currentColor" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round">
                                                            <polyline points="3 6 5 6 21 6"></polyline>
                                                            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6"></path>
                                                            <path d="M10 11v6"></path>
                                                            <path d="M14 11v6"></path>
                                                            <path d="M9 6V4a1 1 0 0 1 1-1h4a1 1 0 0 1 1 1v2"></path>
                                                        </svg>
                                                        <span>{{ __('Delete') }}</span>
                                                    </button>
                                                </div>
                                            </x-slot>
                                        </x-dropdown>
                                    </div>

                                     <x-modal name="confirm-media-delete-{{ $item->id }}" :show="false" focusable>
                                        <form method="POST" action="{{ route('admin.events.media.destroy', [$event, $item]) }}" class="p-6 space-y-4">
                                            @csrf
                                            @method('DELETE')
                                            <div>
                                                <p class="text-lg font-semibold text-ink">{{ __('Delete') }} {{ $item->file_name }}?</p>
                                                <p class="mt-2 text-sm text-muted">{!! __('This file will be <strong class="text-danger">permanently deleted</strong> from the server. This action cannot be undone.') !!}</p>
                                            </div>
                                            <div class="flex justify-end gap-3">
                                                <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                                <x-danger-button type="submit">{{ __('Delete Permanently') }}</x-danger-button>
                                            </div>
                                        </form>
                                    </x-modal>
                                </div>
                            @empty
                                <div class="col-span-full py-12 text-center text-muted">
                                    <p>{{ __('No photos uploaded yet.') }}</p>
                                </div>
                            @endforelse
                        </div>

                        <div class="mt-6">
                            {{ $media->links() }}
                        </div>

                        <x-modal name="confirm-media-bulk-download" :show="false" focusable>
                            <form method="POST" action="{{ route('admin.events.media.bulk-download', $event) }}" class="p-6 space-y-4">
                                @csrf
                                <div data-bulk-target="media-download"></div>
                                <div>
                                    <p class="text-lg font-semibold text-ink">{{ __('Download selected media?') }}</p>
                                    <p class="mt-2 text-sm text-muted">
                                        {!! __('You are about to download <span class="font-semibold text-ink" data-bulk-selected>0</span> item(s).') !!}
                                    </p>
                                </div>
                                <div class="flex justify-end gap-3">
                                    <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                    <x-primary-button type="submit">{{ __('Download') }}</x-primary-button>
                                </div>
                            </form>
                        </x-modal>

                        <x-modal name="confirm-media-bulk-delete" :show="false" focusable>
                            <form method="POST" action="{{ route('admin.events.media.bulk-destroy', $event) }}" class="p-6 space-y-4">
                                @csrf
                                @method('DELETE')
                                <div data-bulk-target="media-delete"></div>
                                <div>
                                    <p class="text-lg font-semibold text-ink">{{ __('Delete selected media?') }}</p>
                                    <p class="mt-2 text-sm text-muted">
                                        {!! __('You are about to <strong class="text-danger">permanently delete</strong> <span class="font-semibold text-ink" data-bulk-selected>0</span> file(s) from the server. All photos/videos will be removed and cannot be recovered.') !!}
                                    </p>
                                </div>
                                <div class="flex justify-end gap-3">
                                    <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                    <x-danger-button type="submit">{{ __('Delete Permanently') }}</x-danger-button>
                                </div>
                            </form>
                        </x-modal>
                    </div>
                </div>

                <!-- GUEST UPLOAD TAB -->
                <div data-tab-panel="guest_upload">
                    @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_guest_upload'))
                     <div class="flex flex-wrap items-center justify-between gap-4 mb-6">
                         <div>
                            <h3 class="text-lg font-semibold text-ink flex items-center gap-2">
                                {{ __('Guest Uploads') }}
                                <span class="badge {{ $event->allow_guest_upload ? 'badge-success' : 'badge-neutral' }}">{{ $event->allow_guest_upload ? __('Active') : __('Disabled') }}</span>
                            </h3>
                            <p class="text-sm text-muted mt-1">{{ __('Manage media uploaded by guests via the shared link.') }}</p>
                        </div>
                        @if(!$event->allow_guest_upload)
                            <a href="{{ route('admin.events.edit', $event) }}" class="btn-primary">{{ __('Enable Guest Uploads') }}</a>
                        @endif
                    </div>

                    @if(!$event->allow_guest_upload)
                         <div class="p-12 text-center border-2 border-dashed border-border rounded-2xl">
                            <h4 class="text-lg font-medium text-ink">{{ __('Guest Uploads Disabled') }}</h4>
                            <p class="text-muted text-sm max-w-md mx-auto mt-2">{{ __('Enable this feature in event settings to allow guests to upload their own photos.') }}</p>
                        </div>
                    @else
                         <div class="grid gap-5 sm:grid-cols-2 lg:grid-cols-4">
                            @forelse ($media->where('is_guest_upload', true) as $item)
                                @php
                                    $previewUrl = $item->thumbnail_url ?: $item->preview_url;
                                @endphp
                                 <div class="media-card group relative">
                                    <div class="overflow-hidden rounded-t-[22px] bg-surface-2/80 aspect-square">
                                        <img src="{{ $previewUrl }}" alt="{{ $item->file_name }}" class="h-full w-full object-cover">
                                    </div>
                                     <div class="p-3 border border-t-0 border-border rounded-b-[22px] bg-surface-2/50 group-hover:bg-surface-2 transition-colors">
                                        <p class="truncate text-sm font-medium text-ink">{{ $item->file_name }}</p>
                                    </div>
                                    <div class="absolute inset-x-0 bottom-0 p-2 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity flex justify-end gap-2">
                                          <button type="button" x-data="" x-on:click.prevent="$dispatch('open-modal', 'confirm-media-delete-{{ $item->id }}')" class="p-1.5 bg-red-500/80 hover:bg-red-600 rounded-lg text-white backdrop-blur-sm transition-colors" title="{{ __('Delete') }}">
                                             <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"/><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/></svg>
                                         </button>
                                    </div>
                                    <!-- Reusing delete modal from loop since IDs are unique -->
                                    <x-modal name="confirm-media-delete-{{ $item->id }}" :show="false" focusable>
                                        <form method="POST" action="{{ route('admin.events.media.destroy', [$event, $item]) }}" class="p-6 space-y-4">
                                            @csrf
                                            @method('DELETE')
                                            <div>
                                                <p class="text-lg font-semibold text-ink">{{ __('Delete') }} {{ $item->file_name }}?</p>
                                                <p class="mt-2 text-sm text-muted">{!! __('This file will be <strong class="text-danger">permanently deleted</strong> from the server. This action cannot be undone.') !!}</p>
                                            </div>
                                            <div class="flex justify-end gap-3">
                                                <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Cancel') }}</x-secondary-button>
                                                <x-danger-button type="submit">{{ __('Delete Permanently') }}</x-danger-button>
                                            </div>
                                        </form>
                                    </x-modal>
                                </div>
                            @empty
                                <div class="col-span-full py-12 text-center text-muted">
                                    <p>{{ __('No guest uploads found.') }}</p>
                                </div>
                            @endforelse
                        </div>
                    @endif
                    @else
                        <div class="p-12 text-center border-2 border-dashed border-border rounded-2xl">
                            <h4 class="text-lg font-medium text-ink">{{ __('Guest Uploads Not Available') }}</h4>
                            <p class="text-muted text-sm max-w-md mx-auto mt-2">{{ __('Your current plan does not include guest uploads. Please upgrade to enable this feature.') }}</p>
                        </div>
                    @endif
                </div>

                 <!-- SHARE & DETAILS TAB -->
                <div data-tab-panel="details">
                     <div class="grid gap-6">
                        <div class="panel p-6">
                            <div class="flex flex-wrap items-start justify-between gap-4">
                                <div>
                                    <p class="panel-heading">{{ __('Event') }}</p>
                                    <h2 class="mt-2 text-xl font-semibold text-ink">{{ $event->name }}</h2>
                                    <p class="mt-1 text-sm text-muted">{{ __('Type:') }} {{ $event->type }}</p>
                                    <p class="mt-1 text-sm text-muted">{{ __('Event Date:') }} {{ \App\Support\FormatSettings::date($event->event_date) }}</p>
                                </div>
                                <div class="flex flex-wrap items-center gap-3">
                                    <a href="{{ route('admin.events.index') }}" class="btn-secondary">{{ __('Back') }}</a>
                                    <a href="{{ route('admin.events.edit', $event) }}" class="btn-primary">{{ __('Edit Event') }}</a>
                                </div>
                            </div>

                            <div class="mt-6 grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Expiry') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $event->expiry_date ? \App\Support\FormatSettings::date($event->expiry_date) : __('Not set') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Notifications') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $event->notifications_enabled ? __('Enabled') : __('Disabled') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Status') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $event->is_active ? __('Active') : __('Inactive') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Media') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $mediaCounts['total'] }} {{ __('total') }}</p>
                                </div>
                            </div>

                            @if ($event->details)
                                <div class="mt-6 rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Details') }}</p>
                                    <p class="mt-2 text-sm text-muted">{{ $event->details }}</p>
                                </div>
                            @endif

                            <div class="mt-6 grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Queue') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $mediaPipeline['queue_total'] }} {{ __('items') }}</p>
                                    <p class="mt-1 text-xs text-muted">{{ __('Pending') }} {{ $mediaPipeline['pending'] }} · {{ __('Processing') }} {{ $mediaPipeline['processing'] }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Ready') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $mediaPipeline['ready'] }} {{ __('items') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Failed') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $mediaPipeline['failed'] }} {{ __('items') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Storage Used') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $mediaPipeline['size_label'] }}</p>
                                </div>
                            </div>

                            <div class="mt-6 grid gap-4 lg:grid-cols-3">
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4 lg:col-span-2">
                                    <p class="panel-heading">{{ __('Share URL') }}</p>
                                    <div class="mt-2 flex flex-wrap items-center gap-3">
                                        <input type="text" value="{{ $shareUrl }}" readonly class="input-field flex-1" id="shareUrlInput">
                                        <button type="button" class="btn-secondary cursor-pointer" id="copyUrlBtn">{{ __('Copy URL') }}</button>
                                    </div>
                                    <div class="mt-3 flex flex-wrap items-center gap-3 text-xs text-muted">
                                        <a href="https://wa.me/?text={{ urlencode($shareUrl) }}" target="_blank" rel="noopener" class="btn-secondary">{{ __('WhatsApp') }}</a>
                                        <a href="https://t.me/share/url?url={{ urlencode($shareUrl) }}" target="_blank" rel="noopener" class="btn-secondary">{{ __('Telegram') }}</a>
                                        <a href="{{ $shareUrl }}" target="_blank" rel="noopener" class="btn-secondary">{{ __('Open Link') }}</a>
                                    </div>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4 text-center">
                                    <p class="panel-heading">{{ __('QR Code') }}</p>
                                    <div class="mt-3 flex items-center justify-center">
                                        <img src="https://api.qrserver.com/v1/create-qr-code/?size=160x160&data={{ urlencode($shareUrl) }}" alt="{{ __('QR code') }}" class="h-32 w-32">
                                    </div>
                                </div>
                            </div>

                            <div class="mt-6 grid gap-4 lg:grid-cols-3">
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Guest Pin') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $event->guest_pin ? __('Set') : __('Not set') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Admin Pin') }}</p>
                                    <p class="mt-2 text-sm font-semibold text-ink">{{ $event->admin_pin ? __('Set') : __('Not set') }}</p>
                                </div>
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4">
                                    <p class="panel-heading">{{ __('Hash Tags') }}</p>
                                    <div class="mt-2 flex flex-wrap gap-2">
                                        @forelse ($event->hashtags ?? [] as $tag)
                                            <span class="badge">{{ $tag }}</span>
                                        @empty
                                            <span class="text-sm text-muted">{{ __('No tags set.') }}</span>
                                        @endforelse
                                    </div>
                                </div>

                            </div>
                            @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_ftp_import'))
                                <div class="rounded-2xl border border-border/70 bg-surface-2/70 p-4 mt-2">
                                    <div class="flex items-center justify-between gap-3">
                                        <p class="panel-heading">{{ __('FTP Tethering') }}</p>
                                        <button type="button" class="text-xs font-medium text-brand hover:underline" x-data="" x-on:click.prevent="$dispatch('open-modal', 'ftp-how-it-works')">{{ __('How it works?') }}</button>
                                    </div>
                                    <p class="mt-2 text-sm font-semibold text-ink break-all">{{ $ftpBasePath }}/{{ $event->id }}</p>
                                    <p class="mt-2 text-xs text-muted">{{ __('Photos uploaded to this folder are automatically imported every 2 minutes.') }}</p>
                                    <form method="POST" action="{{ route('admin.events.media.import', $event) }}" class="mt-3">
                                        @csrf
                                        <button type="submit" class="btn-secondary">{{ __('Ingest FTP Now') }}</button>
                                    </form>
                                </div>

                                <x-modal name="ftp-how-it-works" :show="false" maxWidth="xl">
                                    <div class="p-6 space-y-6">
                                        <div>
                                            <h3 class="text-lg font-semibold text-ink">{{ __('How FTP Camera Tethering Works') }}</h3>
                                            <p class="mt-1 text-sm text-muted">{{ __('Shoot photos and have them appear in your event gallery automatically, no manual uploading needed.') }}</p>
                                        </div>

                                        <div class="space-y-5">
                                            <div class="flex gap-3">
                                                <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">1</span>
                                                <div>
                                                    <p class="text-sm font-semibold text-ink">{{ __('Connect your camera to the same Wi-Fi as your server') }}</p>
                                                    <p class="mt-1 text-xs text-muted">{{ __("Your camera and your hosting server must be on the same network. If you're at a venue, connect both to the venue's Wi-Fi or use a portable router / mobile hotspot.") }}</p>
                                                </div>
                                            </div>

                                            <div class="flex gap-3">
                                                <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">2</span>
                                                <div>
                                                    <p class="text-sm font-semibold text-ink">{{ __("Open your camera's FTP settings") }}</p>
                                                    <p class="mt-1 text-xs text-muted">{!! __('Go to your camera\'s <strong>Wi-Fi / Network</strong> menu and look for <strong>"FTP Transfer"</strong>, <strong>"Wireless Transmitter"</strong>, or <strong>"Auto Send"</strong>. This is available on most professional Canon, Nikon, and Sony cameras.') !!}</p>
                                                    <div class="mt-2 rounded-lg bg-surface/80 border border-border/60 p-3 text-xs text-muted space-y-1">
                                                        <p><strong>Canon:</strong> Menu &rarr; Communication &rarr; Built-in Wireless &rarr; FTP Transfer</p>
                                                        <p><strong>Nikon:</strong> Menu &rarr; Network &rarr; Connect to FTP Server</p>
                                                        <p><strong>Sony:</strong> Menu &rarr; Network &rarr; FTP Transfer Function</p>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="flex gap-3">
                                                <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">3</span>
                                                <div>
                                                    <p class="text-sm font-semibold text-ink">{{ __("Enter your server's FTP details") }}</p>
                                                    <p class="mt-1 text-xs text-muted">{{ __("Fill in the following in your camera's FTP configuration:") }}</p>
                                                    <div class="mt-2 rounded-lg bg-surface/80 border border-border/60 divide-y divide-border/60 text-xs">
                                                        <div class="flex justify-between gap-4 px-3 py-2">
                                                            <span class="text-muted">{{ __('Server / Host') }}</span>
                                                            <span class="font-semibold text-ink">{{ __('Your server IP or domain') }}</span>
                                                        </div>
                                                        <div class="flex justify-between gap-4 px-3 py-2">
                                                            <span class="text-muted">{{ __('Port') }}</span>
                                                            <span class="font-semibold text-ink">{{ __('21 (default)') }}</span>
                                                        </div>
                                                        <div class="flex justify-between gap-4 px-3 py-2">
                                                            <span class="text-muted">{{ __('Username') }}</span>
                                                            <span class="font-semibold text-ink">{{ __('Your FTP username') }}</span>
                                                        </div>
                                                        <div class="flex justify-between gap-4 px-3 py-2">
                                                            <span class="text-muted">{{ __('Password') }}</span>
                                                            <span class="font-semibold text-ink">{{ __('Your FTP password') }}</span>
                                                        </div>
                                                        <div class="flex justify-between gap-4 px-3 py-2">
                                                            <span class="text-muted">{{ __('Directory') }}</span>
                                                            <span class="font-semibold text-ink break-all">{{ $ftpBasePath }}/{{ $event->id }}</span>
                                                        </div>
                                                    </div>
                                                    <p class="mt-2 text-xs text-muted">{{ __('You can find your FTP credentials in your hosting control panel (cPanel, Plesk, etc).') }}</p>
                                                </div>
                                            </div>

                                            <div class="flex gap-3">
                                                <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">4</span>
                                                <div>
                                                    <p class="text-sm font-semibold text-ink">{{ __('Start shooting') }}</p>
                                                    <p class="mt-1 text-xs text-muted">{!! __('Every photo you take will be sent to the server automatically over Wi-Fi. Supported file types: <strong>JPG, JPEG, PNG, WEBP</strong>, and <strong>MP4</strong> videos.') !!}</p>
                                                </div>
                                            </div>

                                            <div class="flex gap-3">
                                                <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">5</span>
                                                <div>
                                                    <p class="text-sm font-semibold text-ink">{{ __('Photos appear in your gallery automatically') }}</p>
                                                    <p class="mt-1 text-xs text-muted">{!! __('The system checks for new FTP photos <strong>every 2 minutes</strong> and adds them to this event\'s gallery automatically. Imported files are moved to a <code class="text-xs">processed/</code> subfolder so they won\'t be imported twice.') !!}</p>
                                                    <p class="mt-1 text-xs text-muted">{!! __('You can also click <strong>"Ingest FTP Now"</strong> to import immediately without waiting.') !!}</p>
                                                </div>
                                            </div>
                                        </div>

                                        <div class="rounded-xl bg-brand/5 border border-brand/10 p-4 space-y-2">
                                            <p class="text-xs font-semibold text-brand">{{ __('Tips for best results') }}</p>
                                            <ul class="text-xs text-muted space-y-1 list-disc list-inside">
                                                <li>{!! __('Use a <strong>dedicated Wi-Fi router</strong> at the venue for stable, fast transfers.') !!}</li>
                                                <li>{!! __("If your camera doesn't have built-in FTP, use an app like <strong>ShutterSnitch</strong> (iOS) or <strong>CamFi</strong> on a laptop as an FTP bridge.") !!}</li>
                                                <li>{!! __('For <strong>multiple cameras</strong>, point them all to the same event folder &mdash; they\'ll all feed into the same gallery.') !!}</li>
                                                <li>{{ __('Guests will see photos appear on the event share page in real time.') }}</li>
                                            </ul>
                                        </div>

                                        <div class="flex justify-end">
                                            <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Got it') }}</x-secondary-button>
                                        </div>
                                    </div>
                                </x-modal>
                                @endif
                        </div>
                    </div>
                </div>


                <!-- SELECTIONS TAB -->
                <div data-tab-panel="selections">
                    <div class="space-y-6">
                        @php
                            $allSelections = \App\Models\ClientSelection::where('event_id', $event->id)
                                ->with('media:id,file_name,optimized_path,original_path,disk,meta')
                                ->orderByDesc('created_at')
                                ->limit(200)
                                ->get();
                            $selectionsByGuest = $allSelections->groupBy(fn($s) => $s->session_id);
                            $totalUnique = $allSelections->unique('event_media_id')->count();
                        @endphp

                        @if($allSelections->isEmpty())
                            <div class="panel p-8 text-center">
                                <p class="text-muted">{{ __('No client selections yet.') }}</p>
                                <p class="mt-2 text-sm text-muted">{{ __('Guests can select their favorite photos on the share page using the heart button.') }}</p>
                            </div>
                        @else
                            <div class="flex items-center justify-between mb-4">
                                <p class="panel-heading">{{ __('Client Selections') }}</p>
                                <span class="text-sm text-muted">{{ $totalUnique }} {{ __('unique photo(s) from') }} {{ $selectionsByGuest->count() }} {{ __('guest(s)') }}</span>
                            </div>

                            @foreach($selectionsByGuest as $sessionId => $guestSelections)
                            @php $first = $guestSelections->first(); @endphp
                            <div class="panel p-5 mb-4">
                                <div class="flex items-center justify-between mb-3">
                                    <div>
                                        <span class="text-sm font-semibold text-ink">{{ $first->guest_name ?: __('Guest') . ' #' . Str::limit($sessionId, 8) }}</span>
                                        @if($first->guest_email)
                                            <span class="ml-2 text-xs text-muted">({{ $first->guest_email }})</span>
                                        @endif
                                    </div>
                                    <span class="badge">{{ $guestSelections->count() }} {{ __('photos') }}</span>
                                </div>
                                <div class="grid grid-cols-3 gap-3 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8 xl:grid-cols-10">
                                    @foreach($guestSelections as $sel)
                                        @if($sel->media)
                                        <div class="group relative overflow-hidden rounded-xl border border-border/60 bg-surface-2/70">
                                            <div class="aspect-square overflow-hidden bg-surface-3">
                                                <img src="{{ $sel->media->preview_url }}" alt="{{ $sel->media->file_name }}" class="h-full w-full object-cover transition-transform duration-200 group-hover:scale-105" loading="lazy">
                                            </div>
                                        </div>
                                        @endif
                                    @endforeach
                                </div>
                            </div>
                            @endforeach
                        @endif
                    </div>
                </div>

                <!-- ANALYTICS TAB -->
                <div data-tab-panel="analytics">
                    <div class="space-y-6">
                        @php
                            $analytics = \App\Models\GalleryAnalytics::where('event_id', $event->id)
                                ->orderByDesc('date')
                                ->limit(30)
                                ->get();
                            $totals = [
                                'views' => $analytics->sum('views'),
                                'unique_visitors' => $analytics->sum('unique_visitors'),
                                'downloads' => $analytics->sum('downloads'),
                                'face_searches' => $analytics->sum('face_searches'),
                                'selections' => $analytics->sum('selections'),
                            ];
                            $topDownloaded = \App\Models\MediaAnalytics::whereHas('media', fn($q) => $q->where('event_id', $event->id))
                                ->orderByDesc('downloads')
                                ->with('media:id,file_name,optimized_path,original_path,disk,meta')
                                ->limit(5)
                                ->get();
                        @endphp

                        <div class="grid grid-cols-2 gap-4 sm:grid-cols-3 lg:grid-cols-5">
                            @foreach([
                                'views' => [__('Total Views'), 'text-brand'],
                                'unique_visitors' => [__('Unique Visitors'), 'text-purple-500'],
                                'downloads' => [__('Downloads'), 'text-green-500'],
                                'face_searches' => [__('Face Searches'), 'text-blue-500'],
                                'selections' => [__('Selections'), 'text-pink-500'],
                            ] as $key => [$label, $color])
                            <div class="panel p-4 text-center">
                                <div class="text-2xl font-bold {{ $color }}">{{ number_format($totals[$key]) }}</div>
                                <div class="text-xs text-muted">{{ $label }}</div>
                            </div>
                            @endforeach
                        </div>

                        @if($topDownloaded->isNotEmpty())
                        <div class="panel p-6">
                            <p class="panel-heading mb-4">{{ __('Most Downloaded') }}</p>
                            <div class="space-y-3">
                                @foreach($topDownloaded as $i => $td)
                                <div class="flex items-center gap-3">
                                    <span class="text-sm font-bold text-muted w-6">{{ $i + 1 }}.</span>
                                    @if($td->media)
                                    <div class="h-10 w-10 overflow-hidden rounded-lg bg-surface-3">
                                        <img src="{{ $td->media->preview_url }}" class="h-full w-full object-cover" loading="lazy">
                                    </div>
                                    <span class="text-sm text-ink flex-1 truncate">{{ $td->media->file_name }}</span>
                                    @endif
                                    <span class="text-sm font-semibold text-brand">{{ $td->downloads }} {{ __('downloads') }}</span>
                                </div>
                                @endforeach
                            </div>
                        </div>
                        @endif

                        @if($analytics->isEmpty())
                        <div class="panel p-8 text-center">
                            <p class="text-muted">{{ __('No analytics data yet. Share the gallery to start tracking.') }}</p>
                        </div>
                        @endif
                    </div>
                </div>

                <!-- SETTINGS TAB -->
                <div data-tab-panel="settings">
                    <div class="panel p-6 max-w-2xl">
                        <h3 class="text-lg font-semibold text-ink mb-6">{{ __('General Settings') }}</h3>

                        <form method="POST" action="{{ route('admin.events.update', $event) }}" class="space-y-6">
                            @csrf
                            @method('PUT')

                            <!-- Pass through required fields from existing event to satisfy validation -->
                            <input type="hidden" name="name" value="{{ $event->name }}">
                            <input type="hidden" name="type" value="{{ $event->type }}">
                            <input type="hidden" name="event_date" value="{{ $event->event_date->format('Y-m-d') }}">

                            <div class="space-y-4">
                                <!-- Public Downloads Toggle -->
                                <div class="flex items-center justify-between p-4 rounded-2xl bg-surface-2/50 border border-border/50">
                                    <div class="pr-4">
                                        <label for="allow_public_downloads" class="block text-sm font-semibold text-ink">{{ __('Allow Public Downloads') }}</label>
                                        <p class="text-xs text-muted mt-1">{{ __('If enabled, everyone can see all photos and download them directly without a face search/selfie requirement.') }}</p>
                                    </div>
                                    <div class="flex items-center">
                                        <input type="checkbox" id="allow_public_downloads" name="allow_public_downloads" value="1"
                                            class="h-5 w-5 rounded border-border text-brand focus:ring-brand/40"
                                            {{ $event->allow_public_downloads ? 'checked' : '' }}>
                                    </div>
                                </div>

                                <!-- Guest Upload Toggle -->
                                @if(app(\App\Services\PlanLimitService::class)->hasFeature(auth()->user(), 'has_guest_upload'))
                                <div class="flex items-center justify-between p-4 rounded-2xl bg-surface-2/50 border border-border/50">
                                    <div class="pr-4">
                                        <label for="allow_guest_upload" class="block text-sm font-semibold text-ink">{{ __('Allow Guest Uploads') }}</label>
                                        <p class="text-xs text-muted mt-1">{{ __('Allow guests to upload their own media to this event.') }}</p>
                                    </div>
                                    <div class="flex items-center">
                                        <input type="checkbox" id="allow_guest_upload" name="allow_guest_upload" value="1"
                                            class="h-5 w-5 rounded border-border text-brand focus:ring-brand/40"
                                            {{ $event->allow_guest_upload ? 'checked' : '' }}>
                                    </div>
                                </div>
                                @endif

                                <!-- Active Toggle -->
                                <div class="flex items-center justify-between p-4 rounded-2xl bg-surface-2/50 border border-border/50">
                                    <div class="pr-4">
                                        <label for="is_active" class="block text-sm font-semibold text-ink">{{ __('Event Active') }}</label>
                                        <p class="text-xs text-muted mt-1">{{ __("If inactive, the gallery links will show an 'inactive' message.") }}</p>
                                    </div>
                                    <div class="flex items-center">
                                        <input type="checkbox" id="is_active" name="is_active" value="1"
                                            class="h-5 w-5 rounded border-border text-brand focus:ring-brand/40"
                                            {{ $event->is_active ? 'checked' : '' }}>
                                    </div>
                                </div>
                            </div>

                            <div class="flex justify-end pt-4 border-t border-border/50">
                                <button type="submit" class="btn-primary px-8">{{ __('Save Settings') }}</button>
                            </div>
                        </form>
                    </div>
                </div>

            </div>
        </div>
    </div>

    {{-- Upload How it works modal --}}
    <x-modal name="upload-how-it-works" :show="false" maxWidth="xl">
        <div class="p-6 space-y-6">
            <div>
                <h3 class="text-lg font-semibold text-ink">{{ __('How Photo Uploading Works') }}</h3>
                <p class="mt-1 text-sm text-muted">{{ __("Upload photos and they'll be processed automatically in the background.") }}</p>
            </div>

            <div class="space-y-5">
                <div class="flex gap-3">
                    <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">1</span>
                    <div>
                        <p class="text-sm font-semibold text-ink">{{ __('Choose your upload method') }}</p>
                        <p class="mt-1 text-xs text-muted">{!! __('<strong>Upload Files</strong> — Best for small batches (up to 50 files). Select multiple images or videos at once.') !!}</p>
                        <p class="mt-1 text-xs text-muted">{!! __('<strong>ZIP Upload</strong> — Recommended for bulk (100-5000+ photos). Zip your folder and upload it in one go. Much faster than individual files.') !!}</p>
                    </div>
                </div>

                <div class="flex gap-3">
                    <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">2</span>
                    <div>
                        <p class="text-sm font-semibold text-ink">{{ __('Files upload instantly') }}</p>
                        <p class="mt-1 text-xs text-muted">{!! __("Your photos are uploaded to the server immediately. You'll see a progress bar during upload. <strong>Keep the tab open</strong> until uploading completes.") !!}</p>
                    </div>
                </div>

                <div class="flex gap-3">
                    <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">3</span>
                    <div>
                        <p class="text-sm font-semibold text-ink">{{ __('Background processing begins') }}</p>
                        <p class="mt-1 text-xs text-muted">{{ __('After upload, each photo is optimized in the background — resizing, generating thumbnails, and applying watermarks if enabled.') }}</p>
                        <div class="mt-2 rounded-lg bg-surface/80 border border-border/60 p-3 text-xs text-muted space-y-1">
                            @if($watermarkEnabled)
                                <p>{!! __('<strong>Watermark ON:</strong> ~1-2 sec per photo') !!}</p>
                                <p>{{ __('100 photos') }} &asymp; {{ __('2 min') }} &middot; {{ __('1000 photos') }} &asymp; {{ __('20 min') }}</p>
                            @else
                                <p>{!! __('<strong>Watermark OFF:</strong> Near-instant processing') !!}</p>
                            @endif
                        </div>
                    </div>
                </div>

                <div class="flex gap-3">
                    <span class="flex h-7 w-7 shrink-0 items-center justify-center rounded-full bg-brand/10 text-xs font-bold text-brand">4</span>
                    <div>
                        <p class="text-sm font-semibold text-ink">{{ __('Page reloads when ready') }}</p>
                        <p class="mt-1 text-xs text-muted">{!! __('A processing banner will appear showing progress. Once all photos are optimized and active, the page <strong>reloads automatically</strong> so you can see the final results.') !!}</p>
                    </div>
                </div>
            </div>

            <div class="rounded-xl bg-brand/5 border border-brand/10 p-4 space-y-2">
                <p class="text-xs font-semibold text-brand">{{ __('Tips for best results') }}</p>
                <ul class="text-xs text-muted space-y-1 list-disc list-inside">
                    <li>{!! __('Use <strong>JPG, JPEG, PNG, WEBP</strong> for images and <strong>MP4</strong> for videos.') !!}</li>
                    <li>{!! __('For large batches (100+ files), always use <strong>ZIP Upload</strong> for the fastest experience.') !!}</li>
                    <li>{!! __('Enable <strong>Chunked mode</strong> for individual files larger than 50 MB.') !!}</li>
                    <li>{{ __('Guests will see photos appear on the event share page once processing is complete.') }}</li>
                </ul>
            </div>

            <div class="flex justify-end">
                <x-secondary-button type="button" x-on:click="$dispatch('close')">{{ __('Got it') }}</x-secondary-button>
            </div>
        </div>
    </x-modal>
    {{-- Processing overlay (hidden by default, shown after upload via JS) --}}
    <div class="fixed inset-0 z-[100000] hidden flex-col items-center justify-center bg-surface/80 backdrop-blur-md transition-opacity" id="media-processing-overlay">
        <div class="mx-auto w-full max-w-md bg-surface p-6 sm:p-8 rounded-3xl border border-border shadow-2xl relative overflow-hidden">
            <div class="absolute top-0 left-0 w-full h-1 bg-border/50 overflow-hidden">
                <div class="h-full bg-brand rounded-full transition-all duration-300" id="media-processing-top-bar" style="width: 0%"></div>
            </div>
            <div class="flex items-center gap-5">
                <div class="flex h-12 w-12 shrink-0 items-center justify-center rounded-2xl bg-brand/10 text-brand">
                    <svg class="h-6 w-6 animate-spin text-brand" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                </div>
                <div class="flex-1">
                    <p class="text-base font-semibold text-ink">{{ __('Processing Media') }}</p>
                    <p class="mt-1 text-sm text-muted" id="media-processing-status">{{ __('Optimizing your photos. Please wait...') }}</p>
                </div>
            </div>
            <div class="mt-6 flex flex-col items-center space-y-2">
                 <div class="w-full h-2 rounded-full bg-border/70 overflow-hidden shadow-inner">
                    <div class="h-full bg-brand rounded-full transition-all duration-500 relative" id="media-processing-bar" style="width: 0%">
                        <div class="absolute inset-0 bg-white/20 w-full h-full animate-pulse"></div>
                    </div>
                </div>
                <p class="text-xs text-muted font-medium w-full text-center tracking-widest" id="media-processing-progress-text">0%</p>
            </div>
            <p class="text-[11px] text-center text-muted uppercase tracking-wider mt-5 font-bold">{{ __('Please do not close or refresh this window') }}</p>
        </div>
    </div>
</x-admin-layout>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {

    // Copy URL functionality
    const copyUrlBtn = document.getElementById('copyUrlBtn');
    const shareUrlInput = document.getElementById('shareUrlInput');

    if (copyUrlBtn && shareUrlInput) {
        copyUrlBtn.addEventListener('click', async () => {
            const url = shareUrlInput.value;
            let copied = false;

            // Try modern clipboard API (requires HTTPS)
            if (window.isSecureContext && navigator.clipboard?.writeText) {
                try {
                    await navigator.clipboard.writeText(url);
                    copied = true;
                } catch { /* fall through to fallback */ }
            }

            // Fallback: temporary textarea works on HTTP and all browsers
            if (!copied) {
                const textarea = document.createElement('textarea');
                textarea.value = url;
                textarea.style.cssText = 'position:fixed;opacity:0;left:-9999px';
                document.body.appendChild(textarea);
                textarea.select();
                textarea.setSelectionRange(0, 99999);
                document.execCommand('copy');
                document.body.removeChild(textarea);
            }

            const origText = copyUrlBtn.textContent;
            copyUrlBtn.textContent = 'Copied!';
            setTimeout(() => { copyUrlBtn.textContent = origText; }, 2000);
        });
    }

    // Drive imports functionality
    const panel = document.getElementById('drive-imports-panel');
    if (!panel) return;
    const statusUrl = panel.getAttribute('data-status-url');
    const list = document.getElementById('drive-import-list');
    const refreshBtn = document.getElementById('drive-import-refresh');

    async function refreshImports() {
        try {
            const response = await fetch(statusUrl, { headers: { 'X-Requested-With': 'XMLHttpRequest' } });
            if (!response.ok) return;
            const data = await response.json();
            if (!data.imports) return;
            renderImports(data.imports);
        } catch (e) {
            console.warn('Drive import refresh failed', e);
        }
    }

    function renderImports(imports) {
        if (!list) return;
        list.innerHTML = '';
        imports.forEach(row => {
            const total = row.total_files || 0;
            const done = row.processed_files || 0;
            const failed = row.failed_files || 0;
            const remaining = Math.max(0, total - done - failed);
            const percent = total > 0 ? Math.round((done / total) * 100) : 0;

            const wrapper = document.createElement('div');
            wrapper.className = 'rounded-xl border border-border/60 bg-surface-2/70 p-3 drive-import-row';
            wrapper.dataset.importId = row.id;
            wrapper.innerHTML = `
                <div class="flex items-center justify-between text-sm">
                    <div class="flex items-center gap-2">
                        <span class="text-xs font-semibold text-muted">#${row.id}</span>
                        <span class="badge">${(row.status || '').charAt(0).toUpperCase() + (row.status || '').slice(1)}</span>
                    </div>
                    <span class="text-xs text-muted">${done} / ${total} processed · ${failed} failed</span>
                </div>
                <div class="mt-2 h-2 rounded-full bg-border/70 overflow-hidden">
                    <div class="h-2 bg-brand rounded-full transition-all" style="width:${percent}%;"></div>
                </div>
                <div class="mt-2 text-xs text-muted">
                    Remaining: <span data-import-remaining>${remaining}</span>
                    ${row.last_error ? `<span class="text-danger ml-2">${row.last_error}</span>` : ''}
                </div>
            `;
            list.appendChild(wrapper);
        });
    }

    refreshBtn?.addEventListener('click', refreshImports);
    refreshImports();
    setInterval(refreshImports, 8000);
});
</script>

{{-- Auto queue processor: processes pending media jobs via AJAX, no cron needed --}}
<script>
(function() {
    // Second reload: after processing completes, status can lag — reload once more after 10s
    if (sessionStorage.getItem('media_processing_reload') === '1') {
        sessionStorage.removeItem('media_processing_reload');
        setTimeout(() => window.location.reload(), 10000);
    }

    const queueUrl = @json(route('admin.queue.process'));
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
    let running = false;
    let totalProcessed = 0;
    let errorCount = 0;
    const maxErrors = 20;
    let queueActive = false;

    function getOverlayElements() {
        return {
            overlay: document.getElementById('media-processing-overlay'),
            statusEl: document.getElementById('media-processing-status'),
            barEl: document.getElementById('media-processing-bar'),
            topBarEl: document.getElementById('media-processing-top-bar'),
            progressTextEl: document.getElementById('media-processing-progress-text')
        };
    }

    function showBanner() {
        const { overlay } = getOverlayElements();
        if (overlay) {
            overlay.classList.remove('hidden');
            overlay.classList.add('flex');
            document.body.style.overflow = 'hidden'; 
        }
    }

    function updateBanner(processed, remaining) {
        const { statusEl, barEl, topBarEl, progressTextEl } = getOverlayElements();
        const total = processed + remaining;
        const pct = total > 0 ? Math.round((processed / total) * 100) : 0;
        
        if (statusEl) statusEl.textContent = `${processed} processed, ${remaining} remaining.`;
        if (barEl) barEl.style.width = pct + '%';
        if (topBarEl) topBarEl.style.width = pct + '%';
        if (progressTextEl) progressTextEl.textContent = pct + '%';
    }

    async function processQueue() {
        if (running) return;
        if (errorCount >= maxErrors) {
            const { statusEl: el } = getOverlayElements();
            if (el) el.textContent = 'Too many errors. Please refresh the page to retry.';
            return;
        }
        running = true;
        try {
            const res = await fetch(queueUrl, {
                method: 'POST',
                headers: { 'Accept': 'application/json', 'X-CSRF-TOKEN': csrfToken },
            });
            if (!res.ok) {
                errorCount++;
                running = false;
                setTimeout(processQueue, 5000);
                return;
            }
            const data = await res.json();
            errorCount = 0;
            totalProcessed += data.processed || 0;
            updateBanner(totalProcessed, data.remaining || 0);
            if (data.remaining > 0) {
                running = false;
                setTimeout(processQueue, 300);
            } else {
                updateBanner(totalProcessed, 0);
                const { statusEl: doneEl } = getOverlayElements();
                if (doneEl) doneEl.textContent = `All ${totalProcessed} photos processed! Reloading...`;
                // Mark that a second reload is needed (status update can lag)
                sessionStorage.setItem('media_processing_reload', '1');
                setTimeout(() => window.location.reload(), 1000);
            }
        } catch (e) {
            errorCount++;
            running = false;
            setTimeout(processQueue, 10000);
            return;
        }
        running = false;
    }

    // Expose globally so upload handlers can trigger it
    window.startMediaProcessing = function() {
        if (queueActive) return;
        queueActive = true;
        totalProcessed = 0;
        errorCount = 0;
        showBanner();
        processQueue();
    };

    // Auto-start if there are pending items on page load
    @if(($mediaPipeline['pending'] ?? 0) + ($mediaPipeline['processing'] ?? 0) > 0)
        window.startMediaProcessing();
    @endif
})();
</script>
@endpush
