@extends('layouts.frontend')

@section('title', __('Checkout') . ' - ' . $plan->name)
@section('hide_header', true)

@section('content')
<main class="flex-1 pt-32 pb-20 overflow-hidden relative" x-data="checkoutForm()">
    <!-- Ambient Background -->
    <div class="absolute top-0 right-0 w-[50%] h-[50%] bg-primary/5 blur-[120px] rounded-full -z-10"></div>
    <div class="absolute bottom-0 left-0 w-[50%] h-[50%] bg-accent/5 blur-[120px] rounded-full -z-10"></div>

    <div class="container mx-auto px-4">
        <div class="max-w-5xl mx-auto">
            <!-- Header -->
            <div class="text-center mb-10 space-y-3">
                <h1
                    x-data="{ shown: false }"
                    x-init="setTimeout(() => shown = true, 0)"
                    class="text-3xl md:text-4xl font-bold tracking-tight text-foreground transition-all duration-700 transform"
                    :class="shown ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-5'">
                    {{ __('Complete Your') }} <span class="text-primary italic">{{ __('Order') }}</span>
                </h1>
                <p class="text-base text-muted-foreground max-w-xl mx-auto">
                    {{ __('You are subscribing to the') }} <strong>{{ $plan->name }}</strong> {{ __('plan. Complete payment to get started.') }}
                </p>
            </div>

            <div class="grid grid-cols-1 lg:grid-cols-5 gap-8 items-start">
                <!-- Left: Payment Form (3 cols) -->
                <div class="lg:col-span-3">
                    <form method="POST" action="{{ route('checkout.process', $plan->slug) }}" id="checkout-form"
                          class="space-y-6"
                          x-on:submit="submitting = true">
                        @csrf
                        <input type="hidden" name="selected_currency" :value="selectedCurrency" />

                        <!-- Currency Selection -->
                        @if($currencies->count() > 1)
                        <div
                            x-data="{ shown: false }"
                            x-init="setTimeout(() => shown = true, 50)"
                            class="relative z-20 p-6 rounded-2xl bg-card border border-border shadow-lg transition-all duration-700 transform"
                            :class="shown ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-5'">

                            <h3 class="text-base font-bold mb-4 flex items-center gap-3">
                                <div class="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M16 8h-6a2 2 0 1 0 0 4h4a2 2 0 1 1 0 4H8"/><path d="M12 18V6"/></svg>
                                </div>
                                {{ __('Select Currency') }}
                            </h3>

                            <!-- Searchable Currency Dropdown -->
                            <div class="relative" x-data="currencySearch()" @click.outside="open = false" @keydown.escape.window="open = false">
                                <!-- Selected value display / trigger -->
                                <button
                                    type="button"
                                    @click="toggle()"
                                    class="flex items-center justify-between h-14 w-full rounded-2xl border border-input bg-muted/20 px-6 py-1 text-base shadow-sm transition-colors focus-visible:outline-none focus:bg-background focus-visible:ring-1 focus-visible:ring-ring md:text-sm font-medium cursor-pointer hover:border-primary/50">
                                    <span class="flex items-center gap-2">
                                        <span class="text-lg" x-text="currentSymbol"></span>
                                        <span x-text="selectedCurrency"></span>
                                        <span class="text-muted-foreground">-</span>
                                        <span class="text-muted-foreground" x-text="currentName"></span>
                                    </span>
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-muted-foreground transition-transform" :class="open ? 'rotate-180' : ''"><path d="m6 9 6 6 6-6"/></svg>
                                </button>

                                <!-- Dropdown panel -->
                                <div
                                    x-show="open"
                                    x-transition:enter="transition ease-out duration-150"
                                    x-transition:enter-start="opacity-0 -translate-y-2"
                                    x-transition:enter-end="opacity-100 translate-y-0"
                                    x-transition:leave="transition ease-in duration-100"
                                    x-transition:leave-start="opacity-100 translate-y-0"
                                    x-transition:leave-end="opacity-0 -translate-y-2"
                                    class="absolute z-50 mt-2 w-full rounded-2xl border border-border bg-card shadow-xl overflow-hidden"
                                    x-cloak>

                                    <!-- Search input -->
                                    <div class="p-3 border-b border-border/50">
                                        <div class="relative">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground"><circle cx="11" cy="11" r="8"/><path d="m21 21-4.3-4.3"/></svg>
                                            <input
                                                type="text"
                                                x-model="search"
                                                x-ref="searchInput"
                                                @keydown.arrow-down.prevent="highlightNext()"
                                                @keydown.arrow-up.prevent="highlightPrev()"
                                                @keydown.enter.prevent="selectHighlighted()"
                                                placeholder="{{ __('Search currencies...') }}"
                                                class="w-full h-10 rounded-xl border border-input bg-muted/20 pl-10 pr-4 text-sm focus:outline-none focus:ring-1 focus:ring-ring"
                                            />
                                        </div>
                                    </div>

                                    <!-- Options list -->
                                    <div class="max-h-64 overflow-y-auto overscroll-contain" x-ref="listbox">
                                        <template x-for="(cur, index) in filtered" :key="cur.code">
                                            <button
                                                type="button"
                                                @click="selectCurrency(cur.code)"
                                                @mouseenter="highlighted = index"
                                                class="w-full flex items-center gap-3 px-4 py-3 text-sm transition-colors cursor-pointer text-left"
                                                :class="{
                                                    'bg-primary/10 text-primary': selectedCurrency === cur.code,
                                                    'bg-muted/50': highlighted === index && selectedCurrency !== cur.code,
                                                    'hover:bg-muted/30': highlighted !== index && selectedCurrency !== cur.code
                                                }">
                                                <span class="w-8 text-center text-lg shrink-0" x-text="cur.symbol"></span>
                                                <span class="font-semibold shrink-0" x-text="cur.code"></span>
                                                <span class="text-muted-foreground truncate" x-text="cur.name"></span>
                                                <svg x-show="selectedCurrency === cur.code" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="ml-auto text-primary shrink-0"><path d="M20 6 9 17l-5-5"/></svg>
                                            </button>
                                        </template>
                                        <div x-show="filtered.length === 0" class="px-4 py-6 text-center text-sm text-muted-foreground">
                                            {{ __('No currencies found') }}
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Exchange rate info -->
                            <div x-show="selectedCurrency !== '{{ $currency }}'" x-cloak class="mt-3 flex items-center gap-2 text-xs text-muted-foreground">
                                <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>
                                <span>{{ __('Prices converted using approximate exchange rates. Final charge may vary slightly.') }}</span>
                            </div>
                        </div>
                        @endif

                        <!-- Coupon Code -->
                        <div
                            x-data="{ shown: false }"
                            x-init="setTimeout(() => shown = true, 100)"
                            class="relative z-10 p-6 rounded-2xl bg-card border border-border shadow-lg transition-all duration-700 transform"
                            :class="shown ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-5'">

                            <h3 class="text-base font-bold mb-4 flex items-center gap-3">
                                <div class="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21.801 10A10 10 0 1 1 17 3.335"/><path d="m9 11 3 3L22 4"/></svg>
                                </div>
                                {{ __('Have a coupon?') }}
                            </h3>

                            <div class="flex gap-3">
                                <div class="flex-1">
                                    <input
                                        type="text"
                                        name="coupon_code"
                                        x-model="couponCode"
                                        placeholder="{{ __('Enter coupon code') }}"
                                        class="flex h-14 w-full rounded-2xl border border-input bg-muted/20 px-6 py-1 text-base shadow-sm transition-colors placeholder:text-muted-foreground focus-visible:outline-none focus:bg-background focus-visible:ring-1 focus-visible:ring-ring md:text-sm"
                                        :disabled="couponApplied"
                                    />
                                </div>
                                <button
                                    type="button"
                                    @click="couponApplied ? removeCoupon() : applyCoupon()"
                                    :disabled="couponLoading || (!couponCode && !couponApplied)"
                                    class="h-14 px-6 rounded-2xl font-bold text-sm transition-all disabled:opacity-50 disabled:cursor-not-allowed"
                                    :class="couponApplied ? 'bg-red-500/10 text-red-500 hover:bg-red-500/20 border border-red-500/20' : 'bg-primary/10 text-primary hover:bg-primary/20 border border-primary/20'">
                                    <span x-show="couponLoading" class="flex items-center gap-2">
                                        <svg class="animate-spin h-4 w-4" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>
                                        {{ __('Checking...') }}
                                    </span>
                                    <span x-show="!couponLoading && !couponApplied">{{ __('Apply') }}</span>
                                    <span x-show="!couponLoading && couponApplied">{{ __('Remove') }}</span>
                                </button>
                            </div>

                            <!-- Coupon feedback -->
                            <div x-show="couponMessage" x-cloak class="mt-3">
                                <p :class="couponApplied ? 'text-green-600 dark:text-green-400' : 'text-red-500'" class="text-sm font-medium" x-text="couponMessage"></p>
                            </div>

                            @error('coupon_code')
                                <p class="text-red-500 text-sm font-medium mt-2">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Payment Gateway Selection -->
                        <div
                            x-data="{ shown: false }"
                            x-init="setTimeout(() => shown = true, 200)"
                            class="p-6 rounded-2xl bg-card border border-border shadow-lg transition-all duration-700 transform"
                            :class="shown ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-5'">

                            <h3 class="text-base font-bold mb-4 flex items-center gap-3">
                                <div class="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="20" height="14" x="2" y="5" rx="2"/><line x1="2" x2="22" y1="10" y2="10"/></svg>
                                </div>
                                {{ __('Payment Method') }}
                            </h3>

                            <!-- No gateways for selected currency -->
                            <div x-show="filteredGateways.length === 0" class="text-center py-8 text-muted-foreground">
                                <p class="font-medium">{{ __('No payment methods available for the selected currency.') }}</p>
                                <p class="text-sm mt-1">{{ __('Please select a different currency or contact support.') }}</p>
                            </div>

                            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                @foreach($gateways as $gw)
                                    <label
                                        x-show="isGatewayVisible('{{ $gw->slug }}')"
                                        x-transition
                                        class="relative flex items-center gap-4 p-5 rounded-2xl border-2 cursor-pointer transition-all hover:shadow-md group"
                                        :class="selectedGateway === '{{ $gw->slug }}' ? 'border-primary bg-primary/5 shadow-md' : 'border-border hover:border-primary/30'">
                                        <input
                                            type="radio"
                                            name="gateway"
                                            value="{{ $gw->slug }}"
                                            x-model="selectedGateway"
                                            class="sr-only"
                                        />
                                        <div class="h-10 w-10 rounded-xl bg-muted/50 flex items-center justify-center shrink-0 group-hover:bg-muted transition-colors">
                                            @if($gw->slug === 'stripe')
                                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="20" height="14" x="2" y="5" rx="2"/><line x1="2" x2="22" y1="10" y2="10"/></svg>
                                            @elseif($gw->slug === 'paypal')
                                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M16 8h-6a2 2 0 1 0 0 4h4a2 2 0 1 1 0 4H8"/><path d="M12 18V6"/></svg>
                                            @else
                                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 12V7H5a2 2 0 0 1 0-4h14v4"/><path d="M3 5v14a2 2 0 0 0 2 2h16v-5"/><path d="M18 12a2 2 0 0 0 0 4h4v-4Z"/></svg>
                                            @endif
                                        </div>
                                        <div>
                                            <span class="font-bold text-sm">{{ $gw->name }}</span>
                                            @if($gw->description)
                                                <p class="text-xs text-muted-foreground mt-0.5">{{ $gw->description }}</p>
                                            @endif
                                        </div>
                                        <!-- Check indicator -->
                                        <div
                                            class="absolute top-3 right-3 h-5 w-5 rounded-full flex items-center justify-center transition-all"
                                            :class="selectedGateway === '{{ $gw->slug }}' ? 'bg-primary' : 'bg-muted'">
                                            <svg x-show="selectedGateway === '{{ $gw->slug }}'" xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="text-primary-foreground"><path d="M20 6 9 17l-5-5"/></svg>
                                        </div>
                                    </label>
                                @endforeach
                            </div>

                            @error('gateway')
                                <p class="text-red-500 text-sm font-medium mt-4">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Submit -->
                        <button
                            type="submit"
                            :disabled="submitting || !selectedGateway || filteredGateways.length === 0"
                            class="w-full h-14 rounded-2xl bg-primary hover:bg-primary/90 text-base font-bold shadow-xl shadow-primary/20 text-primary-foreground flex items-center justify-center transition-all disabled:opacity-50 disabled:cursor-not-allowed group">
                            <span x-show="!submitting" class="flex items-center gap-2">
                                Pay <span x-text="displayTotal"></span>
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="group-hover:translate-x-1 transition-transform"><path d="M5 12h14"/><path d="m12 5 7 7-7 7"/></svg>
                            </span>
                            <span x-show="submitting" class="flex items-center gap-2">
                                <svg class="animate-spin h-5 w-5" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>
                                {{ __('Processing...') }}
                            </span>
                        </button>
                    </form>
                </div>

                <!-- Right: Order Summary (2 cols) -->
                <div class="lg:col-span-2">
                    <div
                        x-data="{ shown: false }"
                        x-init="setTimeout(() => shown = true, 300)"
                        class="p-6 rounded-2xl bg-card border border-border shadow-lg sticky top-28 transition-all duration-700 transform"
                        :class="shown ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-5'">

                        <h3 class="text-base font-bold mb-4 flex items-center gap-3">
                            <div class="h-7 w-7 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M6 2 3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4Z"/><path d="M3 6h18"/><path d="M16 10a4 4 0 0 1-8 0"/></svg>
                            </div>
                            {{ __('Order Summary') }}
                        </h3>

                        <!-- Plan info -->
                        <div class="pb-4 mb-4 border-b border-border/50">
                            <div class="flex items-center justify-between mb-2">
                                <span class="font-bold text-lg">{{ $plan->name }}</span>
                                @if($plan->duration_months == 1)
                                    <span class="text-xs font-bold uppercase tracking-wider text-muted-foreground bg-muted px-2 py-1 rounded-lg">{{ __('Monthly') }}</span>
                                @elseif($plan->duration_months == 12)
                                    <span class="text-xs font-bold uppercase tracking-wider text-muted-foreground bg-muted px-2 py-1 rounded-lg">{{ __('Yearly') }}</span>
                                @elseif($plan->duration_type === 'lifetime')
                                    <span class="text-xs font-bold uppercase tracking-wider text-muted-foreground bg-muted px-2 py-1 rounded-lg">{{ __('Lifetime') }}</span>
                                @elseif($plan->duration_months)
                                    <span class="text-xs font-bold uppercase tracking-wider text-muted-foreground bg-muted px-2 py-1 rounded-lg">{{ $plan->duration_months }} {{ __('Months') }}</span>
                                @endif
                            </div>

                            <!-- Features list -->
                            @php
                                $planFeatures = is_array($plan->features) ? $plan->features : [];
                                $summaryFeatures = array_slice($planFeatures, 0, 4);
                            @endphp
                            @if(count($summaryFeatures) > 0)
                                <div class="space-y-2 mt-4">
                                    @foreach($summaryFeatures as $feature)
                                        <div class="flex items-center gap-2">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="text-primary shrink-0"><path d="M20 6 9 17l-5-5"/></svg>
                                            <span class="text-xs text-muted-foreground font-medium">{{ $feature }}</span>
                                        </div>
                                    @endforeach
                                    @if(count($planFeatures) > 4)
                                        <p class="text-xs text-muted-foreground/60 font-medium ml-5">+{{ count($planFeatures) - 4 }} {{ __('more features') }}</p>
                                    @endif
                                </div>
                            @endif

                            <!-- Limits summary -->
                            <div class="flex flex-wrap gap-2 mt-4">
                                @if($plan->events_limit)
                                    <span class="text-[10px] font-bold uppercase tracking-wider text-primary bg-primary/10 px-2 py-1 rounded-lg border border-primary/20">{{ $plan->events_limit }} {{ __('Events') }}</span>
                                @endif
                                @if($plan->images_limit)
                                    <span class="text-[10px] font-bold uppercase tracking-wider text-primary bg-primary/10 px-2 py-1 rounded-lg border border-primary/20">{{ number_format($plan->images_limit) }} {{ __('Images') }}</span>
                                @endif
                                @if($plan->storage_limit_gb)
                                    <span class="text-[10px] font-bold uppercase tracking-wider text-primary bg-primary/10 px-2 py-1 rounded-lg border border-primary/20">{{ rtrim(rtrim(number_format($plan->storage_limit_gb, 2), '0'), '.') }} GB</span>
                                @endif
                            </div>
                        </div>

                        <!-- Price breakdown -->
                        <div class="space-y-4">
                            <div class="flex items-center justify-between">
                                <span class="text-sm text-muted-foreground">{{ __('Subtotal') }}</span>
                                <span class="text-sm font-semibold" x-text="displaySubtotal"></span>
                            </div>

                            @if($plan->offer_price !== null && (float) $plan->offer_price < (float) $plan->price)
                                <div class="flex items-center justify-between text-green-600 dark:text-green-400">
                                    <span class="text-sm">{{ __('Plan discount') }}</span>
                                    <span class="text-sm font-semibold" x-text="'-' + displayPlanDiscount"></span>
                                </div>
                            @endif

                            <div x-show="couponApplied" x-cloak class="flex items-center justify-between text-green-600 dark:text-green-400">
                                <span class="text-sm">{{ __('Coupon discount') }}</span>
                                <span class="text-sm font-semibold" x-text="'-' + displayCouponDiscount"></span>
                            </div>

                            <div class="pt-4 border-t border-border/50 flex items-center justify-between">
                                <span class="font-bold text-base">{{ __('Total') }}</span>
                                <span class="text-2xl font-bold text-primary" x-text="displayTotal"></span>
                            </div>
                        </div>

                        <!-- Security badges -->
                        <div class="mt-6 pt-4 border-t border-border/50">
                            <div class="flex items-center gap-3 text-muted-foreground/60">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="shrink-0"><rect width="18" height="11" x="3" y="11" rx="2" ry="2"/><path d="M7 11V7a5 5 0 0 1 10 0v4"/></svg>
                                <span class="text-[10px] font-bold uppercase tracking-widest">{{ __('SSL Encrypted & Secure Payment') }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<script>
function checkoutForm() {
    return {
        selectedCurrency: '{{ $currency }}',
        selectedGateway: '',
        couponCode: '{{ old('coupon_code', '') }}',
        couponApplied: false,
        couponLoading: false,
        couponMessage: '',
        couponDiscountRaw: 0,
        submitting: false,

        // Base amounts in the default (store) currency
        basePrice: {{ (float) $plan->price }},
        baseEffectivePrice: {{ (float) $effectivePrice }},
        basePlanDiscount: {{ ($plan->offer_price !== null && (float) $plan->offer_price < (float) $plan->price) ? (float) $plan->price - (float) $plan->offer_price : 0 }},

        // Currency data: { CODE: { symbol, rate, precision } }
        currencyData: {!! \Illuminate\Support\Js::from($currencyData) !!},

        // Default currency rate (store currency relative to USD)
        defaultRate: {{ $defaultRate }},

        // Gateway slug -> supported currency codes
        gatewayCurrencies: {!! \Illuminate\Support\Js::from($gatewayCurrencies) !!},

        // Currency list for searchable dropdown
        currencyList: {!! \Illuminate\Support\Js::from($currencies->map(fn($c) => ['code' => $c->code, 'symbol' => $c->symbol, 'name' => $c->name])) !!},

        // ----- Computed: current currency info -----
        get currentSymbol() {
            return this.currencyData[this.selectedCurrency]?.symbol || this.selectedCurrency;
        },

        get currentName() {
            const found = this.currencyList.find(c => c.code === this.selectedCurrency);
            return found ? found.name : '';
        },

        get currentPrecision() {
            return this.currencyData[this.selectedCurrency]?.precision ?? 2;
        },

        // ----- Exchange rate conversion -----
        convert(baseAmount) {
            const selectedCur = this.currencyData[this.selectedCurrency];
            if (!selectedCur || this.defaultRate === 0) return baseAmount;
            // Convert: baseAmount (in default currency) -> USD -> selected currency
            const amountInUsd = baseAmount / this.defaultRate;
            return amountInUsd * selectedCur.rate;
        },

        formatAmount(amount) {
            return this.currentSymbol + Number(amount).toFixed(this.currentPrecision);
        },

        // ----- Display values (auto-update when currency changes) -----
        get displaySubtotal() {
            return this.formatAmount(this.convert(this.basePrice));
        },

        get displayPlanDiscount() {
            return this.formatAmount(this.convert(this.basePlanDiscount));
        },

        get displayCouponDiscount() {
            return this.formatAmount(this.convert(this.couponDiscountRaw));
        },

        get displayTotal() {
            const effective = this.couponApplied
                ? Math.max(0, this.baseEffectivePrice - this.couponDiscountRaw)
                : this.baseEffectivePrice;
            return this.formatAmount(this.convert(effective));
        },

        // ----- Gateway filtering -----
        get filteredGateways() {
            const currency = this.selectedCurrency;
            return Object.keys(this.gatewayCurrencies).filter(slug => {
                const currencies = this.gatewayCurrencies[slug];
                return currencies.length === 0 || currencies.includes(currency);
            });
        },

        isGatewayVisible(slug) {
            return this.filteredGateways.includes(slug);
        },

        onCurrencyChange() {
            // Reset gateway selection if current one doesn't support new currency
            if (this.selectedGateway && !this.isGatewayVisible(this.selectedGateway)) {
                this.selectedGateway = this.filteredGateways.length > 0 ? this.filteredGateways[0] : '';
            }
            // Reset coupon when currency changes
            if (this.couponApplied) {
                this.removeCoupon();
            }
        },

        init() {
            this.$nextTick(() => {
                if (this.filteredGateways.length > 0) {
                    this.selectedGateway = this.filteredGateways[0];
                }
            });
        },

        // ----- Coupon -----
        async applyCoupon() {
            if (!this.couponCode.trim()) return;
            this.couponLoading = true;
            this.couponMessage = '';

            try {
                const response = await fetch('{{ route('checkout.apply-coupon') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'Accept': 'application/json',
                    },
                    body: JSON.stringify({
                        coupon_code: this.couponCode,
                        plan_id: {{ $plan->id }},
                    }),
                });

                const data = await response.json();

                if (data.valid) {
                    this.couponApplied = true;
                    this.couponDiscountRaw = parseFloat(data.discount);
                    this.couponMessage = data.message;
                } else {
                    this.couponMessage = data.message || 'Invalid coupon code.';
                }
            } catch (error) {
                this.couponMessage = 'Failed to validate coupon. Please try again.';
            } finally {
                this.couponLoading = false;
            }
        },

        removeCoupon() {
            this.couponApplied = false;
            this.couponCode = '';
            this.couponDiscountRaw = 0;
            this.couponMessage = '';
        },
    };
}

function currencySearch() {
    return {
        open: false,
        search: '',
        highlighted: 0,

        get filtered() {
            const q = this.search.toLowerCase().trim();
            // currencyList is accessible from parent scope (Alpine v3 merges nested x-data)
            const list = this.currencyList || [];
            if (!q) return list;
            return list.filter(c =>
                c.code.toLowerCase().includes(q) ||
                c.name.toLowerCase().includes(q) ||
                c.symbol.includes(q)
            );
        },

        toggle() {
            this.open = !this.open;
            if (this.open) {
                this.search = '';
                this.highlighted = 0;
                this.$nextTick(() => this.$refs.searchInput?.focus());
            }
        },

        selectCurrency(code) {
            // Parent scope is merged in Alpine v3 nested x-data
            this.selectedCurrency = code;
            this.onCurrencyChange();
            this.open = false;
            this.search = '';
        },

        highlightNext() {
            if (this.highlighted < this.filtered.length - 1) {
                this.highlighted++;
                this.scrollToHighlighted();
            }
        },

        highlightPrev() {
            if (this.highlighted > 0) {
                this.highlighted--;
                this.scrollToHighlighted();
            }
        },

        selectHighlighted() {
            if (this.filtered.length > 0 && this.filtered[this.highlighted]) {
                this.selectCurrency(this.filtered[this.highlighted].code);
            }
        },

        scrollToHighlighted() {
            this.$nextTick(() => {
                const listbox = this.$refs.listbox;
                if (!listbox) return;
                const items = listbox.querySelectorAll('button');
                if (items[this.highlighted]) {
                    items[this.highlighted].scrollIntoView({ block: 'nearest' });
                }
            });
        },
    };
}
</script>
@endsection
