<x-gallery-layout :title="$event->name" :settings="$event->design_settings ?? []">
    @php
        $coverUrl = $event->coverMedia ? $event->coverMedia->signedPreviewUrl(240) : asset('images/default-cover.jpg');
        $dateLabel = \App\Support\FormatSettings::date($event->event_date);
        $totalPhotos = $media?->count() ?? 0;

        $mediaCount = ($mediaPaginator ?? null) ? $mediaPaginator->total() : ($media?->count() ?? 0);
        $totalPhotos = $mediaCount;
        $hasMatches = $faceSearchReady && ($faceSearchMatchesCount ?? 0) > 0;
        $guestUploads = $media ? $media->where('is_guest_upload', true) : collect();
        $highlights = $media ? $media->where('is_featured', true) : collect();
    @endphp

    <script>
        window.__gallery = {
            selectionsUrl: @json(isset($event) ? route('events.share.selections.list', ['event' => $event->slug, 'token' => $event->share_token]) : ''),
            toggleUrlBase: @json(isset($event) ? url('/events/' . $event->slug . '/selections') : ''),
            token: @json($event->share_token ?? ''),
            slug: @json($event->slug ?? ''),
            name: @json($event->name ?? '')
        };

        function galleryApp() {
            return {
                currentTab: 'all',
                uploadModalOpen: false,
                lightboxOpen: false,
                lightboxIndex: 0,
                lightboxSources: [],
                scrolled: false,

                // Infinite scroll state
                extraItems: [],
                currentPage: {{ ($mediaPaginator ?? null) ? $mediaPaginator->currentPage() : 1 }},
                lastPage: {{ ($mediaPaginator ?? null) ? $mediaPaginator->lastPage() : 1 }},
                totalCount: {{ ($mediaPaginator ?? null) ? $mediaPaginator->total() : ($media->count() ?? 0) }},
                initialCount: {{ $media->count() ?? 0 }},
                loadingMore: false,

                get hasMore() { return this.currentPage < this.lastPage; },
                get remaining() { return Math.max(0, this.totalCount - this.initialCount - this.extraItems.length); },

                cameraOpen: false,
                captured: false,
                imageSrc: null,
                stream: null,
                uploading: false,

                async startCamera() {
                    this.cameraOpen = true;
                    this.captured = false;
                    this.$nextTick(async () => {
                        try {
                            this.stream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: 'user' } });
                            this.$refs.video.srcObject = this.stream;
                            this.$refs.video.play();
                        } catch (err) {
                            console.error('Camera error:', err);
                            alert('Could not access camera. Please allow camera permissions.');
                            this.cameraOpen = false;
                        }
                    });
                },

                stopCamera() {
                    if (this.stream) {
                        this.stream.getTracks().forEach(track => track.stop());
                        this.stream = null;
                    }
                    this.cameraOpen = false;
                },

                capturePhoto() {
                    const canvas = document.createElement('canvas');
                    canvas.width = this.$refs.video.videoWidth;
                    canvas.height = this.$refs.video.videoHeight;
                    const ctx = canvas.getContext('2d');
                    ctx.translate(canvas.width, 0);
                    ctx.scale(-1, 1);
                    ctx.drawImage(this.$refs.video, 0, 0, canvas.width, canvas.height);
                    this.imageSrc = canvas.toDataURL('image/jpeg');
                    this.captured = true;
                    this.stopCamera();
                    this.cameraOpen = true;
                },

                retakePhoto() {
                    this.captured = false;
                    this.imageSrc = null;
                    this.startCamera();
                },

                confirmPhoto() {
                    this.uploading = true;
                    const dataUrl = this.imageSrc;
                    const arr = dataUrl.split(',');
                    const mime = (arr[0].match(/:(.*?);/) || [])[1] || 'image/jpeg';
                    const bstr = atob(arr[1]);
                    const u8arr = new Uint8Array(bstr.length);
                    for (let i = 0; i < bstr.length; i++) { u8arr[i] = bstr.charCodeAt(i); }
                    const blob = new Blob([u8arr], { type: mime });
                    const file = new File([blob], 'selfie.jpg', { type: 'image/jpeg' });
                    const container = new DataTransfer();
                    container.items.add(file);
                    this.$refs.faceSearchInput.files = container.files;
                    this.$refs.faceSearchForm.submit();
                },

                init() {
                    window.addEventListener('scroll', () => {
                        this.scrolled = window.scrollY > 50;
                    });
                    this.lightboxSources = Array.from(document.querySelectorAll('[data-lightbox-src]')).map(el => ({
                        id: Number(el.dataset.lightboxId),
                        src: el.dataset.lightboxSrc,
                        type: el.dataset.lightboxType
                    }));
                    this.loadSelections();
                },

                openLightbox(index) {
                    this.lightboxIndex = index;
                    this.lightboxOpen = true;
                    document.body.style.overflow = 'hidden';
                },

                closeLightbox() {
                    this.lightboxOpen = false;
                    this.shareModalOpen = false;
                    document.body.style.overflow = '';
                },

                selections: [],
                selectionsLoaded: false,
                async loadSelections() {
                    if (this.selectionsLoaded) return;
                    try {
                        var url = window.__gallery.selectionsUrl;
                        if (!url) return;
                        var res = await fetch(url, { headers: { 'Accept': 'application/json', 'X-Requested-With': 'XMLHttpRequest' } });
                        if (!res.ok) return;
                        var data = await res.json();
                        this.selections = (data.selected_ids || []).map(Number);
                        this.selectionsLoaded = true;
                    } catch (e) { console.warn('Failed to load selections', e); }
                },

                async toggleSelection(mediaId) {
                    var csrfToken = document.querySelector('meta[name=csrf-token]');
                    csrfToken = csrfToken ? csrfToken.content : '';
                    try {
                        var res = await fetch(window.__gallery.toggleUrlBase + '/' + mediaId + '/toggle?token=' + encodeURIComponent(window.__gallery.token), {
                            method: 'POST',
                            headers: { 'Accept': 'application/json', 'X-CSRF-TOKEN': csrfToken, 'X-Requested-With': 'XMLHttpRequest' },
                        });
                        if (!res.ok) return;
                        var data = await res.json();
                        if (data.selected) {
                            this.selections = this.selections.concat([mediaId]);
                        } else {
                            this.selections = this.selections.filter(function(id) { return id !== mediaId; });
                        }
                    } catch (e) { console.warn('Selection toggle failed', e); }
                },

                isSelected(mediaId) {
                    return this.selections.indexOf(mediaId) !== -1;
                },

                async loadMorePhotos() {
                    if (this.loadingMore || !this.hasMore) return;
                    this.loadingMore = true;
                    try {
                        var url = new URL(window.location.href);
                        url.searchParams.set('page', this.currentPage + 1);
                        var res = await fetch(url.toString(), {
                            headers: { 'Accept': 'application/json', 'X-Requested-With': 'XMLHttpRequest' }
                        });
                        if (!res.ok) throw new Error('Load failed');
                        var data = await res.json();
                        this.currentPage = data.current_page;
                        this.lastPage = data.last_page;
                        var self = this;
                        (data.items || []).forEach(function(item) {
                            self.extraItems.push(item);
                            self.lightboxSources.push({ id: item.id, src: item.full_url, type: item.file_type });
                        });
                    } catch (e) { console.warn('Load more failed', e); }
                    this.loadingMore = false;
                },

                shareModalOpen: false,
                shareCopied: false,
                sharingNative: false,

                getImageUrl() {
                    var item = this.lightboxSources[this.lightboxIndex];
                    return item ? item.src : '';
                },

                getGalleryUrl() {
                    return window.location.href.split('?')[0] + '?token=' + window.__gallery.token;
                },

                getShareText() {
                    return 'Check out this photo from ' + window.__gallery.name + '!';
                },

                async sharePhoto(index) {
                    this.shareCopied = false;
                    var isMobile = /Android|iPhone|iPad|iPod/i.test(navigator.userAgent);
                    // Mobile only: try native share with actual image file
                    if (isMobile && navigator.share && navigator.canShare) {
                        try {
                            this.sharingNative = true;
                            var imgUrl = this.getImageUrl();
                            var res = await fetch(imgUrl);
                            var blob = await res.blob();
                            var ext = blob.type === 'image/png' ? '.png' : '.jpg';
                            var file = new File([blob], window.__gallery.name + ext, { type: blob.type });
                            if (navigator.canShare({ files: [file] })) {
                                await navigator.share({
                                    files: [file],
                                    title: window.__gallery.name,
                                    text: this.getShareText()
                                });
                                this.sharingNative = false;
                                return;
                            }
                        } catch (e) {
                            // User cancelled or not supported — fall through to modal
                        }
                        this.sharingNative = false;
                    }
                    // Desktop / fallback: show social share modal
                    this.shareModalOpen = true;
                },

                async copyShareLink() {
                    var url = this.getImageUrl();
                    try {
                        await navigator.clipboard.writeText(url);
                        this.shareCopied = true;
                        setTimeout(() => { this.shareCopied = false; }, 2000);
                    } catch (e) {
                        var ta = document.createElement('textarea');
                        ta.value = url;
                        document.body.appendChild(ta);
                        ta.select();
                        document.execCommand('copy');
                        document.body.removeChild(ta);
                        this.shareCopied = true;
                        setTimeout(() => { this.shareCopied = false; }, 2000);
                    }
                },

                shareToSocial(platform) {
                    var imageUrl = encodeURIComponent(this.getImageUrl());
                    var galleryUrl = encodeURIComponent(this.getGalleryUrl());
                    var text = encodeURIComponent(this.getShareText());
                    var title = encodeURIComponent(window.__gallery.name);
                    var links = {
                        facebook: 'https://www.facebook.com/sharer/sharer.php?u=' + galleryUrl,
                        twitter: 'https://twitter.com/intent/tweet?url=' + imageUrl + '&text=' + text,
                        whatsapp: 'https://api.whatsapp.com/send?text=' + text + '%20' + imageUrl,
                        linkedin: 'https://www.linkedin.com/sharing/share-offsite/?url=' + galleryUrl,
                        telegram: 'https://t.me/share/url?url=' + imageUrl + '&text=' + text,
                        pinterest: 'https://pinterest.com/pin/create/button/?url=' + galleryUrl + '&media=' + imageUrl + '&description=' + text,
                        reddit: 'https://reddit.com/submit?url=' + imageUrl + '&title=' + title,
                        email: 'mailto:?subject=' + title + '&body=' + text + '%0A%0A' + imageUrl
                    };
                    if (links[platform]) window.open(links[platform], '_blank', 'width=600,height=500,noopener');
                    this.shareModalOpen = false;
                }
            };
        }
    </script>

    <div x-data="galleryApp()" class="min-h-screen bg-white">

        <!-- Access Control / Pin / Face Search Screens -->
        @if ($pinRequired)
            <div class="fixed inset-0 z-50 flex items-center justify-center bg-white overflow-y-auto">
                <div class="w-full max-w-lg p-8 text-center">
                    <p class="text-sm font-bold tracking-widest text-slate-900 uppercase mb-2">{{ $event->name }}</p>
                    <h2 class="text-4xl font-bold text-slate-900 mb-12">{{ __('Access Gallery') }}</h2>

                    <form method="POST" action="{{ route('events.share.access', ['event' => $event->slug, 'token' => request()->query('token')]) }}" class="space-y-6 max-w-sm mx-auto">
                        @csrf
                        <div class="text-left">
                            <label class="block text-sm font-medium text-slate-700 mb-2">{{ __('Security PIN') }}</label>
                            <input name="pin" type="password" inputmode="numeric" pattern="[0-9]{4,8}"
                                class="w-full h-12 px-4 border border-gray-200 rounded text-slate-900 placeholder-slate-400 text-center text-lg tracking-widest focus:outline-none focus:ring-1 focus:ring-red-500 focus:border-red-500 transition-all font-mono"
                                placeholder="• • • •" required autofocus>
                            <p class="text-[10px] text-slate-400 mt-1.5 Uppercase tracking-wider">{{ __('Please enter the access code provided by the organizer') }}</p>
                        </div>

                        @error('pin')
                            <div class="p-3 bg-red-50 text-red-600 text-xs rounded border border-red-100 font-medium">
                                {{ $message }}
                            </div>
                        @enderror

                        <button type="submit" class="w-full h-12 bg-red-600 hover:bg-red-700 text-white font-bold uppercase tracking-widest text-sm rounded shadow-lg shadow-teal-800/20 transition-all active:scale-[0.98]">
                            {{ __('Unlock Gallery') }}
                        </button>

                        <div class="pt-8 text-[10px] text-slate-400 uppercase tracking-[0.2em]">
                            {{ __('Protected by') }} {{ config('app.name', 'SnapNest') }}
                        </div>
                    </form>
                </div>
            </div>

        @elseif (($faceSearchRequired ?? false) && ! ($faceSearchReady ?? false))
             <div class="fixed inset-0 z-50 flex items-center justify-center bg-white">
                <div class="w-full max-w-lg p-8 text-center">
                    <p class="text-sm font-bold tracking-widest text-slate-900 uppercase mb-2">{{ $event->name }}</p>
                    <h2 class="text-4xl font-bold text-slate-900 mb-12">{{ __('Get Your Photos') }}</h2>

                    <div class="space-y-6 max-w-sm mx-auto">
                        <div class="text-left">
                            <label class="block text-sm font-medium text-slate-700 mb-2">{{ __('Capture Selfie') }}</label>
                            <button @click="startCamera()" class="w-full h-12 flex items-center justify-center gap-2 border border-red-500 text-red-600 rounded bg-white hover:bg-teal-50 transition-colors">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"></path><circle cx="12" cy="13" r="4"></circle></svg>
                                {{ __('Open Camera') }}
                            </button>
                        </div>

                        <div class="text-left">
                             <div class="flex items-start gap-3">
                                <input type="checkbox" id="privacy" class="mt-1 w-5 h-5 rounded border-slate-300 text-red-600 focus:ring-red-500" checked>
                                <label for="privacy" class="text-sm text-slate-600">{{ __('I agree to the') }} <a href="{{ route('legal.privacy') }}" target="_blank" class="underline text-red-600">{{ __('Privacy Policy') }}</a> {{ __('and') }} <a href="{{ route('legal.terms') }}" target="_blank" class="underline text-red-600">{{ __('Terms of Service') }}</a> {{ __('for face recognition processing.') }}</label>
                            </div>
                        </div>

                        <form x-ref="faceSearchForm" method="POST" action="{{ route('events.share.face-search', ['event' => $event->slug, 'token' => request()->query('token')]) }}" enctype="multipart/form-data">
                            @csrf
                            <input x-ref="faceSearchInput" type="file" name="face_image" class="hidden" required>

                            <button type="button" disabled class="w-full h-12 bg-gray-200 text-gray-400 font-bold rounded cursor-not-allowed mt-8">
                                {{ __('Get My Photos') }}
                            </button>
                             <p class="text-xs text-gray-400 mt-2">{{ __('Please capture a selfie to proceed') }}</p>
                        </form>
                    </div>
                </div>
            </div>

        @elseif($inactive || $expired)
            <div class="fixed inset-0 z-50 flex items-center justify-center bg-gray-100">
                <div class="text-center p-8">
                     <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" class="mx-auto text-gray-400 mb-4"><circle cx="12" cy="12" r="10"/><line x1="12" x2="12" y1="8" y2="12"/><line x1="12" x2="12.01" y1="16" y2="16"/></svg>
                     <h2 class="text-xl font-semibold text-gray-700">{{ $inactive ? __('Event Inactive') : __('Event Expired') }}</h2>
                     <p class="text-gray-500 mt-2">{{ __('This gallery is currently unavailable.') }}</p>
                </div>
            </div>

        @else
            <!-- Main Gallery UI -->

            <!-- Navbar -->
            <nav class="fixed top-0 left-0 w-full z-40 transition-all duration-300 border-b border-transparent"
                 :class="{'bg-white/90 backdrop-blur-md border-gray-100 shadow-sm py-2': scrolled, 'bg-transparent py-4': !scrolled}">
                <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 flex items-center justify-between">
                     <!-- Logo / Brand -->
                     <div class="flex items-center gap-2">
                        <span class="text-xl font-bold tracking-tight text-white transition-colors" :class="{'text-slate-900': scrolled}">
                            {{ config('app.name', 'SnapNest') }}
                        </span>
                     </div>

                     <!-- Right Actions -->
                     <div class="flex items-center gap-3">
                        @if ($faceSearchRequired ?? false)
                            <form x-ref="faceSearchForm" method="POST" action="{{ route('events.share.face-search', ['event' => $event->slug, 'token' => request()->query('token')]) }}" enctype="multipart/form-data" class="hidden">
                                @csrf
                                <input x-ref="faceSearchInput" type="file" name="face_image" required>
                            </form>
                             <button @click="startCamera()" type="button" class="px-4 py-2 rounded-full text-xs font-semibold uppercase tracking-wider transition-colors bg-red-600 text-white hover:bg-red-700">
                                 {{ __('Retake Selfie') }}
                             </button>
                        @endif
                     </div>
                </div>
            </nav>

            <!-- Hero Section -->
            <header class="relative h-[80vh] min-h-[500px] flex items-center justify-center overflow-hidden">
                <!-- Background -->
                <div class="absolute inset-0 z-0 bg-gray-900">
                    <img src="{{ $coverUrl }}" class="w-full h-full object-cover">
                    <div class="absolute inset-0 bg-gradient-to-b from-black/60 via-black/20 to-black/80"></div>
                </div>

                <!-- Content -->
                <div class="relative z-10 text-center px-4 max-w-4xl mx-auto -mt-10">
                    <p class="text-red-500 font-bold tracking-[0.2em] uppercase text-sm mb-4 animate-fade-in-up">{{ __('Event Gallery') }}</p>
                    <h1 class="text-5xl md:text-7xl font-bold text-white tracking-tight mb-4 animate-fade-in-up delay-100 shadow-lg font-brand hero-title">{{ $event->name }}</h1>
                    <div class="flex flex-wrap items-center justify-center gap-4 text-white/90 text-sm md:text-base font-medium tracking-wide animate-fade-in-up delay-200">
                         <span>{{ strtoupper($dateLabel) }}</span>
                         <span class="w-1 h-1 rounded-full bg-red-500"></span>
                         <span>{{ $totalPhotos }} {{ __('PHOTOS') }}</span>
                    </div>

                    <button @click="window.scrollTo({top: window.innerHeight - 100, behavior: 'smooth'})"
                        class="mt-10 px-8 py-3 bg-transparent border border-white/30 text-white hover:bg-white hover:text-black transition-all duration-300 rounded-full text-sm font-bold tracking-widest uppercase animate-fade-in-up delay-300">
                        {{ __('View Gallery') }}
                    </button>
                </div>
            </header>

            <!-- Sticky Tabs -->
            <div class="sticky top-[57px] z-30 bg-white border-b border-gray-100 shadow-sm">
                <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div class="gallery-tabs flex items-center justify-start sm:justify-center overflow-x-auto no-scrollbar gap-8 py-4">
                        <button @click="currentTab = 'all'"
                            class="whitespace-nowrap pb-1 text-sm font-bold uppercase tracking-wider transition-all border-b-2"
                            :class="currentTab === 'all' ? 'text-red-600 border-red-600' : 'text-slate-500 border-transparent hover:text-slate-800'">
                            {{ __('All Photos') }}
                        </button>
                        @if($event->allow_guest_upload)
                            <button @click="currentTab = 'guest'"
                                class="whitespace-nowrap pb-1 text-sm font-bold uppercase tracking-wider transition-all border-b-2"
                                :class="currentTab === 'guest' ? 'text-red-600 border-red-600' : 'text-slate-500 border-transparent hover:text-slate-800'">
                                {{ __('Guest Uploads') }}
                            </button>
                         @endif
                        <button @click="currentTab = 'highlights'"
                            class="whitespace-nowrap pb-1 text-sm font-bold uppercase tracking-wider transition-all border-b-2"
                            :class="currentTab === 'highlights' ? 'text-red-600 border-red-600' : 'text-slate-500 border-transparent hover:text-slate-800'">
                            {{ __('Highlights') }}
                        </button>
                        <button @click="currentTab = 'selections'"
                            class="whitespace-nowrap pb-1 text-sm font-bold uppercase tracking-wider transition-all border-b-2 flex items-center gap-1.5"
                            :class="currentTab === 'selections' ? 'text-red-600 border-red-600' : 'text-slate-500 border-transparent hover:text-slate-800'">
                            {{ __('My Selections') }}
                            <span x-show="selections.length > 0" x-text="selections.length" class="inline-flex h-5 min-w-[1.25rem] items-center justify-center rounded-full bg-red-500 px-1.5 text-[10px] font-bold text-white"></span>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <main class="max-w-[1600px] mx-auto min-h-[50vh] p-4 sm:p-6 pb-20">

                @if($totalPhotos === 0)
                     <div class="flex flex-col items-center justify-center py-20 text-center">
                        <div class="w-64 h-64 mb-6 opacity-80">
                            <!-- Illustration placeholder -->
                            <svg viewBox="0 0 200 200" fill="none" xmlns="http://www.w3.org/2000/svg" class="w-full h-full text-gray-200">
                                <circle cx="100" cy="100" r="80" fill="#F3F4F6"/>
                                <path d="M65 115L85 95L115 125" stroke="#9CA3AF" stroke-width="8" stroke-linecap="round" stroke-linejoin="round"/>
                                <circle cx="115" cy="85" r="8" fill="#9CA3AF"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-slate-800">{{ __('No Images Found') }}</h3>
                        <p class="text-slate-500 mt-2">{{ __('Looks like this gallery is empty or no matches found.') }}</p>
                     </div>
                @else
                    <!-- Grid -->
                    <div class="gallery-grid grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5">
                        {{-- Server-rendered first page --}}
                        @foreach ($media as $index => $item)
                            @php
                                $previewUrl = $item->signedThumbnailUrl(240) ?: $item->signedPreviewUrl(240);
                                $fullUrl = $item->signedPreviewUrl(240);
                            @endphp
                            <div class="gallery-item relative group aspect-[3/4] bg-gray-100 cursor-zoom-in"
                                 x-show="currentTab === 'all'
                                      || (currentTab === 'guest' && {{ $item->is_guest_upload ? 'true' : 'false' }})
                                      || (currentTab === 'highlights' && {{ $item->is_featured ? 'true' : 'false' }})
                                      || (currentTab === 'selections' && isSelected({{ $item->id }}))"
                                 x-transition.opacity
                                 data-lightbox-id="{{ $item->id }}"
                                 data-lightbox-src="{{ $fullUrl }}"
                                 data-lightbox-type="{{ $item->file_type }}"
                                 @click="openLightbox({{ $index }})">

                                <img src="{{ $previewUrl }}" class="w-full h-full object-cover transition-transform duration-700 ease-out group-hover:scale-105" loading="lazy">

                                <div class="absolute inset-0 bg-black/0 group-hover:bg-black/10 transition-all duration-300"></div>

                                @if($item->file_type === 'video')
                                    <div class="absolute top-2 right-2 text-white drop-shadow-md">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor" class="opacity-90"><path d="M8 5v14l11-7z"/></svg>
                                    </div>
                                @endif
                                {{-- Heart / Selection button --}}
                                <button @click.stop="toggleSelection({{ $item->id }})" class="absolute top-2 left-2 z-10 p-1.5 rounded-full transition-all duration-200" :class="isSelected({{ $item->id }}) ? 'bg-red-500 text-white shadow-lg scale-110' : 'bg-black/30 text-white/80 opacity-0 group-hover:opacity-100 hover:bg-teal-500 hover:text-white'">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" :fill="isSelected({{ $item->id }}) ? 'currentColor' : 'none'" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>
                                </button>

                                @if($event->allow_public_downloads)
                                 <div class="absolute bottom-0 left-0 w-full p-4 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex justify-between items-end">
                                    <a href="{{ route('events.share.download', ['event' => $event->slug, 'media' => $item->id, 'token' => request()->query('token')]) }}"
                                       @click.stop
                                       class="flex items-center gap-1.5 text-white text-xs font-bold hover:text-teal-400 transition-colors bg-black/20 backdrop-blur-md px-3 py-1.5 rounded-full border border-white/20">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" x2="12" y1="15" y2="3"/></svg>
                                        {{ __('Download') }}
                                    </a>
                                </div>
                                @endif
                            </div>
                        @endforeach

                        {{-- AJAX-loaded additional pages --}}
                        <template x-for="(item, i) in extraItems" :key="item.id">
                            <div class="gallery-item relative group aspect-[3/4] bg-gray-100 cursor-zoom-in"
                                 x-show="currentTab === 'all'
                                      || (currentTab === 'guest' && item.is_guest)
                                      || (currentTab === 'highlights' && item.is_featured)
                                      || (currentTab === 'selections' && isSelected(item.id))"
                                 x-transition.opacity
                                 @click="openLightbox(initialCount + i)">

                                <img :src="item.preview_url" class="w-full h-full object-cover transition-transform duration-700 ease-out group-hover:scale-105" loading="lazy">

                                <div class="absolute inset-0 bg-black/0 group-hover:bg-black/10 transition-all duration-300"></div>

                                <template x-if="item.file_type === 'video'">
                                    <div class="absolute top-2 right-2 text-white drop-shadow-md">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor" class="opacity-90"><path d="M8 5v14l11-7z"/></svg>
                                    </div>
                                </template>

                                <button @click.stop="toggleSelection(item.id)" class="absolute top-2 left-2 z-10 p-1.5 rounded-full transition-all duration-200" :class="isSelected(item.id) ? 'bg-red-500 text-white shadow-lg scale-110' : 'bg-black/30 text-white/80 opacity-0 group-hover:opacity-100 hover:bg-teal-500 hover:text-white'">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" :fill="isSelected(item.id) ? 'currentColor' : 'none'" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>
                                </button>

                                @if($event->allow_public_downloads)
                                <div class="absolute bottom-0 left-0 w-full p-4 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex justify-between items-end">
                                    <a :href="item.download_url"
                                       @click.stop
                                       class="flex items-center gap-1.5 text-white text-xs font-bold hover:text-teal-400 transition-colors bg-black/20 backdrop-blur-md px-3 py-1.5 rounded-full border border-white/20">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" x2="12" y1="15" y2="3"/></svg>
                                        {{ __('Download') }}
                                    </a>
                                </div>
                                @endif
                            </div>
                        </template>
                    </div>

                    {{-- Infinite scroll: load more sentinel --}}
                    <div x-show="hasMore && (currentTab === 'all' || currentTab === 'guest')" x-cloak class="flex justify-center mt-10">
                        <button @click="loadMorePhotos()"
                                x-intersect:enter.margin.200px="loadMorePhotos()"
                                class="group flex items-center gap-3 px-6 py-3 rounded-full border border-gray-200 bg-white hover:border-gray-300 hover:shadow-md transition-all"
                                :disabled="loadingMore">
                            <template x-if="loadingMore">
                                <span class="flex items-center gap-2 text-sm text-slate-500">
                                    <svg class="animate-spin h-4 w-4" viewBox="0 0 24 24" fill="none"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"/><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"/></svg>
                                    {{ __('Loading...') }}
                                </span>
                            </template>
                            <template x-if="!loadingMore">
                                <span class="flex items-center gap-2 text-sm font-semibold text-slate-600 group-hover:text-slate-900">
                                    <span class="flex items-center justify-center h-7 min-w-[1.75rem] px-1.5 rounded-full bg-red-500 text-[11px] font-bold text-white" x-text="'+' + remaining"></span>
                                    {{ __('more photos') }}
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="opacity-40 group-hover:opacity-100 transition-opacity"><polyline points="6 9 12 15 18 9"/></svg>
                                </span>
                            </template>
                        </button>
                    </div>
                @endif

                {{-- Empty state: Highlights --}}
                @if($highlights->isEmpty())
                <div x-show="currentTab === 'highlights'" x-cloak class="flex flex-col items-center justify-center py-24 text-center">
                    <div class="w-20 h-20 rounded-full bg-amber-50 flex items-center justify-center mb-6">
                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" class="text-amber-400"><polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/></svg>
                    </div>
                    <h3 class="text-xl font-bold text-slate-800">{{ __('No Highlights Yet') }}</h3>
                    <p class="text-slate-400 mt-2 max-w-sm text-sm">{{ __('The photographer hasn\'t selected any highlights for this event yet.') }}</p>
                </div>
                @endif

                {{-- Empty state: Selections --}}
                <div x-show="currentTab === 'selections' && selections.length === 0" x-cloak class="flex flex-col items-center justify-center py-24 text-center">
                    <div class="w-20 h-20 rounded-full bg-red-50 flex items-center justify-center mb-6">
                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" class="text-red-400"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>
                    </div>
                    <h3 class="text-xl font-bold text-slate-800">{{ __('No Selections Yet') }}</h3>
                    <p class="text-slate-400 mt-2 max-w-sm text-sm">{{ __('Tap the heart icon on any photo to add it to your selections.') }}</p>
                </div>

                {{-- All photos loaded indicator --}}
                <div x-show="!hasMore && totalCount > initialCount && (currentTab === 'all' || currentTab === 'guest')" x-cloak class="text-center mt-10">
                    <p class="text-xs text-slate-400 font-medium tracking-wider uppercase">{{ __('All') }} <span x-text="totalCount"></span> {{ __('photos loaded') }}</p>
                </div>

                <!-- Guest Upload CTA -->
                 @if($event->allow_guest_upload)
                    <div class="mt-20 text-center border-t border-gray-100 pt-16 pb-8">
                        <h3 class="text-2xl font-bold text-slate-900">{{ __('Upload Photos taken by You') }}</h3>
                        <p class="text-slate-500 mt-2 mb-8 max-w-lg mx-auto">{{ __('Share and Showcase your perspective of the event. Your photos will be added to the Guest Uploads tab.') }}</p>

                        <button @click="uploadModalOpen = true" class="px-8 py-4 bg-red-600 hover:bg-red-700 text-white font-bold rounded-full shadow-xl shadow-teal-800/20 transition-all hover:-translate-y-1 flex items-center gap-2 mx-auto">
                            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="17 8 12 3 7 8"/><line x1="12" x2="12" y1="3" y2="15"/></svg>
                            {{ __('Upload Photos') }}
                        </button>
                    </div>
                @endif

                @if($showBranding ?? true)
                <footer class="mt-20 text-center text-xs text-slate-400 font-medium tracking-widest uppercase">
                    {{ __('Powered by') }} {{ config('app.name', 'SnapNest') }}
                </footer>
                @endif
            </main>

            <!-- Upload Modal -->
            <div x-show="uploadModalOpen" x-cloak class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80 backdrop-blur-sm" x-transition>
                <div class="bg-white rounded-2xl w-full max-w-lg p-6 relative shadow-2xl" @click.away="uploadModalOpen = false">
                    <button @click="uploadModalOpen = false" class="absolute top-4 right-4 text-slate-400 hover:text-slate-600">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
                    </button>

                    <h3 class="text-xl font-bold text-slate-900 mb-2">{{ __('Upload Photos') }}</h3>
                    <p class="text-sm text-slate-500 mb-6">{{ __('Select photos and videos to add to the event gallery.') }}</p>

                     <form action="{{ route('events.share.guest-upload', ['event' => $event->slug, 'token' => request()->query('token')]) }}" method="POST" enctype="multipart/form-data" class="space-y-4">
                        @csrf
                        <div class="relative group cursor-pointer">
                            <input type="file" name="files[]" multiple accept="image/*,video/*" class="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10" required>
                            <div class="w-full h-40 border-2 border-dashed border-slate-200 rounded-xl flex flex-col items-center justify-center bg-slate-50 group-hover:bg-slate-100 group-hover:border-teal-400 transition-all">
                                <span class="bg-white p-3 rounded-full shadow-sm mb-3 text-red-500">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="17 8 12 3 7 8"/><line x1="12" x2="12" y1="3" y2="15"/></svg>
                                </span>
                                <span class="font-medium text-slate-700">{{ __('Click to Select Files') }}</span>
                                <span class="text-xs text-slate-400 mt-1">{{ __('Up to 10 files, 50MB each') }}</span>
                            </div>
                        </div>
                        <button type="submit" class="w-full h-12 bg-red-600 hover:bg-red-700 text-white font-bold rounded-xl shadow-lg transition-transform active:scale-[0.98]">
                            {{ __('Start Upload') }}
                        </button>
                    </form>
                </div>
            </div>

            <!-- Lightbox -->
            <div x-show="lightboxOpen" x-cloak
                 class="fixed inset-0 z-[60] bg-black text-white flex flex-col"
                 x-transition:enter="transition ease-out duration-200"
                 x-transition:enter-start="opacity-0"
                 x-transition:enter-end="opacity-100"
                 x-transition:leave="transition ease-in duration-200"
                 x-transition:leave-start="opacity-100"
                 x-transition:leave-end="opacity-0"
                 @keydown.window.escape="shareModalOpen ? shareModalOpen = false : closeLightbox()">

                <!-- Lightbox Toolbar -->
                 <div class="flex items-center justify-between p-4 absolute top-0 left-0 w-full z-30 bg-gradient-to-b from-black/80 to-transparent">
                     <span class="text-sm font-medium tracking-wide text-white"><span x-text="lightboxIndex + 1"></span> / <span x-text="lightboxSources.length"></span></span>
                     <div class="flex items-center gap-2">
                        {{-- Heart toggle in lightbox --}}
                        <button @click="toggleSelection(lightboxSources[lightboxIndex]?.id)" class="p-2 rounded-full transition-all" :class="isSelected(lightboxSources[lightboxIndex]?.id) ? 'bg-red-500 text-white' : 'bg-white/10 hover:bg-white/20 text-white'">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" :fill="isSelected(lightboxSources[lightboxIndex]?.id) ? 'currentColor' : 'none'" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>
                        </button>
                        {{-- Share button --}}
                        <button @click="sharePhoto(lightboxIndex)" class="p-2 bg-white/10 hover:bg-white/20 rounded-full transition-colors text-white">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="18" cy="5" r="3"/><circle cx="6" cy="12" r="3"/><circle cx="18" cy="19" r="3"/><line x1="8.59" y1="13.51" x2="15.42" y2="17.49"/><line x1="15.41" y1="6.51" x2="8.59" y2="10.49"/></svg>
                        </button>
                        @if($event->allow_public_downloads)
                        {{-- Download --}}
                        <a :href="`/events/{{ $event->slug }}/media/${lightboxSources[lightboxIndex]?.id}/download?token={{ request()->query('token') }}`"
                           class="flex items-center gap-2 px-4 py-2 bg-white/10 hover:bg-white/20 text-white rounded-full text-xs font-bold transition-all border border-white/20">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" x2="12" y1="15" y2="3"/></svg>
                            {{ __('Download') }}
                        </a>
                        @endif
                        <button @click="closeLightbox()" class="p-2 hover:bg-white/10 rounded-full transition-colors text-white">
                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
                        </button>
                     </div>
                 </div>

                 <!-- Share Modal -->
                 <div x-show="shareModalOpen" x-cloak
                      class="fixed inset-0 z-[70] flex items-center justify-center bg-black/60 backdrop-blur-sm"
                      @click.self="shareModalOpen = false"
                      x-transition:enter="transition ease-out duration-200"
                      x-transition:enter-start="opacity-0"
                      x-transition:enter-end="opacity-100"
                      x-transition:leave="transition ease-in duration-150"
                      x-transition:leave-start="opacity-100"
                      x-transition:leave-end="opacity-0">
                     <div class="bg-white rounded-2xl shadow-2xl w-[90%] max-w-md mx-auto overflow-hidden"
                          x-transition:enter="transition ease-out duration-200"
                          x-transition:enter-start="opacity-0 scale-95 translate-y-4"
                          x-transition:enter-end="opacity-100 scale-100 translate-y-0"
                          @click.stop>
                         {{-- Header with preview --}}
                         <div class="flex items-center gap-3 px-5 py-4 border-b border-gray-100">
                             <div class="w-12 h-12 rounded-lg overflow-hidden bg-gray-100 flex-shrink-0">
                                 <img :src="getImageUrl()" class="w-full h-full object-cover">
                             </div>
                             <div class="flex-1 min-w-0">
                                 <h3 class="text-base font-bold text-gray-900">{{ __('Share Photo') }}</h3>
                                 <p class="text-xs text-gray-400 truncate" x-text="window.__gallery.name"></p>
                             </div>
                             <button @click="shareModalOpen = false" class="p-1 hover:bg-gray-100 rounded-full transition-colors flex-shrink-0">
                                 <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-gray-500"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>
                             </button>
                         </div>

                         {{-- Copy Link --}}
                         <div class="px-5 pt-4 pb-3">
                             <div class="flex items-center gap-2 bg-gray-50 rounded-xl p-3 border border-gray-200">
                                 <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-gray-400 flex-shrink-0"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/></svg>
                                 <span class="text-sm text-gray-600 truncate flex-1" x-text="getImageUrl()"></span>
                                 <button @click="copyShareLink()" class="flex-shrink-0 px-3 py-1.5 text-xs font-bold rounded-lg transition-all"
                                         :class="shareCopied ? 'bg-green-500 text-white' : 'bg-gray-900 text-white hover:bg-gray-700'">
                                     <span x-show="!shareCopied">{{ __('Copy') }}</span>
                                     <span x-show="shareCopied" x-cloak>{{ __('Copied!') }}</span>
                                 </button>
                             </div>
                         </div>

                         {{-- Social Icons --}}
                         <div class="px-5 pb-5">
                             <p class="text-[10px] font-bold text-gray-400 uppercase tracking-widest mb-3">{{ __('Share via') }}</p>
                             <div class="grid grid-cols-4 gap-3">
                                 {{-- Facebook --}}
                                 <button @click="shareToSocial('facebook')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#1877F2] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="white"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('Facebook') }}</span>
                                 </button>
                                 {{-- Twitter/X --}}
                                 <button @click="shareToSocial('twitter')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-black flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="white"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('X') }}</span>
                                 </button>
                                 {{-- WhatsApp --}}
                                 <button @click="shareToSocial('whatsapp')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#25D366] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="22" height="22" viewBox="0 0 24 24" fill="white"><path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('WhatsApp') }}</span>
                                 </button>
                                 {{-- Telegram --}}
                                 <button @click="shareToSocial('telegram')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#0088cc] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="white"><path d="M11.944 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0a12 12 0 0 0-.056 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 0 1 .171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.479.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('Telegram') }}</span>
                                 </button>
                                 {{-- LinkedIn --}}
                                 <button @click="shareToSocial('linkedin')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#0A66C2] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="white"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('LinkedIn') }}</span>
                                 </button>
                                 {{-- Pinterest --}}
                                 <button @click="shareToSocial('pinterest')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#E60023] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="white"><path d="M12.017 0C5.396 0 .029 5.367.029 11.987c0 5.079 3.158 9.417 7.618 11.162-.105-.949-.199-2.403.041-3.439.219-.937 1.406-5.957 1.406-5.957s-.359-.72-.359-1.781c0-1.668.967-2.914 2.171-2.914 1.023 0 1.518.769 1.518 1.69 0 1.029-.655 2.568-.994 3.995-.283 1.194.599 2.169 1.777 2.169 2.133 0 3.772-2.249 3.772-5.495 0-2.873-2.064-4.882-5.012-4.882-3.414 0-5.418 2.561-5.418 5.207 0 1.031.397 2.138.893 2.738a.36.36 0 0 1 .083.345l-.333 1.36c-.053.22-.174.267-.402.161-1.499-.698-2.436-2.889-2.436-4.649 0-3.785 2.75-7.262 7.929-7.262 4.163 0 7.398 2.967 7.398 6.931 0 4.136-2.607 7.464-6.227 7.464-1.216 0-2.359-.631-2.75-1.378l-.748 2.853c-.271 1.043-1.002 2.35-1.492 3.146C9.57 23.812 10.763 24 12.017 24c6.624 0 11.99-5.367 11.99-11.988C24.007 5.367 18.641 0 12.017 0z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('Pinterest') }}</span>
                                 </button>
                                 {{-- Reddit --}}
                                 <button @click="shareToSocial('reddit')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-[#FF4500] flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="white"><path d="M12 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0zm5.01 4.744c.688 0 1.25.561 1.25 1.249a1.25 1.25 0 0 1-2.498.056l-2.597-.547-.8 3.747c1.824.07 3.48.632 4.674 1.488.308-.309.73-.491 1.207-.491.968 0 1.754.786 1.754 1.754 0 .716-.435 1.333-1.01 1.614a3.111 3.111 0 0 1 .042.52c0 2.694-3.13 4.87-7.004 4.87-3.874 0-7.004-2.176-7.004-4.87 0-.183.015-.366.043-.534A1.748 1.748 0 0 1 4.028 12c0-.968.786-1.754 1.754-1.754.463 0 .898.196 1.207.49 1.207-.883 2.878-1.43 4.744-1.487l.885-4.182a.342.342 0 0 1 .14-.197.35.35 0 0 1 .238-.042l2.906.617a1.214 1.214 0 0 1 1.108-.701zM9.25 12C8.561 12 8 12.562 8 13.25c0 .687.561 1.248 1.25 1.248.687 0 1.248-.561 1.248-1.249 0-.688-.561-1.249-1.249-1.249zm5.5 0c-.687 0-1.248.561-1.248 1.25 0 .687.561 1.248 1.249 1.248.688 0 1.249-.561 1.249-1.249 0-.687-.562-1.249-1.25-1.249zm-5.466 3.99a.327.327 0 0 0-.231.094.33.33 0 0 0 0 .463c.842.842 2.484.913 2.961.913.477 0 2.105-.056 2.961-.913a.361.361 0 0 0 .029-.463.33.33 0 0 0-.464 0c-.547.533-1.684.73-2.512.73-.828 0-1.979-.196-2.512-.73a.326.326 0 0 0-.232-.095z"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('Reddit') }}</span>
                                 </button>
                                 {{-- Email --}}
                                 <button @click="shareToSocial('email')" class="flex flex-col items-center gap-1.5 p-3 rounded-xl hover:bg-gray-50 transition-colors group">
                                     <div class="w-11 h-11 rounded-full bg-gray-600 flex items-center justify-center group-hover:scale-110 transition-transform">
                                         <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="20" height="16" x="2" y="4" rx="2"/><path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"/></svg>
                                     </div>
                                     <span class="text-[10px] font-semibold text-gray-600">{{ __('Email') }}</span>
                                 </button>
                             </div>
                         </div>
                     </div>
                 </div>

                 <!-- Lightbox Content -->
                 <div class="flex-1 flex items-center justify-center p-2 relative h-full">
                     <!-- Nav Buttons -->
                     <button @click="lightboxIndex = (lightboxIndex > 0) ? lightboxIndex - 1 : lightboxSources.length - 1" class="absolute left-4 p-4 hover:bg-white/10 rounded-full transition-colors z-20 hidden md:block">
                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="15 18 9 12 15 6"/></svg>
                     </button>

                     <div class="relative w-full h-full flex items-center justify-center"
                          x-swipe:left="lightboxIndex = (lightboxIndex < lightboxSources.length - 1) ? lightboxIndex + 1 : 0"
                          x-swipe:right="lightboxIndex = (lightboxIndex > 0) ? lightboxIndex - 1 : lightboxSources.length - 1">

                         <template x-for="(item, index) in lightboxSources" :key="index">
                             <div x-show="lightboxIndex === index" class="w-full h-full flex items-center justify-center" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 scale-95" x-transition:enter-end="opacity-100 scale-100">
                                 <template x-if="item.type === 'video'">
                                     <video :src="item.src" controls autoplay class="max-w-full max-h-full object-contain shadow-2xl"></video>
                                 </template>
                                 <template x-if="item.type !== 'video'">
                                     <img :src="item.src" class="max-w-full max-h-full object-contain shadow-2xl">
                                 </template>
                             </div>
                         </template>
                     </div>

                     <button @click="lightboxIndex = (lightboxIndex < lightboxSources.length - 1) ? lightboxIndex + 1 : 0" class="absolute right-4 p-4 hover:bg-white/10 rounded-full transition-colors z-20 hidden md:block">
                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="9 18 15 12 9 6"/></svg>
                     </button>
                 </div>
            </div>
        @endif

        <!-- Global Camera Modal -->
        <div x-show="cameraOpen" x-cloak class="fixed inset-0 z-[60] flex items-center justify-center bg-black/90 backdrop-blur-sm" x-transition>
            <!-- Close Button -->
            <button @click="stopCamera()" class="absolute top-6 right-6 text-white hover:text-red-500 transition-colors z-[70]">
                <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
            </button>

            <div class="relative w-full max-w-md p-6 text-center z-[65]">
                <template x-if="!captured">
                    <div>
                        <h3 class="text-2xl font-bold text-white mb-2">{{ __('Capture Selfie') }}</h3>
                        <p class="text-cyan-400 text-sm font-medium mb-6">{{ __('Keep Face in the Center and Press Capture') }}</p>

                        <div class="relative w-64 h-64 mx-auto mb-8 rounded-full overflow-hidden border-4 border-dashed border-yellow-400">
                            <video x-ref="video" autoplay playsinline muted class="w-full h-full object-cover transform scale-x-[-1]"></video>

                            <!-- Face Mesh Overlay -->
                            <div class="absolute inset-0 flex items-center justify-center md:opacity-50 opacity-30 pointer-events-none">
                                    <svg viewBox="0 0 200 250" class="w-full h-full text-yellow-400" fill="none" stroke="currentColor" stroke-width="1">
                                    <path d="M100 20 C60 20 30 60 30 110 C30 170 60 220 100 230 C140 220 170 170 170 110 C170 60 140 20 100 20 Z" stroke-dasharray="4 4"/>
                                    <path d="M70 100 Q100 100 130 100" stroke-opacity="0.5"/>
                                    <path d="M90 140 L100 150 L110 140" stroke-opacity="0.5"/>
                                    <path d="M80 180 Q100 190 120 180" stroke-opacity="0.5"/>
                                    <line x1="100" y1="20" x2="100" y2="230" stroke-opacity="0.3"/>
                                    <line x1="30" y1="110" x2="170" y2="110" stroke-opacity="0.3"/>
                                    </svg>
                            </div>
                        </div>

                        <button @click="capturePhoto()" class="bg-red-600 hover:bg-red-700 text-white font-bold py-3 px-8 rounded flex items-center gap-2 mx-auto transition-transform active:scale-95">
                            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"></path><circle cx="12" cy="13" r="4"></circle></svg>
                            {{ __('Capture') }}
                        </button>
                    </div>
                </template>

                <template x-if="captured">
                    <div class="bg-gray-900 p-8 rounded-2xl border border-gray-700">
                        <h3 class="text-2xl font-bold text-green-500 mb-2 flex items-center justify-center gap-2">
                            <svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>
                            {{ __('Selfie Verified') }}
                        </h3>
                        <p class="text-white mb-6">{{ __('Click Done to confirm or Retake to reject') }}</p>

                        <div class="relative w-64 h-64 mx-auto mb-8 rounded-full overflow-hidden border-4 border-green-500">
                            <img :src="imageSrc" class="w-full h-full object-cover">
                        </div>

                        <div class="flex items-center justify-center gap-4">
                            <button @click="retakePhoto()" class="px-6 py-2 bg-gray-700 hover:bg-gray-600 text-white rounded font-medium transition-colors">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="inline mr-1"><path d="M23 4v6h-6"/><path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10"/></svg>
                                {{ __('Retake') }}
                            </button>
                            <button @click="confirmPhoto()" class="px-6 py-2 bg-green-500 hover:bg-green-600 text-white rounded font-medium transition-colors" :class="{'opacity-50 cursor-wait': uploading}" :disabled="uploading">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="inline mr-1"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
                                {{ __('Done') }}
                            </button>
                        </div>
                    </div>
                </template>
            </div>
        </div>

    </div>
</x-gallery-layout>
