@extends('install.layout')
@section('title', __('Installing'))
@section('step-name', __('Install'))

@section('content')
<div class="space-y-5">
    <div class="space-y-1">
        <h1 class="text-2xl font-bold tracking-tight">
            {{ __('Installing') }} <span class="text-primary italic">SnapNest.</span>
        </h1>
        <p class="text-muted-foreground text-xs font-medium">{{ __('Please wait while we set everything up...') }}</p>
    </div>
</div>

<style>
    @keyframes installer-spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
    .installer-spinner { animation: installer-spin 1s linear infinite; }

    @keyframes pulse-glow {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.5; }
    }
    .step-pulse { animation: pulse-glow 1.5s ease-in-out infinite; }

    @keyframes progress-fill {
        from { width: var(--from); }
        to { width: var(--to); }
    }

    @keyframes step-enter {
        from { opacity: 0; transform: translateY(4px); }
        to { opacity: 1; transform: translateY(0); }
    }
    .step-enter { animation: step-enter 0.3s ease-out forwards; }

    @keyframes check-draw {
        from { stroke-dashoffset: 20; }
        to { stroke-dashoffset: 0; }
    }
    .check-animate path {
        stroke-dasharray: 20;
        stroke-dashoffset: 20;
        animation: check-draw 0.4s ease-out 0.1s forwards;
    }
</style>

<div x-data="installer()" x-init="start()" class="mt-5">
    {{-- Overall progress bar --}}
    <div class="mb-5">
        <div class="flex items-center justify-between mb-1.5">
            <span class="text-[10px] font-bold uppercase tracking-widest text-muted-foreground" x-text="progressLabel"></span>
            <span class="text-[10px] font-bold uppercase tracking-widest text-primary tabular-nums" x-text="Math.round(progress) + '%'"></span>
        </div>
        <div class="h-1.5 w-full rounded-full bg-muted/50 overflow-hidden">
            <div class="h-full rounded-full bg-gradient-to-r from-primary to-primary/80 transition-all duration-700 ease-out"
                 :style="'width: ' + progress + '%'"></div>
        </div>
    </div>

    {{-- Steps --}}
    <div class="space-y-1">
        <template x-for="(step, index) in steps" :key="index">
            <div class="flex items-center gap-3 py-2.5 px-3 rounded-xl transition-all duration-300"
                 :class="{
                     'bg-primary/5 step-enter': step.status === 'running',
                     'bg-emerald-500/5': step.status === 'done',
                     'bg-red-500/5': step.status === 'failed',
                 }">
                {{-- Icon --}}
                <div class="shrink-0 w-5 h-5 flex items-center justify-center">
                    <template x-if="step.status === 'pending'">
                        <svg class="w-4 h-4 text-muted-foreground/30" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10" /></svg>
                    </template>
                    <template x-if="step.status === 'running'">
                        <svg class="w-4 h-4 text-primary installer-spinner" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><path d="M12 2a10 10 0 0 1 10 10" stroke-linecap="round" /></svg>
                    </template>
                    <template x-if="step.status === 'done'">
                        <svg class="w-4 h-4 text-emerald-500 check-animate" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2.5"><path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" /></svg>
                    </template>
                    <template x-if="step.status === 'failed'">
                        <svg class="w-4 h-4 text-red-500" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2.5"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" /></svg>
                    </template>
                </div>

                {{-- Label --}}
                <div class="flex-1 min-w-0">
                    <span class="text-xs font-medium transition-colors"
                          :class="{
                              'text-muted-foreground/50': step.status === 'pending',
                              'text-primary font-semibold': step.status === 'running',
                              'text-emerald-600 dark:text-emerald-400': step.status === 'done',
                              'text-red-500': step.status === 'failed',
                          }"
                          x-text="step.status === 'running' ? step.activeLabel : step.label"></span>
                </div>

                {{-- Duration --}}
                <div class="shrink-0">
                    <span x-show="step.status === 'done' && step.duration"
                          class="text-[10px] text-muted-foreground tabular-nums"
                          x-text="step.duration"></span>
                    <span x-show="step.status === 'running'"
                          class="text-[10px] text-primary/60 tabular-nums step-pulse"
                          x-text="elapsed"></span>
                </div>
            </div>
        </template>
    </div>

    {{-- Error --}}
    <div x-show="error" x-cloak class="mt-5 rounded-xl border border-red-500/30 bg-red-500/10 px-4 py-3 text-sm font-medium text-red-400" x-text="error"></div>

    {{-- Retry --}}
    <div x-show="error" x-cloak class="mt-3">
        <button @click="retry()"
                class="w-full h-11 rounded-xl bg-primary hover:bg-primary/90 text-sm font-bold shadow-lg shadow-primary/20 transition-all text-primary-foreground">
            {{ __('Retry Installation') }}
        </button>
    </div>

    {{-- Success --}}
    <div x-show="allDone" x-cloak
         class="mt-5 flex items-center gap-3 rounded-xl border border-emerald-500/30 bg-emerald-500/10 px-4 py-3">
        <svg class="w-5 h-5 text-emerald-500 shrink-0" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
        <span class="text-sm font-semibold text-emerald-600 dark:text-emerald-400">{{ __('Installation complete! Redirecting...') }}</span>
    </div>
</div>

<script>
    function installer() {
        return {
            steps: [
                { label: '{{ __("Environment configured") }}', activeLabel: '{{ __("Configuring environment...") }}', status: 'pending', duration: null, startedAt: null },
                { label: '{{ __("Database tables created") }}', activeLabel: '{{ __("Running database migrations...") }}', status: 'pending', duration: null, startedAt: null },
                { label: '{{ __("Essential data seeded") }}', activeLabel: '{{ __("Seeding essential data...") }}', status: 'pending', duration: null, startedAt: null },
                { label: '{{ __("Admin account created") }}', activeLabel: '{{ __("Creating admin account...") }}', status: 'pending', duration: null, startedAt: null },
                { label: '{{ __("File permissions set") }}', activeLabel: '{{ __("Setting file permissions...") }}', status: 'pending', duration: null, startedAt: null },
                { label: '{{ __("Installation finalized") }}', activeLabel: '{{ __("Finalizing installation...") }}', status: 'pending', duration: null, startedAt: null },
            ],
            error: null,
            allDone: false,
            elapsed: '0.0s',
            _timer: null,

            get progress() {
                const done = this.steps.filter(s => s.status === 'done').length;
                const running = this.steps.filter(s => s.status === 'running').length;
                return ((done + running * 0.5) / this.steps.length) * 100;
            },

            get progressLabel() {
                const current = this.steps.find(s => s.status === 'running');
                if (this.allDone) return '{{ __("Complete") }}';
                if (this.error) return '{{ __("Failed") }}';
                if (current) return current.activeLabel.replace('...', '');
                return '{{ __("Preparing") }}';
            },

            formatDuration(ms) {
                if (ms < 1000) return ms + 'ms';
                return (ms / 1000).toFixed(1) + 's';
            },

            startTimer(step) {
                step.startedAt = Date.now();
                this.elapsed = '0.0s';
                this._timer = setInterval(() => {
                    const diff = Date.now() - step.startedAt;
                    this.elapsed = (diff / 1000).toFixed(1) + 's';
                }, 100);
            },

            stopTimer(step) {
                clearInterval(this._timer);
                if (step.startedAt) {
                    step.duration = this.formatDuration(Date.now() - step.startedAt);
                }
            },

            async runStep(index) {
                const step = this.steps[index];
                step.status = 'running';
                this.startTimer(step);

                try {
                    const res = await fetch('{{ route("install.execute") }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name=csrf-token]').content,
                            'Accept': 'application/json',
                        },
                        body: JSON.stringify({ step: index + 1 }),
                    });

                    const data = await res.json();

                    this.stopTimer(step);

                    if (res.ok && data.success) {
                        step.status = 'done';
                        return true;
                    } else {
                        step.status = 'failed';
                        this.error = data.message || '{{ __("Step failed. Please try again.") }}';
                        return false;
                    }
                } catch (e) {
                    this.stopTimer(step);
                    step.status = 'failed';
                    this.error = '{{ __("Network error. Please check your connection and try again.") }}';
                    return false;
                }
            },

            async start() {
                this.error = null;

                for (let i = 0; i < this.steps.length; i++) {
                    if (this.steps[i].status === 'done') continue;

                    const success = await this.runStep(i);
                    if (!success) return;
                }

                this.allDone = true;
                setTimeout(() => {
                    window.location.href = '{{ route("install.complete") }}';
                }, 2000);
            },

            retry() {
                this.error = null;
                // Reset failed step and all after it to pending
                let failedIndex = this.steps.findIndex(s => s.status === 'failed');
                if (failedIndex === -1) failedIndex = 0;
                for (let i = failedIndex; i < this.steps.length; i++) {
                    this.steps[i].status = 'pending';
                    this.steps[i].duration = null;
                }
                this.start();
            }
        }
    }
</script>
@endsection
