<?php

use App\Models\Event;
use App\Services\EventMediaService;
use App\Support\EventMediaSettings;
use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schedule;

// Queue worker — processes jobs automatically
// No manual queue:work needed; on hosting just add cron: * * * * * php artisan schedule:run
Schedule::command('queue:work --queue=default --stop-when-empty --tries=3 --timeout=600 --memory=512 --sleep=3')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

// GDPR & Maintenance scheduled tasks
Schedule::command('media:cleanup-deleted --force')->daily()->at('02:00');
Schedule::command('gdpr:cleanup-face-data')->daily()->at('03:00');

// Auto-ingest FTP uploads every 2 minutes for all active events
Schedule::command('events:ingest-ftp-all')->everyTwoMinutes()->withoutOverlapping();

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote');

Artisan::command('events:ingest-ftp {event}', function (EventMediaService $service) {
    $eventId = (int) $this->argument('event');
    $event = Event::find($eventId);

    if (! $event) {
        $this->error('Event not found.');
        return;
    }

    $basePath = (string) EventMediaSettings::getValue('ftp_path', config('events.media.ftp_path'));
    $basePath = rtrim($basePath, DIRECTORY_SEPARATOR);
    $path = $basePath . DIRECTORY_SEPARATOR . $eventId;

    if (! File::isDirectory($path)) {
        $this->warn('FTP directory not found: ' . $path);
        return;
    }

    $processedDir = $path . DIRECTORY_SEPARATOR . 'processed';
    File::ensureDirectoryExists($processedDir);

    $imported = 0;
    $skipped = 0;
    $failed = 0;
    $allowed = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];

    foreach (File::files($path) as $file) {
        if (! $file->isFile()) {
            continue;
        }

        $extension = strtolower($file->getExtension());
        if (! in_array($extension, $allowed, true)) {
            $skipped++;
            continue;
        }

        try {
            $media = $service->storeLocalFile($event, $file->getPathname(), $file->getFilename());
            if ($media) {
                $imported++;
                File::move($file->getPathname(), $processedDir . DIRECTORY_SEPARATOR . $file->getFilename());
            }
        } catch (\Throwable $e) {
            $failed++;
            $this->error("Failed: {$file->getFilename()} - {$e->getMessage()}");
        }
    }

    $this->info("Imported: {$imported}, Skipped: {$skipped}, Failed: {$failed}");
})->purpose('Ingest FTP camera uploads into event media');

Artisan::command('events:ingest-ftp-all', function (EventMediaService $service) {
    $basePath = (string) EventMediaSettings::getValue('ftp_path', config('events.media.ftp_path'));
    $basePath = rtrim($basePath, DIRECTORY_SEPARATOR);

    if (! File::isDirectory($basePath)) {
        return;
    }

    $allowed = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];
    $totalImported = 0;

    foreach (File::directories($basePath) as $eventDir) {
        $eventId = (int) basename($eventDir);
        if ($eventId <= 0) {
            continue;
        }

        $event = Event::where('id', $eventId)->where('is_active', true)->first();
        if (! $event) {
            continue;
        }

        $files = File::files($eventDir);
        if (empty($files)) {
            continue;
        }

        $processedDir = $eventDir . DIRECTORY_SEPARATOR . 'processed';
        File::ensureDirectoryExists($processedDir);

        foreach ($files as $file) {
            if (! $file->isFile()) {
                continue;
            }

            $extension = strtolower($file->getExtension());
            if (! in_array($extension, $allowed, true)) {
                continue;
            }

            try {
                $media = $service->storeLocalFile($event, $file->getPathname(), $file->getFilename());
                if ($media) {
                    $totalImported++;
                    File::move($file->getPathname(), $processedDir . DIRECTORY_SEPARATOR . $file->getFilename());
                }
            } catch (\Throwable $e) {
                Log::warning("FTP auto-ingest failed for event {$eventId}: {$file->getFilename()} - {$e->getMessage()}");
            }
        }
    }

    if ($totalImported > 0) {
        $this->info("Auto-ingested {$totalImported} file(s) across events.");
    }
})->purpose('Auto-ingest FTP uploads for all active events');

